<?php
// Activar errores para depuración (comentar en producción)
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once 'config.php';

// Verificar que sea operador
if (!isset($_SESSION['operator_id'])) {
    header('Location: index.php');
    exit();
}

// Obtener información del operador
try {
    $stmt = $conn->prepare("SELECT * FROM operators WHERE id = ?");
    $stmt->execute([$_SESSION['operator_id']]);
    $operator = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$operator) {
        session_destroy();
        header('Location: index.php');
        exit();
    }
} catch (PDOException $e) {
    die("Error al obtener información del operador: " . $e->getMessage());
}

// Configuración de idioma - Persistente en BD
$lang = $operator['language'] ?? 'es';

if (isset($_GET['lang']) && in_array($_GET['lang'], ['es', 'en'])) {
    $lang = $_GET['lang'];
    $_SESSION['lang'] = $lang;
    
    // Guardar en la base de datos para persistencia
    try {
        $stmt = $conn->prepare("UPDATE operators SET language = ? WHERE id = ?");
        $stmt->execute([$lang, $_SESSION['operator_id']]);
    } catch(PDOException $e) {
        // Si falla, continuamos con el idioma en sesión
    }
}

$translations = [
    'es' => [
        'admin_panel' => 'Panel de Administración',
        'dashboard' => 'Dashboard',
        'users' => 'Usuarios',
        'operators' => 'Operadores',
        'statistics' => 'Estadísticas',
        'settings' => 'Configuración',
        'logout' => 'Cerrar Sesión',
        'welcome' => 'Bienvenido',
        'search' => 'Buscar...',
        'total_users' => 'Total Usuarios',
        'total_operators' => 'Total Operadores',
        'total_alerts' => 'Total Alertas',
        'active_alerts' => 'Alertas Activas',
        'resolved_today' => 'Resueltas Hoy',
        'most_active_users' => 'Usuarios Más Activos',
        'alert_distribution' => 'Distribución de Alertas',
        'recent_activity' => 'Actividad Reciente',
        'user_management' => 'Gestión de Usuarios',
        'operator_management' => 'Gestión de Operadores',
        'view_all' => 'Ver Todos',
        'medical' => 'Médica',
        'fire' => 'Incendio',
        'security' => 'Seguridad',
        'alerts_this_month' => 'Alertas este Mes',
        'theme' => 'Tema',
        'light' => 'Claro',
        'dark' => 'Oscuro',
        'blue_dark' => 'Azul Oscuro',
    ],
    'en' => [
        'admin_panel' => 'Administration Panel',
        'dashboard' => 'Dashboard',
        'users' => 'Users',
        'operators' => 'Operators',
        'statistics' => 'Statistics',
        'settings' => 'Settings',
        'logout' => 'Logout',
        'welcome' => 'Welcome',
        'search' => 'Search...',
        'total_users' => 'Total Users',
        'total_operators' => 'Total Operators',
        'total_alerts' => 'Total Alerts',
        'active_alerts' => 'Active Alerts',
        'resolved_today' => 'Resolved Today',
        'most_active_users' => 'Most Active Users',
        'alert_distribution' => 'Alert Distribution',
        'recent_activity' => 'Recent Activity',
        'user_management' => 'User Management',
        'operator_management' => 'Operator Management',
        'view_all' => 'View All',
        'medical' => 'Medical',
        'fire' => 'Fire',
        'security' => 'Security',
        'alerts_this_month' => 'Alerts this Month',
        'theme' => 'Theme',
        'light' => 'Light',
        'dark' => 'Dark',
        'blue_dark' => 'Blue Dark',
    ]
];

$t = $translations[$lang];

// Obtener estadísticas (con manejo de errores)
$stats = [];

try {
    // Total de usuarios
    $stmt = $conn->query("SELECT COUNT(*) as total FROM users");
    $stats['total_users'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
} catch (PDOException $e) {
    $stats['total_users'] = 0;
}

try {
    // Total de operadores
    $stmt = $conn->query("SELECT COUNT(*) as total FROM operators");
    $stats['total_operators'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
} catch (PDOException $e) {
    $stats['total_operators'] = 0;
}

try {
    // Total de alertas
    $stmt = $conn->query("SELECT COUNT(*) as total FROM alerts");
    $stats['total_alerts'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
} catch (PDOException $e) {
    $stats['total_alerts'] = 0;
}

try {
    // Alertas activas
    $stmt = $conn->query("SELECT COUNT(*) as total FROM alerts WHERE status IN ('pending', 'in_progress')");
    $stats['active_alerts'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
} catch (PDOException $e) {
    $stats['active_alerts'] = 0;
}

try {
    // Resueltas hoy
    $stmt = $conn->query("SELECT COUNT(*) as total FROM alerts WHERE status = 'resolved' AND DATE(updated_at) = CURDATE()");
    $stats['resolved_today'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
} catch (PDOException $e) {
    $stats['resolved_today'] = 0;
}

try {
    // Alertas este mes
    $stmt = $conn->query("SELECT COUNT(*) as total FROM alerts WHERE MONTH(created_at) = MONTH(CURRENT_DATE()) AND YEAR(created_at) = YEAR(CURRENT_DATE())");
    $stats['alerts_month'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
} catch (PDOException $e) {
    $stats['alerts_month'] = 0;
}

try {
    // Distribución de alertas por tipo
    $stmt = $conn->query("SELECT alert_type, COUNT(*) as count FROM alerts GROUP BY alert_type");
    $alert_distribution = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $alert_distribution = [];
}

try {
    // Usuarios más activos
    $stmt = $conn->query("
        SELECT u.id as user_id, u.full_name as name, u.email, COUNT(a.id) as alert_count
        FROM users u
        LEFT JOIN alerts a ON u.id = a.user_id
        GROUP BY u.id
        ORDER BY alert_count DESC
        LIMIT 5
    ");
    $most_active_users = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $most_active_users = [];
}

try {
    // Actividad reciente
    $stmt = $conn->query("
        SELECT a.*, u.full_name as user_name, u.email as user_email
        FROM alerts a
        LEFT JOIN users u ON a.user_id = u.id
        ORDER BY a.created_at DESC
        LIMIT 10
    ");
    $recent_alerts = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $recent_alerts = [];
}
?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $t['admin_panel']; ?> - Azuero Safety Services S.A</title>
    
    <!-- PWA Meta Tags -->
    <meta name="theme-color" content="#667eea">
    <meta name="mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="black-translucent">
    <meta name="apple-mobile-web-app-title" content="AzueroSafe Admin">
    <meta name="description" content="Panel de Administración - Azuero Safety Services S.A">
    
    <!-- PWA Manifest -->
    <link rel="manifest" href="manifest.json">
    
    <!-- Icons -->
    <link rel="icon" type="image/png" sizes="192x192" href="icons/icon-192x192.png">
    <link rel="apple-touch-icon" sizes="192x192" href="icons/icon-192x192.png">
    
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary: #667eea;
            --primary-dark: #5568d3;
            --secondary: #764ba2;
            --success: #10b981;
            --warning: #f59e0b;
            --danger: #ef4444;
            --info: #3b82f6;
            --bg-primary: #ffffff;
            --bg-secondary: #f8fafc;
            --bg-card: #ffffff;
            --text-primary: #1e293b;
            --text-secondary: #64748b;
            --border-color: #e2e8f0;
            --shadow-sm: 0 1px 3px rgba(0,0,0,0.1);
            --shadow-md: 0 4px 6px rgba(0,0,0,0.1);
            --shadow-lg: 0 10px 25px rgba(0,0,0,0.15);
            --sidebar-width: 280px;
        }

        [data-theme="dark"] {
            --primary: #818cf8;
            --primary-dark: #6366f1;
            --bg-primary: #0f172a;
            --bg-secondary: #1e293b;
            --bg-card: #1e293b;
            --text-primary: #f1f5f9;
            --text-secondary: #94a3b8;
            --border-color: #334155;
            --shadow-sm: 0 1px 3px rgba(0,0,0,0.3);
            --shadow-md: 0 4px 6px rgba(0,0,0,0.3);
            --shadow-lg: 0 10px 25px rgba(0,0,0,0.5);
        }

        [data-theme="blue-dark"] {
            --primary: #60a5fa;
            --primary-dark: #3b82f6;
            --secondary: #06b6d4;
            --bg-primary: #0c1e35;
            --bg-secondary: #1a2942;
            --bg-card: #1a2942;
            --text-primary: #e0f2fe;
            --text-secondary: #7dd3fc;
            --border-color: #1e3a5f;
            --shadow-sm: 0 1px 3px rgba(14,165,233,0.2);
            --shadow-md: 0 4px 6px rgba(14,165,233,0.2);
            --shadow-lg: 0 10px 25px rgba(14,165,233,0.3);
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: var(--bg-primary);
            color: var(--text-primary);
            transition: all 0.3s ease;
        }

        /* Sidebar */
        .sidebar {
            position: fixed;
            left: 0;
            top: 0;
            bottom: 0;
            width: var(--sidebar-width);
            background: var(--bg-card);
            border-right: 1px solid var(--border-color);
            box-shadow: var(--shadow-md);
            z-index: 1000;
            transition: all 0.3s ease;
            overflow-y: auto;
        }

        .sidebar-header {
            padding: 30px 20px;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            text-align: center;
        }

        .sidebar-header h2 {
            font-size: 24px;
            margin-bottom: 5px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }

        .sidebar-header p {
            font-size: 14px;
            opacity: 0.9;
        }

        .sidebar-menu {
            padding: 20px 0;
        }

        .menu-item {
            display: flex;
            align-items: center;
            padding: 15px 25px;
            color: var(--text-primary);
            text-decoration: none;
            transition: all 0.3s ease;
            cursor: pointer;
            border-left: 3px solid transparent;
        }

        .menu-item:hover, .menu-item.active {
            background: var(--bg-secondary);
            border-left-color: var(--primary);
            color: var(--primary);
        }

        .menu-item i {
            width: 24px;
            font-size: 20px;
            margin-right: 15px;
        }

        .menu-section {
            padding: 10px 25px;
            font-size: 12px;
            font-weight: 600;
            color: var(--text-secondary);
            text-transform: uppercase;
            letter-spacing: 1px;
            margin-top: 20px;
        }

        /* Main Content */
        .main-content {
            margin-left: var(--sidebar-width);
            min-height: 100vh;
            transition: margin-left 0.3s ease;
        }

        /* Header */
        .header {
            background: var(--bg-card);
            border-bottom: 1px solid var(--border-color);
            padding: 20px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: var(--shadow-sm);
            position: sticky;
            top: 0;
            z-index: 999;
        }

        .header-left h1 {
            font-size: 28px;
            color: var(--text-primary);
            margin-bottom: 5px;
        }

        .breadcrumb {
            color: var(--text-secondary);
            font-size: 14px;
        }

        .header-right {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .search-box {
            position: relative;
        }

        .search-box input {
            padding: 10px 40px 10px 15px;
            border: 1px solid var(--border-color);
            border-radius: 25px;
            background: var(--bg-secondary);
            color: var(--text-primary);
            width: 300px;
            transition: all 0.3s ease;
        }

        .search-box input:focus {
            outline: none;
            border-color: var(--primary);
            width: 350px;
        }

        .search-box i {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--text-secondary);
        }

        .notification-bell {
            position: relative;
            cursor: pointer;
            font-size: 24px;
            color: var(--text-primary);
            transition: color 0.3s ease;
        }

        .notification-bell:hover {
            color: var(--primary);
        }

        .notification-badge {
            position: absolute;
            top: -8px;
            right: -8px;
            background: var(--danger);
            color: white;
            border-radius: 50%;
            width: 20px;
            height: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 11px;
            font-weight: bold;
            animation: pulse 2s infinite;
        }

        @keyframes pulse {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.1); }
        }

        .theme-switcher {
            display: flex;
            gap: 10px;
            background: var(--bg-secondary);
            padding: 5px;
            border-radius: 25px;
        }

        .theme-btn {
            padding: 8px 15px;
            border: none;
            background: transparent;
            color: var(--text-secondary);
            cursor: pointer;
            border-radius: 20px;
            font-size: 13px;
            transition: all 0.3s ease;
        }

        .theme-btn.active {
            background: var(--primary);
            color: white;
        }

        .user-profile {
            display: flex;
            align-items: center;
            gap: 12px;
            cursor: pointer;
            padding: 8px 15px;
            border-radius: 25px;
            transition: all 0.3s ease;
        }

        .user-profile:hover {
            background: var(--bg-secondary);
        }

        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
            font-size: 16px;
        }

        .user-info h4 {
            font-size: 14px;
            margin-bottom: 2px;
        }

        .user-info p {
            font-size: 12px;
            color: var(--text-secondary);
        }

        /* Content Container */
        .content {
            padding: 30px;
        }

        /* Stats Cards */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 15px;
            padding: 25px;
            box-shadow: var(--shadow-sm);
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, var(--primary), var(--secondary));
        }

        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: var(--shadow-lg);
        }

        .stat-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 15px;
        }

        .stat-icon {
            width: 50px;
            height: 50px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: white;
        }

        .stat-icon.blue { background: linear-gradient(135deg, #667eea, #764ba2); }
        .stat-icon.green { background: linear-gradient(135deg, #10b981, #059669); }
        .stat-icon.orange { background: linear-gradient(135deg, #f59e0b, #d97706); }
        .stat-icon.red { background: linear-gradient(135deg, #ef4444, #dc2626); }
        .stat-icon.purple { background: linear-gradient(135deg, #8b5cf6, #7c3aed); }
        .stat-icon.cyan { background: linear-gradient(135deg, #06b6d4, #0891b2); }

        .stat-value {
            font-size: 32px;
            font-weight: bold;
            color: var(--text-primary);
            margin-bottom: 5px;
        }

        .stat-label {
            font-size: 14px;
            color: var(--text-secondary);
        }

        .stat-trend {
            display: inline-flex;
            align-items: center;
            gap: 5px;
            font-size: 12px;
            padding: 4px 8px;
            border-radius: 12px;
            margin-top: 10px;
        }

        .stat-trend.up {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
        }

        .stat-trend.down {
            background: rgba(239, 68, 68, 0.1);
            color: var(--danger);
        }

        /* Charts Section */
        .charts-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .chart-card {
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 15px;
            padding: 25px;
            box-shadow: var(--shadow-sm);
        }

        .chart-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        .chart-header h3 {
            font-size: 18px;
            color: var(--text-primary);
        }

        .chart-container {
            position: relative;
            height: 300px;
        }

        /* Tables */
        .table-card {
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 15px;
            padding: 25px;
            box-shadow: var(--shadow-sm);
            margin-bottom: 30px;
        }

        .table-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        .table-header h3 {
            font-size: 18px;
            color: var(--text-primary);
        }

        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 500;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
            box-shadow: var(--shadow-md);
        }

        .table-wrapper {
            overflow-x: auto;
        }

        table {
            width: 100%;
            border-collapse: collapse;
        }

        thead {
            background: var(--bg-secondary);
        }

        th, td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid var(--border-color);
        }

        th {
            font-weight: 600;
            color: var(--text-secondary);
            font-size: 13px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        td {
            color: var(--text-primary);
        }

        tbody tr {
            transition: background 0.3s ease;
        }

        tbody tr:hover {
            background: var(--bg-secondary);
        }

        .badge {
            display: inline-block;
            padding: 5px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 500;
        }

        .badge-success { background: rgba(16, 185, 129, 0.1); color: var(--success); }
        .badge-warning { background: rgba(245, 158, 11, 0.1); color: var(--warning); }
        .badge-danger { background: rgba(239, 68, 68, 0.1); color: var(--danger); }
        .badge-info { background: rgba(59, 130, 246, 0.1); color: var(--info); }

        /* Notification Toast */
        .toast-container {
            position: fixed;
            top: 80px;
            right: 20px;
            z-index: 9999;
            max-width: 400px;
        }

        .toast {
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 10px;
            box-shadow: var(--shadow-lg);
            display: flex;
            align-items: flex-start;
            gap: 15px;
            animation: slideIn 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        @keyframes slideIn {
            from {
                transform: translateX(400px);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }

        .toast::before {
            content: '';
            position: absolute;
            left: 0;
            top: 0;
            bottom: 0;
            width: 4px;
        }

        .toast.emergency::before { background: var(--danger); }
        .toast.warning::before { background: var(--warning); }
        .toast.info::before { background: var(--info); }
        .toast.success::before { background: var(--success); }

        .toast-icon {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 20px;
            flex-shrink: 0;
        }

        .toast.emergency .toast-icon {
            background: rgba(239, 68, 68, 0.1);
            color: var(--danger);
        }

        .toast-content h4 {
            font-size: 14px;
            margin-bottom: 5px;
            color: var(--text-primary);
        }

        .toast-content p {
            font-size: 13px;
            color: var(--text-secondary);
            line-height: 1.5;
        }

        .toast-time {
            font-size: 11px;
            color: var(--text-secondary);
            margin-top: 5px;
        }

        .toast-close {
            position: absolute;
            top: 10px;
            right: 10px;
            background: none;
            border: none;
            color: var(--text-secondary);
            cursor: pointer;
            font-size: 18px;
            padding: 5px;
            line-height: 1;
        }

        /* Responsive */
        @media (max-width: 1024px) {
            .sidebar {
                transform: translateX(-100%);
            }

            .sidebar.active {
                transform: translateX(0);
            }

            .main-content {
                margin-left: 0;
            }

            .charts-grid {
                grid-template-columns: 1fr;
            }
        }

        @media (max-width: 768px) {
            .header {
                flex-direction: column;
                gap: 15px;
            }

            .search-box input {
                width: 100%;
            }

            .stats-grid {
                grid-template-columns: 1fr;
            }

            .header-right {
                flex-wrap: wrap;
                width: 100%;
            }

            .theme-switcher {
                width: 100%;
                justify-content: space-between;
            }
        }

        /* Section Container */
        .section {
            display: none;
        }

        .section.active {
            display: block;
        }
    </style>
</head>
<body data-theme="light">
    <script>
        (function(){
            function changeTheme(theme) {
                document.body.setAttribute('data-theme', theme);
                try {
                    var btns = document.querySelectorAll('.theme-btn');
                    for (var i = 0; i < btns.length; i++) {
                        btns[i].classList.remove('active');
                        if (btns[i].getAttribute('data-theme') === theme) {
                            btns[i].classList.add('active');
                        }
                    }
                } catch (e) { /* no-op */ }
                try { localStorage.setItem('theme', theme); } catch (e) { /* no-op */ }
            }
            window.changeTheme = changeTheme;

            window.showSection = function(sectionName, element) {
                try {
                    var sections = document.querySelectorAll('.section');
                    for (var i = 0; i < sections.length; i++) {
                        sections[i].classList.remove('active');
                    }
                    var target = document.getElementById(sectionName + '-section');
                    if (target) { target.classList.add('active'); }

                    var items = document.querySelectorAll('.menu-item');
                    for (var j = 0; j < items.length; j++) { items[j].classList.remove('active'); }
                    if (element) {
                        var mi = element.closest ? element.closest('.menu-item') : element;
                        if (mi) { mi.classList.add('active'); }
                    }

                    if (sectionName === 'users' && typeof loadUsers === 'function') { loadUsers(); }
                    else if (sectionName === 'operators' && typeof loadOperators === 'function') { loadOperators(); }
                } catch (e) { /* no-op */ }
            };

            try {
                var saved = (typeof localStorage !== 'undefined' && localStorage.getItem('theme')) || 'light';
                changeTheme(saved);
            } catch (e) {
                changeTheme('light');
            }

            // Fallback data loaders (avoid template literals to prevent parse stops)
            if (!window.loadUsers) {
                window.loadUsers = function() {
                    try {
                        var tbody = document.getElementById('usersTableBody');
                        if (tbody) tbody.innerHTML = '<tr><td colspan="8" style="text-align:center;padding:20px;"><i class="fas fa-spinner fa-spin"></i> Cargando...</td></tr>';
                        
                        fetch('manage-users.php?action=list')
                            .then(function(r){ 
                                console.log('Users API status:', r.status);
                                return r.json(); 
                            })
                            .then(function(data){
                                console.log('Users API response:', data);
                                if (!data || !data.success) {
                                    if (tbody) tbody.innerHTML = '<tr><td colspan="8" style="text-align:center;padding:20px;color:red;">Error: ' + (data.error || 'No se pudieron cargar usuarios') + '</td></tr>';
                                    return;
                                }
                                var users = data.users || [];
                                if (!tbody) return;
                                if (users.length === 0) {
                                    tbody.innerHTML = '<tr><td colspan="8" style="text-align:center;padding:40px;">Sin usuarios registrados</td></tr>';
                                } else {
                                    var rows = '';
                                    for (var i=0;i<users.length;i++) {
                                        var u = users[i];
                                        var initial = (u.name && u.name.charAt(0).toUpperCase()) || 'U';
                                        var name = u.name || 'Sin nombre';
                                        var email = u.email || '-';
                                        var phone = u.phone || '-';
                                        var alertCount = (typeof u.alert_count !== 'undefined') ? u.alert_count : 0;
                                        var status = u.status === 'active' ? 'Activo' : 'Inactivo';
                                        var statusColor = u.status === 'active' ? 'success' : 'danger';
                                        var created = u.created_at ? new Date(u.created_at).toLocaleDateString() : '-';
                                        rows += '<tr>' +
                                            '<td>#' + u.user_id + '</td>' +
                                            '<td><div style="display:flex;align-items:center;gap:10px;"><div class="user-avatar" style="width:32px;height:32px;font-size:12px;background:linear-gradient(135deg,#667eea,#764ba2);color:white;display:flex;align-items:center;justify-content:center;border-radius:50%;">' + initial + '</div><div><div style="font-weight:500;">' + name + '</div></div></div></td>' +
                                            '<td>' + email + '</td>' +
                                            '<td>' + phone + '</td>' +
                                            '<td><span class="badge badge-info"><i class="fas fa-bell"></i> ' + alertCount + '</span></td>' +
                                            '<td><span class="badge badge-' + statusColor + '">' + status + '</span></td>' +
                                            '<td>' + created + '</td>' +
                                            '<td><div style="display:flex;gap:5px;"><button class="btn btn-primary btn-sm" onclick="alert(\'Ver detalles de ' + name + '\')"><i class="fas fa-eye"></i></button></div></td>' +
                                        '</tr>';
                                    }
                                    tbody.innerHTML = rows;
                                }
                                // Update stats if present
                                if (data.stats) {
                                    var s = data.stats;
                                    var el;
                                    if (el = document.getElementById('totalUsersCount')) el.textContent = s.total || 0;
                                    if (el = document.getElementById('activeUsersCount')) el.textContent = s.active || 0;
                                    if (el = document.getElementById('inactiveUsersCount')) el.textContent = s.inactive || 0;
                                    if (el = document.getElementById('newUsersCount')) el.textContent = s.new_users || 0;
                                }
                            })
                            .catch(function(err){
                                console.error('Users API error:', err);
                                if (tbody) tbody.innerHTML = '<tr><td colspan="8" style="text-align:center;padding:20px;color:red;">Error de conexión: ' + err.message + '</td></tr>';
                            });
                    } catch (e) { 
                        console.error('loadUsers exception:', e);
                    }
                };
            }

            if (!window.loadOperators) {
                window.loadOperators = function() {
                    try {
                        var tbody = document.getElementById('operatorsTableBody');
                        if (tbody) tbody.innerHTML = '<tr><td colspan="8" style="text-align:center;padding:20px;"><i class="fas fa-spinner fa-spin"></i> Cargando...</td></tr>';
                        
                        fetch('manage-operators.php?action=list')
                            .then(function(r){ 
                                console.log('Operators API status:', r.status);
                                return r.json(); 
                            })
                            .then(function(data){
                                console.log('Operators API response:', data);
                                if (!data || !data.success) {
                                    if (tbody) tbody.innerHTML = '<tr><td colspan="8" style="text-align:center;padding:20px;color:red;">Error: ' + (data.error || 'No se pudieron cargar operadores') + '</td></tr>';
                                    return;
                                }
                                var ops = data.operators || [];
                                if (!tbody) return;
                                if (ops.length === 0) {
                                    tbody.innerHTML = '<tr><td colspan="8" style="text-align:center;padding:40px;">Sin operadores</td></tr>';
                                } else {
                                    var rows = '';
                                    for (var i=0;i<ops.length;i++) {
                                        var o = ops[i];
                                        var initial = (o.name && o.name.charAt(0).toUpperCase()) || 'O';
                                        var name = o.name || 'Sin nombre';
                                        var email = o.email || '-';
                                        var role = o.role === 'admin' ? 'Administrador' : 'Operador';
                                        var roleColor = o.role === 'admin' ? 'danger' : 'primary';
                                        var handled = (typeof o.alerts_handled !== 'undefined') ? o.alerts_handled : 0;
                                        var status = o.status === 'active' ? 'Activo' : 'Inactivo';
                                        var statusColor = o.status === 'active' ? 'success' : 'danger';
                                        var lastLogin = o.last_login ? new Date(o.last_login).toLocaleString() : 'Nunca';
                                        rows += '<tr>' +
                                            '<td>#' + o.operator_id + '</td>' +
                                            '<td><div style="display:flex;align-items:center;gap:10px;"><div class="user-avatar" style="width:32px;height:32px;font-size:12px;background:linear-gradient(135deg,#667eea,#764ba2);color:white;display:flex;align-items:center;justify-content:center;border-radius:50%;">' + initial + '</div><div><div style="font-weight:500;">' + name + '</div></div></div></td>' +
                                            '<td>' + email + '</td>' +
                                            '<td><span class="badge badge-' + roleColor + '">' + role + '</span></td>' +
                                            '<td><span class="badge badge-info"><i class="fas fa-tasks"></i> ' + handled + '</span></td>' +
                                            '<td><span class="badge badge-' + statusColor + '">' + status + '</span></td>' +
                                            '<td>' + lastLogin + '</td>' +
                                            '<td><div style="display:flex;gap:5px;"><button class="btn btn-info btn-sm" onclick="alert(\'Ver detalles de ' + name + '\')"><i class="fas fa-key"></i></button></div></td>' +
                                        '</tr>';
                                    }
                                    tbody.innerHTML = rows;
                                }
                                // Update stats if present
                                if (data.stats) {
                                    var s = data.stats, el;
                                    if (el = document.getElementById('totalOperatorsCount')) el.textContent = s.total || 0;
                                    if (el = document.getElementById('activeOperatorsCount')) el.textContent = s.active || 0;
                                    if (el = document.getElementById('adminOperatorsCount')) el.textContent = s.admins || 0;
                                    if (el = document.getElementById('operatorAlertsCount')) el.textContent = s.total_alerts_handled || 0;
                                }
                            })
                            .catch(function(err){
                                console.error('Operators API error:', err);
                                if (tbody) tbody.innerHTML = '<tr><td colspan="8" style="text-align:center;padding:20px;color:red;">Error de conexión: ' + err.message + '</td></tr>';
                            });
                    } catch (e) { 
                        console.error('loadOperators exception:', e);
                    }
                };
            }
            
            // Modal handlers
            window.showAddUserModal = function() {
                var m = document.getElementById('addUserModal');
                if (m) m.style.display = 'flex';
            };
            window.closeAddUserModal = function() {
                var m = document.getElementById('addUserModal');
                if (m) m.style.display = 'none';
                var f = document.getElementById('addUserForm');
                if (f) f.reset();
            };
            window.showAddOperatorModal = function() {
                var m = document.getElementById('addOperatorModal');
                if (m) m.style.display = 'flex';
            };
            window.closeAddOperatorModal = function() {
                var m = document.getElementById('addOperatorModal');
                if (m) m.style.display = 'none';
                var f = document.getElementById('addOperatorForm');
                if (f) f.reset();
            };
        })();
    </script>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <h2>
                <i class="fas fa-shield-alt"></i>
                Azuero Safety Services S.A
            </h2>
            <p style="font-size: 11px; opacity: 0.85;">Somos tu seguridad</p>
        </div>

        <div class="sidebar-menu">
            <a href="#" class="menu-item active" onclick="showSection('dashboard', this); return false;">
                <i class="fas fa-home"></i>
                <span><?php echo $t['dashboard']; ?></span>
            </a>

            <div class="menu-section">Gestión</div>

            <a href="#" class="menu-item" onclick="showSection('users', this); return false;">
                <i class="fas fa-users"></i>
                <span><?php echo $t['users']; ?></span>
            </a>

            <a href="#" class="menu-item" onclick="showSection('operators', this); return false;">
                <i class="fas fa-user-shield"></i>
                <span><?php echo $t['operators']; ?></span>
            </a>

            <a href="alertas.php" class="menu-item">
                <i class="fas fa-exclamation-triangle"></i>
                <span>Alertas</span>
            </a>

            <div class="menu-section">Análisis</div>

            <a href="#" class="menu-item" onclick="showSection('statistics', this); return false;">
                <i class="fas fa-chart-line"></i>
                <span><?php echo $t['statistics']; ?></span>
            </a>

            <a href="ver-logs.php" class="menu-item" target="_blank">
                <i class="fas fa-file-alt"></i>
                <span>Logs</span>
            </a>

            <div class="menu-section">Sistema</div>

            <a href="#" class="menu-item" onclick="showSection('settings', this); return false;">
                <i class="fas fa-cog"></i>
                <span><?php echo $t['settings']; ?></span>
            </a>            <a href="logout.php" class="menu-item">
                <i class="fas fa-sign-out-alt"></i>
                <span><?php echo $t['logout']; ?></span>
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Header -->
        <div class="header">
            <div class="header-left">
                <h1><?php echo $t['admin_panel']; ?></h1>
                <div class="breadcrumb">
                    <i class="fas fa-home"></i> / <?php echo $t['dashboard']; ?>
                </div>
            </div>

            <div class="header-right">
                <div class="search-box">
                    <input type="text" placeholder="<?php echo $t['search']; ?>">
                    <i class="fas fa-search"></i>
                </div>

                <div class="notification-bell" onclick="toggleNotifications()">
                    <i class="fas fa-bell"></i>
                    <span class="notification-badge" id="notificationCount"><?php echo $stats['active_alerts']; ?></span>
                </div>

                <div class="theme-switcher">
                    <button class="theme-btn active" data-theme="light" onclick="changeTheme('light')">
                        <i class="fas fa-sun"></i> <?php echo $t['light']; ?>
                    </button>
                    <button class="theme-btn" data-theme="dark" onclick="changeTheme('dark')">
                        <i class="fas fa-moon"></i> <?php echo $t['dark']; ?>
                    </button>
                    <button class="theme-btn" data-theme="blue-dark" onclick="changeTheme('blue-dark')">
                        <i class="fas fa-water"></i> <?php echo $t['blue_dark']; ?>
                    </button>
                </div>

                <div class="user-profile">
                    <div class="user-avatar">
                        <?php echo strtoupper(substr($operator['full_name'], 0, 1)); ?>
                    </div>
                    <div class="user-info">
                        <h4><?php echo htmlspecialchars($operator['full_name']); ?></h4>
                        <p><?php echo htmlspecialchars($operator['role']); ?></p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Toast Container -->
        <div class="toast-container" id="toastContainer"></div>

        <!-- Content -->
        <div class="content">
            <!-- Dashboard Section -->
            <div class="section active" id="dashboard-section">
                <!-- Stats Cards -->
                <div class="stats-grid">
                    <div class="stat-card">
                        <div class="stat-header">
                            <div>
                                <div class="stat-value"><?php echo $stats['total_users']; ?></div>
                                <div class="stat-label"><?php echo $t['total_users']; ?></div>
                                <span class="stat-trend up">
                                    <i class="fas fa-arrow-up"></i> 12%
                                </span>
                            </div>
                            <div class="stat-icon blue">
                                <i class="fas fa-users"></i>
                            </div>
                        </div>
                    </div>

                    <div class="stat-card">
                        <div class="stat-header">
                            <div>
                                <div class="stat-value"><?php echo $stats['total_operators']; ?></div>
                                <div class="stat-label"><?php echo $t['total_operators']; ?></div>
                                <span class="stat-trend up">
                                    <i class="fas fa-arrow-up"></i> 5%
                                </span>
                            </div>
                            <div class="stat-icon purple">
                                <i class="fas fa-user-shield"></i>
                            </div>
                        </div>
                    </div>

                    <div class="stat-card">
                        <div class="stat-header">
                            <div>
                                <div class="stat-value"><?php echo $stats['active_alerts']; ?></div>
                                <div class="stat-label"><?php echo $t['active_alerts']; ?></div>
                                <span class="stat-trend <?php echo $stats['active_alerts'] > 5 ? 'up' : 'down'; ?>">
                                    <i class="fas fa-arrow-<?php echo $stats['active_alerts'] > 5 ? 'up' : 'down'; ?>"></i>
                                    <?php echo $stats['active_alerts']; ?>
                                </span>
                            </div>
                            <div class="stat-icon orange">
                                <i class="fas fa-exclamation-circle"></i>
                            </div>
                        </div>
                    </div>

                    <div class="stat-card">
                        <div class="stat-header">
                            <div>
                                <div class="stat-value"><?php echo $stats['resolved_today']; ?></div>
                                <div class="stat-label"><?php echo $t['resolved_today']; ?></div>
                                <span class="stat-trend up">
                                    <i class="fas fa-check"></i> Hoy
                                </span>
                            </div>
                            <div class="stat-icon green">
                                <i class="fas fa-check-circle"></i>
                            </div>
                        </div>
                    </div>

                    <div class="stat-card">
                        <div class="stat-header">
                            <div>
                                <div class="stat-value"><?php echo $stats['total_alerts']; ?></div>
                                <div class="stat-label"><?php echo $t['total_alerts']; ?></div>
                                <span class="stat-trend up">
                                    <i class="fas fa-arrow-up"></i> 24%
                                </span>
                            </div>
                            <div class="stat-icon red">
                                <i class="fas fa-bell"></i>
                            </div>
                        </div>
                    </div>

                    <div class="stat-card">
                        <div class="stat-header">
                            <div>
                                <div class="stat-value"><?php echo $stats['alerts_month']; ?></div>
                                <div class="stat-label"><?php echo $t['alerts_this_month']; ?></div>
                                <span class="stat-trend up">
                                    <i class="fas fa-calendar"></i> Mes
                                </span>
                            </div>
                            <div class="stat-icon cyan">
                                <i class="fas fa-chart-bar"></i>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Charts -->
                <div class="charts-grid">
                    <div class="chart-card">
                        <div class="chart-header">
                            <h3><?php echo $t['alert_distribution']; ?></h3>
                            <button class="btn btn-primary btn-sm">
                                <i class="fas fa-download"></i> Exportar
                            </button>
                        </div>
                        <div class="chart-container">
                            <canvas id="alertDistributionChart"></canvas>
                        </div>
                    </div>

                    <div class="chart-card">
                        <div class="chart-header">
                            <h3><?php echo $t['most_active_users']; ?></h3>
                            <button class="btn btn-primary btn-sm">
                                <i class="fas fa-eye"></i> <?php echo $t['view_all']; ?>
                            </button>
                        </div>
                        <div class="chart-container">
                            <canvas id="topUsersChart"></canvas>
                        </div>
                    </div>
                </div>

                <!-- Management Tools -->
                <div class="table-card" style="background: linear-gradient(135deg, rgba(52, 152, 219, 0.1) 0%, rgba(155, 89, 182, 0.1) 100%); border: 1px solid rgba(52, 152, 219, 0.2);">
                    <div class="table-header">
                        <div>
                            <h3 style="margin-bottom: 5px;">
                                <i class="fas fa-toolbox" style="margin-right: 8px; color: var(--primary);"></i>Herramientas de Administración
                            </h3>
                            <p style="color: var(--text-secondary); font-size: 13px;">Accede a las herramientas avanzadas para gestionar usuarios y operadores</p>
                        </div>
                    </div>
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(280px, 1fr)); gap: 15px; padding: 20px 0 0 0;">
                        <!-- Editar Usuarios y Operadores -->
                        <a href="usuarios-operadores.php" style="text-decoration: none;">
                            <div style="background: var(--bg-card); border: 2px solid var(--primary); border-radius: 12px; padding: 20px; cursor: pointer; transition: all 0.3s ease; display: flex; align-items: center; gap: 15px; height: 100%;" onmouseover="this.style.background='rgba(52, 152, 219, 0.1)'; this.style.transform='translateY(-3px)'; this.style.boxShadow='var(--shadow-md)';" onmouseout="this.style.background='var(--bg-card)'; this.style.transform='translateY(0)'; this.style.boxShadow='none';">
                                <div style="font-size: 36px; color: var(--primary);">
                                    <i class="fas fa-user-cog"></i>
                                </div>
                                <div>
                                    <h4 style="margin: 0 0 5px 0; color: var(--text-primary); font-size: 15px; font-weight: 600;">Gestión Completa</h4>
                                    <p style="margin: 0; color: var(--text-secondary); font-size: 13px;">Editar usuarios, operadores y cambiar contraseñas</p>
                                </div>
                            </div>
                        </a>

                        <!-- Nueva Funcionalidad Futura -->
                        <div style="background: var(--bg-card); border: 2px solid var(--border-color); border-radius: 12px; padding: 20px; display: flex; align-items: center; gap: 15px; opacity: 0.6; cursor: not-allowed; height: 100%;">
                            <div style="font-size: 36px; color: var(--text-secondary);">
                                <i class="fas fa-lock"></i>
                            </div>
                            <div>
                                <h4 style="margin: 0 0 5px 0; color: var(--text-secondary); font-size: 15px; font-weight: 600;">Permisos y Roles</h4>
                                <p style="margin: 0; color: var(--text-secondary); font-size: 13px;">Próximamente disponible</p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Recent Activity Table -->
                <div class="table-card">
                    <div class="table-header">
                        <h3><?php echo $t['recent_activity']; ?></h3>
                        <a href="alertas.php" class="btn btn-primary">
                            <i class="fas fa-list"></i> <?php echo $t['view_all']; ?>
                        </a>
                    </div>
                    <div class="table-wrapper">
                        <table>
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Usuario</th>
                                    <th>Tipo</th>
                                    <th>Estado</th>
                                    <th>Fecha</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach($recent_alerts as $alert): ?>
                                <tr>
                                    <td>#<?php echo $alert['id']; ?></td>
                                    <td>
                                        <div style="display: flex; align-items: center; gap: 10px;">
                                            <div class="user-avatar" style="width: 32px; height: 32px; font-size: 12px;">
                                                <?php echo strtoupper(substr($alert['user_name'] ?? 'U', 0, 1)); ?>
                                            </div>
                                            <div>
                                                <div style="font-weight: 500;"><?php echo htmlspecialchars($alert['user_name'] ?? 'Usuario'); ?></div>
                                                <div style="font-size: 12px; color: var(--text-secondary);"><?php echo htmlspecialchars($alert['user_email'] ?? ''); ?></div>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <?php
                                        $type_badges = [
                                            'medical' => ['icon' => 'fa-heartbeat', 'color' => 'danger', 'text' => $t['medical']],
                                            'fire' => ['icon' => 'fa-fire', 'color' => 'warning', 'text' => $t['fire']],
                                            'security' => ['icon' => 'fa-shield-alt', 'color' => 'info', 'text' => $t['security']]
                                        ];
                                        $badge = $type_badges[$alert['alert_type']] ?? ['icon' => 'fa-bell', 'color' => 'info', 'text' => $alert['alert_type']];
                                        ?>
                                        <span class="badge badge-<?php echo $badge['color']; ?>">
                                            <i class="fas <?php echo $badge['icon']; ?>"></i> <?php echo $badge['text']; ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php
                                        $status_badges = [
                                            'pending' => 'warning',
                                            'in_progress' => 'info',
                                            'resolved' => 'success'
                                        ];
                                        ?>
                                        <span class="badge badge-<?php echo $status_badges[$alert['status']] ?? 'info'; ?>">
                                            <?php echo ucfirst($alert['status']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo date('d/m/Y H:i', strtotime($alert['created_at'])); ?></td>
                                    <td>
                                        <a href="ver-alerta.php?id=<?php echo $alert['id']; ?>" class="btn btn-primary btn-sm">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Users Section -->
            <div class="section" id="users-section">
                <div class="table-card">
                    <div class="table-header">
                        <div>
                            <h3><?php echo $t['user_management']; ?></h3>
                            <p style="color: var(--text-secondary); font-size: 14px; margin-top: 5px;">Gestión completa de usuarios ciudadanos</p>
                        </div>
                        <div style="display: flex; gap: 10px;">
                            <input type="text" id="userSearchInput" placeholder="Buscar usuario..." style="padding: 10px 15px; border: 1px solid var(--border-color); border-radius: 8px; background: var(--bg-secondary); color: var(--text-primary);">
                            <button class="btn btn-success" onclick="showAddUserModal()">
                                <i class="fas fa-user-plus"></i> Nuevo Usuario
                            </button>
                            <button class="btn btn-primary" onclick="loadUsers()">
                                <i class="fas fa-sync"></i> Actualizar
                            </button>
                        </div>
                    </div>

                    <!-- User Stats -->
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-bottom: 20px;">
                        <div style="background: var(--bg-secondary); padding: 15px; border-radius: 10px;">
                            <div style="font-size: 24px; font-weight: bold; color: var(--primary);" id="totalUsersCount"><?php echo $stats['total_users']; ?></div>
                            <div style="font-size: 13px; color: var(--text-secondary);">Total Usuarios</div>
                        </div>
                        <div style="background: var(--bg-secondary); padding: 15px; border-radius: 10px;">
                            <div style="font-size: 24px; font-weight: bold; color: var(--success);" id="activeUsersCount">-</div>
                            <div style="font-size: 13px; color: var(--text-secondary);">Usuarios Activos</div>
                        </div>
                        <div style="background: var(--bg-secondary); padding: 15px; border-radius: 10px;">
                            <div style="font-size: 24px; font-weight: bold; color: var(--danger);" id="inactiveUsersCount">-</div>
                            <div style="font-size: 13px; color: var(--text-secondary);">Usuarios Inactivos</div>
                        </div>
                        <div style="background: var(--bg-secondary); padding: 15px; border-radius: 10px;">
                            <div style="font-size: 24px; font-weight: bold; color: var(--info);" id="newUsersCount">-</div>
                            <div style="font-size: 13px; color: var(--text-secondary);">Nuevos (7 días)</div>
                        </div>
                    </div>

                    <div class="table-wrapper">
                        <table id="usersTable">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Usuario</th>
                                    <th>Email</th>
                                    <th>Teléfono</th>
                                    <th>Alertas</th>
                                    <th>Estado</th>
                                    <th>Registro</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody id="usersTableBody">
                                <tr>
                                    <td colspan="8" style="text-align: center; padding: 40px;">
                                        <i class="fas fa-spinner fa-spin" style="font-size: 32px; color: var(--primary);"></i>
                                        <p style="margin-top: 10px; color: var(--text-secondary);">Cargando usuarios...</p>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Add User Modal -->
            <div id="addUserModal" style="display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.7); z-index: 10000; align-items: center; justify-content: center;">
                <div style="background: var(--bg-card); border-radius: 15px; padding: 30px; max-width: 500px; width: 90%; box-shadow: var(--shadow-lg);">
                    <h3 style="margin-bottom: 20px; color: var(--text-primary);">
                        <i class="fas fa-user-plus"></i> Nuevo Usuario
                    </h3>
                    
                    <form id="addUserForm" onsubmit="createUser(event)">
                        <div style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; color: var(--text-secondary); font-size: 14px;">Nombre Completo *</label>
                            <input type="text" name="name" required style="width: 100%; padding: 10px; border: 1px solid var(--border-color); border-radius: 8px; background: var(--bg-secondary); color: var(--text-primary);">
                        </div>
                        
                        <div style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; color: var(--text-secondary); font-size: 14px;">Email *</label>
                            <input type="email" name="email" required style="width: 100%; padding: 10px; border: 1px solid var(--border-color); border-radius: 8px; background: var(--bg-secondary); color: var(--text-primary);">
                        </div>
                        
                        <div style="margin-bottom: 15px; display: grid; grid-template-columns: 1fr 1fr; gap: 10px;">
                            <div>
                                <label style="display: block; margin-bottom: 5px; color: var(--text-secondary); font-size: 14px;">Teléfono *</label>
                                <input type="text" name="phone" required style="width: 100%; padding: 10px; border: 1px solid var(--border-color); border-radius: 8px; background: var(--bg-secondary); color: var(--text-primary);">
                            </div>
                            <div>
                                <label style="display: block; margin-bottom: 5px; color: var(--text-secondary); font-size: 14px;">Cédula</label>
                                <input type="text" name="cedula" placeholder="0-000-0000" style="width: 100%; padding: 10px; border: 1px solid var(--border-color); border-radius: 8px; background: var(--bg-secondary); color: var(--text-primary);">
                            </div>
                        </div>
                        
                        <div style="margin-bottom: 20px;">
                            <label style="display: block; margin-bottom: 5px; color: var(--text-secondary); font-size: 14px;">Usuario (opcional)</label>
                            <input type="text" name="username" placeholder="Se generará automáticamente" style="width: 100%; padding: 10px; border: 1px solid var(--border-color); border-radius: 8px; background: var(--bg-secondary); color: var(--text-primary);">
                            <small style="color: var(--text-secondary); font-size: 12px;">Contraseña temporal se generará automáticamente</small>
                        </div>
                        
                        <div style="display: flex; gap: 10px; justify-content: flex-end;">
                            <button type="button" class="btn" onclick="closeAddUserModal()" style="background: var(--text-secondary); color: white;">
                                <i class="fas fa-times"></i> Cancelar
                            </button>
                            <button type="submit" class="btn btn-success">
                                <i class="fas fa-check"></i> Crear Usuario
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Operators Section -->
            <div class="section" id="operators-section">
                <div class="table-card">
                    <div class="table-header">
                        <div>
                            <h3><?php echo $t['operator_management']; ?></h3>
                            <p style="color: var(--text-secondary); font-size: 14px; margin-top: 5px;">Gestión de operadores del sistema</p>
                        </div>
                        <div style="display: flex; gap: 10px;">
                            <input type="text" id="operatorSearchInput" placeholder="Buscar operador..." style="padding: 10px 15px; border: 1px solid var(--border-color); border-radius: 8px; background: var(--bg-secondary); color: var(--text-primary);">
                            <button class="btn btn-success" onclick="showAddOperatorModal()">
                                <i class="fas fa-plus"></i> Nuevo Operador
                            </button>
                            <button class="btn btn-primary" onclick="loadOperators()">
                                <i class="fas fa-sync"></i> Actualizar
                            </button>
                        </div>
                    </div>

                    <!-- Operator Stats -->
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-bottom: 20px;">
                        <div style="background: var(--bg-secondary); padding: 15px; border-radius: 10px;">
                            <div style="font-size: 24px; font-weight: bold; color: var(--primary);" id="totalOperatorsCount"><?php echo $stats['total_operators']; ?></div>
                            <div style="font-size: 13px; color: var(--text-secondary);">Total Operadores</div>
                        </div>
                        <div style="background: var(--bg-secondary); padding: 15px; border-radius: 10px;">
                            <div style="font-size: 24px; font-weight: bold; color: var(--success);" id="activeOperatorsCount">-</div>
                            <div style="font-size: 13px; color: var(--text-secondary);">Activos</div>
                        </div>
                        <div style="background: var(--bg-secondary); padding: 15px; border-radius: 10px;">
                            <div style="font-size: 24px; font-weight: bold; color: var(--info);" id="adminOperatorsCount">-</div>
                            <div style="font-size: 13px; color: var(--text-secondary);">Administradores</div>
                        </div>
                        <div style="background: var(--bg-secondary); padding: 15px; border-radius: 10px;">
                            <div style="font-size: 24px; font-weight: bold; color: var(--warning);" id="operatorAlertsCount">-</div>
                            <div style="font-size: 13px; color: var(--text-secondary);">Alertas Atendidas</div>
                        </div>
                    </div>

                    <div class="table-wrapper">
                        <table id="operatorsTable">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Operador</th>
                                    <th>Email</th>
                                    <th>Rol</th>
                                    <th>Alertas Atendidas</th>
                                    <th>Estado</th>
                                    <th>Último Acceso</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody id="operatorsTableBody">
                                <tr>
                                    <td colspan="8" style="text-align: center; padding: 40px;">
                                        <i class="fas fa-spinner fa-spin" style="font-size: 32px; color: var(--primary);"></i>
                                        <p style="margin-top: 10px; color: var(--text-secondary);">Cargando operadores...</p>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Add Operator Modal -->
            <div id="addOperatorModal" style="display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.7); z-index: 10000; align-items: center; justify-content: center;">
                <div style="background: var(--bg-card); border-radius: 15px; padding: 30px; max-width: 500px; width: 90%; box-shadow: var(--shadow-lg);">
                    <h3 style="margin-bottom: 20px; color: var(--text-primary);">
                        <i class="fas fa-user-plus"></i> Nuevo Operador
                    </h3>
                    
                    <form id="addOperatorForm" onsubmit="createOperator(event)">
                        <div style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; color: var(--text-secondary); font-size: 14px;">Nombre Completo</label>
                            <input type="text" name="name" required style="width: 100%; padding: 10px; border: 1px solid var(--border-color); border-radius: 8px; background: var(--bg-secondary); color: var(--text-primary);">
                        </div>
                        
                        <div style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; color: var(--text-secondary); font-size: 14px;">Email</label>
                            <input type="email" name="email" required style="width: 100%; padding: 10px; border: 1px solid var(--border-color); border-radius: 8px; background: var(--bg-secondary); color: var(--text-primary);">
                        </div>
                        
                        <div style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; color: var(--text-secondary); font-size: 14px;">Contraseña</label>
                            <input type="password" name="password" required minlength="6" style="width: 100%; padding: 10px; border: 1px solid var(--border-color); border-radius: 8px; background: var(--bg-secondary); color: var(--text-primary);">
                        </div>
                        
                        <div style="margin-bottom: 20px;">
                            <label style="display: block; margin-bottom: 5px; color: var(--text-secondary); font-size: 14px;">Rol</label>
                            <select name="role" required style="width: 100%; padding: 10px; border: 1px solid var(--border-color); border-radius: 8px; background: var(--bg-secondary); color: var(--text-primary);">
                                <option value="operator">Operador</option>
                                <option value="admin">Administrador</option>
                            </select>
                        </div>
                        
                        <div style="display: flex; gap: 10px; justify-content: flex-end;">
                            <button type="button" class="btn" onclick="closeAddOperatorModal()" style="background: var(--text-secondary); color: white;">
                                <i class="fas fa-times"></i> Cancelar
                            </button>
                            <button type="submit" class="btn btn-success">
                                <i class="fas fa-check"></i> Crear Operador
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Statistics Section (placeholder) -->
            <div class="section" id="statistics-section">
                <h2><?php echo $t['statistics']; ?></h2>
                <p style="color: var(--text-secondary);">Estadísticas avanzadas (próximamente)</p>
            </div>

            <!-- Settings Section (placeholder) -->
            <div class="section" id="settings-section">
                <h2><?php echo $t['settings']; ?></h2>
                <p style="color: var(--text-secondary);">Configuración del sistema (próximamente)</p>
            </div>
        </div>
    </div>

    <!-- Alarm Sound -->
    <audio id="alarmSound" preload="auto">
        <source src="data:audio/wav;base64,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" type="audio/wav">
    </audio>

    <script>
        // Theme Management
        function changeTheme(theme) {
            document.body.setAttribute('data-theme', theme);
            localStorage.setItem('theme', theme);
            
            // Update active button
            document.querySelectorAll('.theme-btn').forEach(btn => {
                btn.classList.remove('active');
                if(btn.getAttribute('data-theme') === theme) {
                    btn.classList.add('active');
                }
            });
        }

        // Load saved theme
        const savedTheme = localStorage.getItem('theme') || 'light';
        changeTheme(savedTheme);

        // Section Management
        function showSection(sectionName, element) {
            // Hide all sections
            document.querySelectorAll('.section').forEach(section => {
                section.classList.remove('active');
            });
            
            // Show selected section
            const targetSection = document.getElementById(sectionName + '-section');
            if(targetSection) {
                targetSection.classList.add('active');
            }
            
            // Update menu items
            document.querySelectorAll('.menu-item').forEach(item => {
                item.classList.remove('active');
            });
            
            // Set active menu item
            if(element) {
                const menuItem = element.closest ? element.closest('.menu-item') : element;
                if(menuItem) {
                    menuItem.classList.add('active');
                }
            }
            
            // Load data based on section
            if(sectionName === 'users') {
                loadUsers();
            } else if(sectionName === 'operators') {
                loadOperators();
            }
        }

        // Notification state and utilities
        var lastCheckTs = parseInt(localStorage.getItem('lastCheckTs') || '0', 10) || Math.floor(Date.now()/1000);
        var notifiedAlerts = [];
        try { notifiedAlerts = JSON.parse(localStorage.getItem('notifiedAlerts') || '[]'); } catch(e) { notifiedAlerts = []; }
        function saveNotified() {
            try { localStorage.setItem('notifiedAlerts', JSON.stringify(notifiedAlerts.slice(-100))); } catch(e) {}
        }
        var lastAlarmMs = 0;
        var alarmInterval = null;
        var alarmActive = false;
        
        function playAlarmThrottled() {
            var now = Date.now();
            if (now - lastAlarmMs > 8000) { // 8s cooldown between alarm sounds
                lastAlarmMs = now;
                var a = document.getElementById('alarmSound');
                if (a) {
                    try { a.currentTime = 0; a.play(); } catch(err) {}
                }
            }
        }
        
        // Nueva función para alarma persistente
        function startPersistentAlarm() {
            if (alarmActive) return; // Ya está sonando
            
            alarmActive = true;
            var a = document.getElementById('alarmSound');
            
            // Reproducir inmediatamente
            if (a) {
                try { 
                    a.currentTime = 0; 
                    a.play(); 
                } catch(err) {
                    console.warn('No se pudo reproducir alarma:', err);
                }
            }
            
            // Repetir cada 3 segundos hasta que se detenga
            alarmInterval = setInterval(function() {
                if (a && alarmActive) {
                    try { 
                        a.currentTime = 0; 
                        a.play(); 
                    } catch(err) {}
                }
            }, 3000);
        }
        
        function stopAlarm() {
            alarmActive = false;
            if (alarmInterval) {
                clearInterval(alarmInterval);
                alarmInterval = null;
            }
            var a = document.getElementById('alarmSound');
            if (a) {
                try { a.pause(); a.currentTime = 0; } catch(err) {}
            }
        }

        // Toast Notifications
        function showToast(type, title, message) {
            const toast = document.createElement('div');
            toast.className = `toast ${type}`;
            
            const icons = {
                'emergency': 'fa-exclamation-circle',
                'warning': 'fa-exclamation-triangle',
                'info': 'fa-info-circle',
                'success': 'fa-check-circle'
            };
            
            // Botón para detener alarma en notificaciones de emergencia
            var stopAlarmBtn = '';
            if (type === 'emergency') {
                stopAlarmBtn = `
                    <button class="btn btn-sm btn-danger" onclick="stopAlarm(); this.disabled=true; this.innerHTML='<i class=\\'fas fa-check\\'></i> Silenciada';" 
                            style="margin: 10px 0 5px 0; width: 100%; font-size: 12px; padding: 6px 12px; border: none; border-radius: 6px; cursor: pointer;">
                        <i class="fas fa-volume-mute"></i> Detener Alarma
                    </button>`;
            }
            
            toast.innerHTML = `
                <div class="toast-icon">
                    <i class="fas ${icons[type]}"></i>
                </div>
                <div class="toast-content">
                    <h4>${title}</h4>
                    <p>${message}</p>
                    ${stopAlarmBtn}
                    <div class="toast-time">${new Date().toLocaleTimeString()}</div>
                </div>
                <button class="toast-close" onclick="this.parentElement.remove()">
                    <i class="fas fa-times"></i>
                </button>
            `;
            
            document.getElementById('toastContainer').appendChild(toast);
            
            // Play alarm for emergency (persistente)
            if(type === 'emergency') {
                startPersistentAlarm();
            }
            
            // Auto remove after 5 seconds
            setTimeout(() => {
                toast.style.animation = 'slideIn 0.3s ease reverse';
                setTimeout(() => toast.remove(), 300);
            }, 5000);
        }

        // Check for new alerts
        function checkNewAlerts() {
            var url = 'check-new-alerts.php?last_check=' + encodeURIComponent(lastCheckTs || 0);
            fetch(url)
                .then(response => response.json())
                .then(data => {
                    if(!data || data.error) return;
                    // Compute truly new alerts since lastCheckTs and not already notified this session
                    var newAlerts = (data.alerts || []).filter(function(a){
                        return a && a.timestamp && a.timestamp > (lastCheckTs || 0) && notifiedAlerts.indexOf(a.id) === -1;
                    });

                    if(newAlerts.length > 0) {
                        // Update badge
                        var badgeEl = document.getElementById('notificationCount');
                        if (badgeEl && data.stats) {
                            badgeEl.textContent = (data.stats.pending + data.stats.in_progress);
                        }
                        // Show one toast for the first new alert
                        var alert = newAlerts[0];
                        function esc(s){return String(s||'').replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;').replace(/\"/g,'&quot;').replace(/'/g,'&#39;');}
                        var desc = alert.description ? esc(alert.description).slice(0,140) : '';
                        var extra = desc ? ' — "' + desc + (alert.description && alert.description.length>140 ? '…' : '') + '"' : '';
                        var tipo = (alert && alert.alert_type) ? alert.alert_type : 'emergencia';
                        
                        // Agregar información de ubicación GPS si existe
                        var gpsInfo = '';
                        if (alert.latitude && alert.longitude) {
                            gpsInfo = '<br><small style="font-size:12px;opacity:0.9;"><i class="fas fa-map-marker-alt"></i> GPS: ' + 
                                     parseFloat(alert.latitude).toFixed(6) + ', ' + 
                                     parseFloat(alert.longitude).toFixed(6) + '</small>';
                        }
                        
                        var linkHtml = ' <a href="ver-alerta.php?id=' + alert.id + '" class="btn btn-primary btn-sm" style="margin-left:8px;text-decoration:none;">Ver alerta</a>';
                        showToast('emergency', '¡Nueva Alerta!', 'Alerta ' + esc(tipo) + extra + '.' + gpsInfo + linkHtml);
                        // Mark all new alerts as notified in this session
                        newAlerts.forEach(function(a){ notifiedAlerts.push(a.id); });
                        saveNotified();
                    }
                    // Advance lastCheckTs to server timestamp (or now) to avoid re-alerting
                    if (typeof data.timestamp === 'number') {
                        lastCheckTs = data.timestamp;
                    } else {
                        lastCheckTs = Math.floor(Date.now()/1000);
                    }
                    try { localStorage.setItem('lastCheckTs', String(lastCheckTs)); } catch(e) {}
                })
                .catch(error => console.error('Error checking alerts:', error));
        }

        // Poll for new alerts every 5 seconds
        setInterval(checkNewAlerts, 5000);

        // Initialize Charts
        const alertDistData = <?php echo json_encode($alert_distribution); ?>;
        const topUsersData = <?php echo json_encode($most_active_users); ?>;

        // Alert Distribution Chart (Doughnut)
        const ctx1 = document.getElementById('alertDistributionChart').getContext('2d');
        new Chart(ctx1, {
            type: 'doughnut',
            data: {
                labels: alertDistData.map(item => item.alert_type.toUpperCase()),
                datasets: [{
                    data: alertDistData.map(item => item.count),
                    backgroundColor: [
                        'rgba(239, 68, 68, 0.8)',
                        'rgba(245, 158, 11, 0.8)',
                        'rgba(59, 130, 246, 0.8)',
                        'rgba(16, 185, 129, 0.8)',
                        'rgba(139, 92, 246, 0.8)'
                    ],
                    borderWidth: 2,
                    borderColor: getComputedStyle(document.body).getPropertyValue('--bg-card')
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom',
                        labels: {
                            color: getComputedStyle(document.body).getPropertyValue('--text-primary'),
                            padding: 15,
                            font: {
                                size: 12
                            }
                        }
                    }
                }
            }
        });

        // Top Users Chart (Bar)
        const ctx2 = document.getElementById('topUsersChart').getContext('2d');
        new Chart(ctx2, {
            type: 'bar',
            data: {
                labels: topUsersData.map(user => user.name || 'Usuario'),
                datasets: [{
                    label: 'Alertas',
                    data: topUsersData.map(user => user.alert_count),
                    backgroundColor: 'rgba(102, 126, 234, 0.8)',
                    borderColor: 'rgba(102, 126, 234, 1)',
                    borderWidth: 2,
                    borderRadius: 8
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            color: getComputedStyle(document.body).getPropertyValue('--text-secondary')
                        },
                        grid: {
                            color: getComputedStyle(document.body).getPropertyValue('--border-color')
                        }
                    },
                    x: {
                        ticks: {
                            color: getComputedStyle(document.body).getPropertyValue('--text-secondary')
                        },
                        grid: {
                            display: false
                        }
                    }
                }
            }
        });

        // Test notification on load
        setTimeout(() => {
            // showToast('info', 'Sistema Iniciado', 'Panel administrativo cargado correctamente');
        }, 1000);

        // ==================== USER MANAGEMENT ====================
        let currentUsers = [];
        let currentOperators = [];

        // Load users on page load
        document.addEventListener('DOMContentLoaded', function() {
            loadUsers();
            loadOperators();
            
            // Search functionality (avoid optional chaining for wider compatibility)
            const userSearchEl = document.getElementById('userSearchInput');
            if (userSearchEl) {
                userSearchEl.addEventListener('input', function(e) {
                    const searchTerm = e.target.value.toLowerCase();
                    filterUsers(searchTerm);
                });
            }
            
            const operatorSearchEl = document.getElementById('operatorSearchInput');
            if (operatorSearchEl) {
                operatorSearchEl.addEventListener('input', function(e) {
                    const searchTerm = e.target.value.toLowerCase();
                    filterOperators(searchTerm);
                });
            }
        });

        // ===== Users Create Modal Handlers =====
        function showAddUserModal() {
            var m = document.getElementById('addUserModal');
            if (m) { m.style.display = 'flex'; }
        }

        function closeAddUserModal() {
            var m = document.getElementById('addUserModal');
            if (m) { m.style.display = 'none'; }
            var f = document.getElementById('addUserForm');
            if (f) { f.reset(); }
        }

        function createUser(event) {
            event.preventDefault();
            var form = event.target;
            var formData = new FormData(form);
            formData.append('action', 'create');
            
            fetch('manage-users.php', {
                method: 'POST',
                body: formData
            })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    let msg = 'Usuario creado correctamente';
                    let alertMsg = '✅ Usuario creado exitosamente\n\n';
                    
                    // Mostrar username si se proporcionó
                    var usernameInput = form.querySelector('[name="username"]');
                    var username = usernameInput ? usernameInput.value : '';
                    if (username) {
                        alertMsg += 'Usuario: ' + username + '\n';
                    }
                    
                    // Mostrar email
                    var emailInput = form.querySelector('[name="email"]');
                    if (emailInput && emailInput.value) {
                        alertMsg += 'Email: ' + emailInput.value + '\n';
                    }
                    
                    if (data.temp_password) {
                        msg += '<br><strong>Contraseña temporal:</strong> <code style="background:#f0f0f0;padding:2px 8px;border-radius:4px;">' + data.temp_password + '</code>';
                        alertMsg += '\n🔑 Contraseña temporal: ' + data.temp_password;
                        alertMsg += '\n\n⚠️ IMPORTANTE: Guarda esta contraseña ya que no se volverá a mostrar.';
                        alertMsg += '\n\nEl usuario puede iniciar sesión con:';
                        if (username) alertMsg += '\n• Usuario: ' + username;
                        alertMsg += '\n• Email: ' + (emailInput ? emailInput.value : '');
                    }
                    
                    alert(alertMsg);
                    showToast('success', 'Éxito', msg);
                    closeAddUserModal();
                    form.reset();
                    loadUsers();
                } else {
                    showToast('warning', 'Error', data.error || 'No se pudo crear el usuario');
                }
            })
            .catch(err => {
                console.error(err);
                showToast('warning', 'Error', 'Error de conexión');
            });
        }

        /* COMMENTED OUT - Using fallback version defined earlier to avoid syntax errors
        function loadUsers() {
            fetch('manage-users.php?action=list')
                .then(response => response.json())
                .then(data => {
                    if(data.success) {
                        currentUsers = data.users;
                        displayUsers(currentUsers);
                        updateUserStats(data.stats);
                    } else {
                        showToast('warning', 'Error', data.error || 'No se pudieron cargar los usuarios');
                    }
                })
                .catch(error => {
                    console.error('Error loading users:', error);
                    showToast('warning', 'Error', 'Error de conexión al cargar usuarios');
                });
        }

        function displayUsers(users) {
            const tbody = document.getElementById('usersTableBody');
            
            if(users.length === 0) {
                tbody.innerHTML = `
                    <tr>
                        <td colspan="8" style="text-align: center; padding: 40px;">
                            <i class="fas fa-users" style="font-size: 48px; color: var(--text-secondary); opacity: 0.5;"></i>
                            <p style="margin-top: 15px; color: var(--text-secondary);">No se encontraron usuarios</p>
                        </td>
                    </tr>
                `;
                return;
            }
            
            tbody.innerHTML = users.map(user => {
                const statusColor = user.status === 'active' ? 'success' : 'danger';
                const statusText = user.status === 'active' ? 'Activo' : 'Inactivo';
                const statusIcon = user.status === 'active' ? 'fa-check-circle' : 'fa-times-circle';
                const statusAction = user.status === 'active' ? 'inactive' : 'active';
                const statusBtnClass = user.status === 'active' ? 'btn-warning' : 'btn-success';
                const statusBtnIcon = user.status === 'active' ? 'ban' : 'check';
                const statusBtnTitle = user.status === 'active' ? 'Desactivar' : 'Activar';
                const userInitial = user.name ? user.name.charAt(0).toUpperCase() : 'U';
                const userName = user.name || 'Sin nombre';
                const userEmail = user.email || '-';
                const userPhone = user.phone || '-';
                
                return `
                    <tr>
                        <td>#${user.user_id}</td>
                        <td>
                            <div style="display: flex; align-items: center; gap: 10px;">
                                <div class="user-avatar" style="width: 32px; height: 32px; font-size: 12px;">
                                    ${userInitial}
                                </div>
                                <div>
                                    <div style="font-weight: 500;">${userName}</div>
                                </div>
                            </div>
                        </td>
                        <td>${userEmail}</td>
                        <td>${userPhone}</td>
                        <td>
                            <span class="badge badge-info">
                                <i class="fas fa-bell"></i> ${user.alert_count}
                            </span>
                        </td>
                        <td>
                            <span class="badge badge-${statusColor}">
                                <i class="fas ${statusIcon}"></i> ${statusText}
                            </span>
                        </td>
                        <td>${new Date(user.created_at).toLocaleDateString()}</td>
                        <td>
                            <div style="display: flex; gap: 5px;">
                                <button class="btn btn-primary btn-sm" onclick="viewUserDetails(${user.user_id})" title="Ver detalles">
                                    <i class="fas fa-eye"></i>
                                </button>
                                <button class="btn ${statusBtnClass} btn-sm" data-user-id="${user.user_id}" data-status="${statusAction}" onclick="toggleUserStatusClick(this)" title="${statusBtnTitle}">
                                    <i class="fas fa-${statusBtnIcon}"></i>
                                </button>
                                <button class="btn btn-info btn-sm" onclick="openPasswordModal('user', ${user.user_id})" title="Cambiar/Resetear contraseña">
                                    <i class="fas fa-key"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                `;
            }).join('');
        }
        END COMMENT */

        function updateUserStats(stats) {
            document.getElementById('totalUsersCount').textContent = stats.total || 0;
            document.getElementById('activeUsersCount').textContent = stats.active || 0;
            document.getElementById('inactiveUsersCount').textContent = stats.inactive || 0;
            document.getElementById('newUsersCount').textContent = stats.new_users || 0;
        }

        function filterUsers(searchTerm) {
            const filtered = currentUsers.filter(user => {
                return (
                    (user.name && user.name.toLowerCase().includes(searchTerm)) ||
                    (user.email && user.email.toLowerCase().includes(searchTerm)) ||
                    (user.phone && user.phone.includes(searchTerm)) ||
                    user.user_id.toString().includes(searchTerm)
                );
            });
            displayUsers(filtered);
        }

        function toggleUserStatusClick(button) {
            const userId = button.getAttribute('data-user-id');
            const newStatus = button.getAttribute('data-status');
            toggleUserStatus(userId, newStatus);
        }

        function toggleUserStatus(userId, newStatus) {
            const statusText = newStatus === 'active' ? 'activar' : 'desactivar';
            
            if(!confirm(`¿Está seguro de ${statusText} este usuario?`)) {
                return;
            }
            
            const formData = new FormData();
            formData.append('action', 'toggle_status');
            formData.append('user_id', userId);
            formData.append('status', newStatus);
            
            fetch('manage-users.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if(data.success) {
                    showToast('success', 'Éxito', data.message);
                    loadUsers(); // Recargar lista
                } else {
                    showToast('warning', 'Error', data.error);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showToast('warning', 'Error', 'Error al cambiar el estado');
            });
        }

        function resetUserPassword(userId) {
            // Mostrar modal con opciones
            const newPassword = prompt('🔑 CAMBIAR CONTRASEÑA DE USUARIO\n\nOpciones:\n1. Dejar vacío para generar contraseña automática\n2. Escribir una contraseña específica (mínimo 4 caracteres)\n\nIngrese la nueva contraseña o deje vacío:');
            
            // Si presiona cancelar
            if(newPassword === null) {
                return;
            }
            
            // Validar si es contraseña personalizada
            if(newPassword.trim() !== '' && newPassword.trim().length < 4) {
                alert('❌ La contraseña debe tener al menos 4 caracteres');
                return;
            }
            
            const formData = new FormData();
            formData.append('action', 'reset_password');
            formData.append('user_id', userId);
            if(newPassword.trim() !== '') {
                formData.append('new_password', newPassword.trim());
            }
            
            fetch('manage-users.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if(data.success) {
                    const passwordType = data.is_custom ? '✅ Contraseña Personalizada Establecida' : '🔄 Contraseña Temporal Generada';
                    const message = `${passwordType}\n\nNueva contraseña: ${data.temp_password}\nEmail del usuario: ${data.user_email}\n\n⚠️ Guarde esta información y envíela al usuario.`;
                    
                    // Copiar al portapapeles
                    navigator.clipboard.writeText(data.temp_password).then(() => {
                        alert(message + '\n\n📋 Contraseña copiada al portapapeles');
                    }).catch(() => {
                        alert(message);
                    });
                    
                    showToast('success', 'Contraseña Actualizada', `Nueva contraseña: <strong>${data.temp_password}</strong>`);
                } else {
                    showToast('warning', 'Error', data.error);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showToast('warning', 'Error', 'Error al cambiar la contraseña');
            });
        }

        function viewUserDetails(userId) {
            fetch(`manage-users.php?action=get_user_details&user_id=${userId}`)
                .then(response => response.json())
                .then(data => {
                    if(data.success) {
                        const user = data.user;
                        const statusDisplayText = user.status === 'active' ? '✅ Activo' : '❌ Inactivo';
                        const lastAlertText = user.last_alert_date ? new Date(user.last_alert_date).toLocaleString() : 'Nunca';
                        const details = `
📋 Detalles del Usuario #${user.user_id}

👤 Nombre: ${user.name || 'No especificado'}
📧 Email: ${user.email || 'No especificado'}
📱 Teléfono: ${user.phone || 'No especificado'}
🏠 Dirección: ${user.address || 'No especificada'}

📊 Estadísticas:
━━━━━━━━━━━━━━━━━━━━━━━━━━
🔔 Total de Alertas: ${user.total_alerts}
🏥 Alertas Médicas: ${user.medical_alerts}
🔥 Alertas de Incendio: ${user.fire_alerts}
🛡️ Alertas de Seguridad: ${user.security_alerts}
✅ Alertas Resueltas: ${user.resolved_alerts}
📅 Última Alerta: ${lastAlertText}

📅 Registro: ${new Date(user.created_at).toLocaleString()}
🔄 Estado: ${statusDisplayText}
                        `;
                        alert(details);
                    } else {
                        showToast('warning', 'Error', data.error);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showToast('warning', 'Error', 'Error al cargar los detalles');
                });
        }

        // ==================== OPERATOR MANAGEMENT ====================
        /* COMMENTED OUT - Using fallback version to avoid syntax errors
        function loadOperators() {
            fetch('manage-operators.php?action=list')
                .then(response => response.json())
                .then(data => {
                    if(data.success) {
                        currentOperators = data.operators;
                        displayOperators(currentOperators);
                        updateOperatorStats(data.stats);
                    } else {
                        showToast('warning', 'Error', data.error || 'No se pudieron cargar los operadores');
                    }
                })
                .catch(error => {
                    console.error('Error loading operators:', error);
                    showToast('warning', 'Error', 'Error de conexión al cargar operadores');
                });
        }

        function displayOperators(operators) {
            const tbody = document.getElementById('operatorsTableBody');
            
            if(operators.length === 0) {
                tbody.innerHTML = `
                    <tr>
                        <td colspan="8" style="text-align: center; padding: 40px;">
                            <i class="fas fa-user-shield" style="font-size: 48px; color: var(--text-secondary); opacity: 0.5;"></i>
                            <p style="margin-top: 15px; color: var(--text-secondary);">No se encontraron operadores</p>
                        </td>
                    </tr>
                `;
                return;
            }
            
            tbody.innerHTML = operators.map(op => {
                const statusColor = op.status === 'active' ? 'success' : 'danger';
                const statusText = op.status === 'active' ? 'Activo' : 'Inactivo';
                const roleColor = op.role === 'admin' ? 'info' : 'warning';
                const roleText = op.role === 'admin' ? 'Admin' : 'Operador';
                const roleIcon = op.role === 'admin' ? 'fa-user-shield' : 'fa-user';
                const statusAction = op.status === 'active' ? 'inactive' : 'active';
                const statusBtnClass = op.status === 'active' ? 'btn-warning' : 'btn-success';
                const statusBtnIcon = op.status === 'active' ? 'ban' : 'check';
                const statusBtnTitle = op.status === 'active' ? 'Desactivar' : 'Activar';
                const opInitial = op.name ? op.name.charAt(0).toUpperCase() : 'O';
                const opName = op.name || 'Sin nombre';
                const opEmail = op.email || '-';
                const lastLogin = op.last_login ? new Date(op.last_login).toLocaleString() : 'Nunca';
                
                return `
                    <tr>
                        <td>#${op.operator_id}</td>
                        <td>
                            <div style="display: flex; align-items: center; gap: 10px;">
                                <div class="user-avatar" style="width: 32px; height: 32px; font-size: 12px; background: linear-gradient(135deg, var(--primary), var(--secondary));">
                                    ${opInitial}
                                </div>
                                <div>
                                    <div style="font-weight: 500;">${opName}</div>
                                </div>
                            </div>
                        </td>
                        <td>${opEmail}</td>
                        <td>
                            <span class="badge badge-${roleColor}">
                                <i class="fas ${roleIcon}"></i> ${roleText}
                            </span>
                        </td>
                        <td>
                            <span class="badge badge-info">
                                <i class="fas fa-tasks"></i> ${op.alerts_handled}
                            </span>
                        </td>
                        <td>
                            <span class="badge badge-${statusColor}">
                                ${statusText}
                            </span>
                        </td>
                        <td>${lastLogin}</td>
                        <td>
                            <div style="display: flex; gap: 5px;">
                                ${op.operator_id != <?php echo $_SESSION['operator_id']; ?> ? `
                                    <button class="btn ${statusBtnClass} btn-sm" data-operator-id="${op.operator_id}" data-status="${statusAction}" onclick="toggleOperatorStatusClick(this)" title="${statusBtnTitle}">
                                        <i class="fas fa-${statusBtnIcon}"></i>
                                    </button>
                                ` : ''}
                                <button class="btn btn-info btn-sm" onclick="openPasswordModal('operator', ${op.operator_id})" title="Cambiar/Resetear contraseña">
                                    <i class="fas fa-key"></i>
                                </button>
                                ${op.operator_id != <?php echo $_SESSION['operator_id']; ?> && '<?php echo $currentOperator['role']; ?>' === 'admin' ? `
                                    <button class="btn btn-danger btn-sm" onclick="deleteOperator(${op.operator_id})" title="Eliminar">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                ` : ''}
                            </div>
                        </td>
                    </tr>
                `;
            }).join('');
        }
        END COMMENT */

        function updateOperatorStats(stats) {
            document.getElementById('totalOperatorsCount').textContent = stats.total || 0;
            document.getElementById('activeOperatorsCount').textContent = stats.active || 0;
            document.getElementById('adminOperatorsCount').textContent = stats.admins || 0;
            document.getElementById('operatorAlertsCount').textContent = stats.total_alerts_handled || 0;
        }

        function filterOperators(searchTerm) {
            const filtered = currentOperators.filter(op => {
                return (
                    (op.name && op.name.toLowerCase().includes(searchTerm)) ||
                    (op.email && op.email.toLowerCase().includes(searchTerm)) ||
                    (op.role && op.role.toLowerCase().includes(searchTerm)) ||
                    op.operator_id.toString().includes(searchTerm)
                );
            });
            displayOperators(filtered);
        }

        function showAddOperatorModal() {
            document.getElementById('addOperatorModal').style.display = 'flex';
        }

        function closeAddOperatorModal() {
            document.getElementById('addOperatorModal').style.display = 'none';
            document.getElementById('addOperatorForm').reset();
        }

        function createOperator(event) {
            event.preventDefault();
            
            const formData = new FormData(event.target);
            formData.append('action', 'create');
            
            fetch('manage-operators.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if(data.success) {
                    showToast('success', 'Éxito', data.message);
                    closeAddOperatorModal();
                    loadOperators();
                } else {
                    showToast('warning', 'Error', data.error);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showToast('warning', 'Error', 'Error al crear el operador');
            });
        }

        function toggleOperatorStatusClick(button) {
            const operatorId = button.getAttribute('data-operator-id');
            const newStatus = button.getAttribute('data-status');
            toggleOperatorStatus(operatorId, newStatus);
        }

        function toggleOperatorStatus(operatorId, newStatus) {
            const statusText = newStatus === 'active' ? 'activar' : 'desactivar';
            
            if(!confirm(`¿Está seguro de ${statusText} este operador?`)) {
                return;
            }
            
            const formData = new FormData();
            formData.append('action', 'toggle_status');
            formData.append('operator_id', operatorId);
            formData.append('status', newStatus);
            
            fetch('manage-operators.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if(data.success) {
                    showToast('success', 'Éxito', data.message);
                    loadOperators();
                } else {
                    showToast('warning', 'Error', data.error);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showToast('warning', 'Error', 'Error al cambiar el estado');
            });
        }

        function resetOperatorPassword(operatorId) {
            // Mostrar modal con opciones
            const newPassword = prompt('🔑 CAMBIAR CONTRASEÑA DE OPERADOR\n\nOpciones:\n1. Dejar vacío para generar contraseña automática\n2. Escribir una contraseña específica (mínimo 4 caracteres)\n\nIngrese la nueva contraseña o deje vacío:');
            
            // Si presiona cancelar
            if(newPassword === null) {
                return;
            }
            
            // Validar si es contraseña personalizada
            if(newPassword.trim() !== '' && newPassword.trim().length < 4) {
                alert('❌ La contraseña debe tener al menos 4 caracteres');
                return;
            }
            
            const formData = new FormData();
            formData.append('action', 'reset_password');
            formData.append('operator_id', operatorId);
            if(newPassword.trim() !== '') {
                formData.append('new_password', newPassword.trim());
            }
            
            fetch('manage-operators.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if(data.success) {
                    const passwordType = data.is_custom ? '✅ Contraseña Personalizada Establecida' : '🔄 Contraseña Temporal Generada';
                    const message = `${passwordType}\n\nNueva contraseña: ${data.temp_password}\nEmail del operador: ${data.operator_email}\n\n⚠️ Guarde esta información y envíela al operador.`;
                    
                    // Copiar al portapapeles
                    navigator.clipboard.writeText(data.temp_password).then(() => {
                        alert(message + '\n\n📋 Contraseña copiada al portapapeles');
                    }).catch(() => {
                        alert(message);
                    });
                    
                    showToast('success', 'Contraseña Actualizada', `Nueva contraseña: <strong>${data.temp_password}</strong>`);
                } else {
                    showToast('warning', 'Error', data.error);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showToast('warning', 'Error', 'Error al cambiar la contraseña');
            });
        }

        function deleteOperator(operatorId) {
            if(!confirm('⚠️ ¿Está SEGURO de eliminar este operador?\n\nEsta acción NO se puede deshacer.')) {
                return;
            }
            
            const formData = new FormData();
            formData.append('action', 'delete');
            formData.append('operator_id', operatorId);
            
            fetch('manage-operators.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if(data.success) {
                    showToast('success', 'Éxito', data.message);
                    loadOperators();
                } else {
                    showToast('warning', 'Error', data.error);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showToast('warning', 'Error', 'Error al eliminar el operador');
            });
        }

        // Additional CSS for buttons and modal
        const additionalStyles = document.createElement('style');
        additionalStyles.textContent = `
            .btn-sm {
                padding: 6px 10px;
                font-size: 12px;
            }
            .btn-success {
                background: var(--success);
                color: white;
            }
            .btn-success:hover {
                background: #059669;
            }
            .btn-warning {
                background: var(--warning);
                color: white;
            }
            .btn-warning:hover {
                background: #d97706;
            }
            .btn-info {
                background: var(--info);
                color: white;
            }
            .btn-info:hover {
                background: #2563eb;
            }
            .btn-danger {
                background: var(--danger);
                color: white;
            }
            .btn-danger:hover {
                background: #dc2626;
            }
        `;
        document.head.appendChild(additionalStyles);
        
        // ============================================
        // PWA: Registro del Service Worker
        // ============================================
        if ('serviceWorker' in navigator) {
            navigator.serviceWorker.register('service-worker.js')
                .then((registration) => {
                    console.log('[PWA] Service Worker registrado:', registration.scope);
                })
                .catch((error) => {
                    console.error('[PWA] Error al registrar Service Worker:', error);
                });
        }
    </script>
    
        <!-- Password Change Modal -->
        <div id="passwordModal" style="display:none; position:fixed; inset:0; background:rgba(0,0,0,0.5); z-index:9999; align-items:center; justify-content:center;">
            <div style="background:#fff; color:#111; width:90%; max-width:420px; border-radius:12px; box-shadow:0 20px 60px rgba(0,0,0,0.3); overflow:hidden;">
                <div style="padding:18px 20px; background:var(--primary); color:#fff; display:flex; align-items:center; justify-content:space-between;">
                    <div style="font-weight:700; display:flex; align-items:center; gap:10px;"><i class="fas fa-key"></i> Cambiar/Resetear Contraseña</div>
                    <button onclick="closePasswordModal()" style="background:transparent; border:0; color:#fff; font-size:18px; cursor:pointer;">×</button>
                </div>
                <div style="padding:20px;">
                    <p style="margin:0 0 10px; color:#444;">Opciones:</p>
                    <ol style="margin:0 0 15px 18px; color:#555;">
                        <li>Dejar el campo vacío para generar una contraseña automática</li>
                        <li>Escribir una contraseña personalizada (mínimo 4 caracteres)</li>
                    </ol>
                    <label for="pm_new_password" style="display:block; font-weight:600; margin-bottom:6px; color:#333;">Nueva contraseña (opcional)</label>
                    <div style="display:flex; gap:8px;">
                        <input id="pm_new_password" type="text" placeholder="Ej: MiPass2025" style="flex:1; padding:10px 12px; border:2px solid #e5e7eb; border-radius:8px; font-size:14px; outline:none;" />
                        <button type="button" onclick="generateAutoPassword()" class="btn btn-warning" style="padding:10px 12px; border-radius:8px; border:0;">Auto</button>
                    </div>
                    <small style="display:block; margin-top:8px; color:#6b7280;">Si lo dejas vacío, se generará automáticamente.</small>
                    <div id="pm_error" style="display:none; margin-top:10px; background:#ffebee; color:#c62828; padding:10px; border-radius:8px; font-size:13px;"></div>
                </div>
                <div style="padding:14px 20px; background:#f9fafb; display:flex; justify-content:flex-end; gap:10px;">
                    <button type="button" onclick="closePasswordModal()" class="btn" style="background:#e5e7eb; color:#111;">Cancelar</button>
                    <button type="button" onclick="submitPasswordChange()" class="btn btn-info">Guardar</button>
                </div>
            </div>
        </div>
    
        <script>
            // State for modal
            let pmTarget = { type: null, id: null };
            function openPasswordModal(type, id){
                pmTarget.type = type; pmTarget.id = id;
                document.getElementById('pm_new_password').value = '';
                document.getElementById('pm_error').style.display = 'none';
                const modal = document.getElementById('passwordModal');
                modal.style.display = 'flex';
            }
            function closePasswordModal(){
                document.getElementById('passwordModal').style.display = 'none';
            }
            function generateAutoPassword(){
                const prefix = pmTarget.type === 'operator' ? 'Oper' : 'Temp';
                const pw = prefix + Math.floor(Math.random()*9000 + 1000) + '!';
                document.getElementById('pm_new_password').value = pw;
            }
            function submitPasswordChange(){
                const input = document.getElementById('pm_new_password');
                const errorBox = document.getElementById('pm_error');
                const newPw = (input.value || '').trim();
                if(newPw !== '' && newPw.length < 4){
                    errorBox.textContent = 'La contraseña debe tener al menos 4 caracteres';
                    errorBox.style.display = 'block';
                    return;
                }
                const formData = new FormData();
                formData.append('action','reset_password');
                if(pmTarget.type === 'user'){
                    formData.append('user_id', pmTarget.id);
                } else {
                    formData.append('operator_id', pmTarget.id);
                }
                if(newPw !== '') formData.append('new_password', newPw);
                const url = pmTarget.type === 'user' ? 'manage-users.php' : 'manage-operators.php';
                
                fetch(url, { method:'POST', body: formData, credentials: 'include' })
                  .then(r => {
                      console.log('Response status:', r.status);
                      return r.json().then(data => ({status: r.status, data}));
                  })
                  .then(({status, data}) => {
                      console.log('Response data:', data);
                      if(status === 200 && data && data.success){
                          const msg = `Nueva contraseña: <strong>${data.temp_password}</strong>`;
                          showToast('success','Contraseña Actualizada', msg);
                          // Copy to clipboard
                          if(data.temp_password){
                              navigator.clipboard.writeText(String(data.temp_password)).catch(()=>{});
                          }
                          closePasswordModal();
                      } else {
                          const errMsg = (data && data.error) ? data.error : 'Error desconocido';
                          errorBox.textContent = '❌ ' + errMsg;
                          errorBox.style.display = 'block';
                          console.error('Error response:', data);
                      }
                  })
                  .catch(err => {
                      console.error('Password change error', err);
                      errorBox.textContent = '❌ Error de red: ' + err.message;
                      errorBox.style.display = 'block';
                  });
            }
        </script>
</body>
</html>
