<?php
require_once 'config.php';

if (!isset($_SESSION['operator_id'])) {
    header('Location: index.php');
    exit;
}

// Obtener información del operador
try {
    $stmt = $conn->prepare("SELECT * FROM operators WHERE id = ?");
    $stmt->execute([$_SESSION['operator_id']]);
    $operator = $stmt->fetch(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    die("Error al obtener información del operador");
}

// Configuración de idioma
$lang = $_SESSION['lang'] ?? 'es';

$t = [
    'es' => [
        'alerts_management' => 'Gestión de Alertas',
        'all_alerts' => 'Todas las Alertas',
        'my_alerts' => 'Mis Alertas',
        'filter' => 'Filtrar',
        'all' => 'Todas',
        'pending' => 'Pendientes',
        'in_progress' => 'En Progreso',
        'resolved' => 'Resueltas',
        'search' => 'Buscar por ubicación, tipo o ciudadano...',
        'id' => 'ID',
        'citizen' => 'Ciudadano',
        'type' => 'Tipo',
        'location' => 'Ubicación',
        'priority' => 'Prioridad',
        'status' => 'Estado',
        'operator' => 'Operador',
        'date' => 'Fecha',
        'actions' => 'Acciones',
        'view' => 'Ver',
        'assign_to_me' => 'Asignar a mí',
        'high' => 'ALTA',
        'medium' => 'MEDIA',
        'low' => 'BAJA',
        'medical' => 'Médica',
        'fire' => 'Incendio',
        'security' => 'Seguridad',
        'not_assigned' => 'Sin asignar',
        'no_alerts' => 'No hay alertas',
        'total_alerts' => 'Total de Alertas',
        'active_now' => 'Activas Ahora',
        'assigned_to_me' => 'Asignadas a mí',
        'refresh' => 'Actualizar'
    ],
    'en' => [
        'alerts_management' => 'Alerts Management',
        'all_alerts' => 'All Alerts',
        'my_alerts' => 'My Alerts',
        'filter' => 'Filter',
        'all' => 'All',
        'pending' => 'Pending',
        'in_progress' => 'In Progress',
        'resolved' => 'Resolved',
        'search' => 'Search by location, type or citizen...',
        'id' => 'ID',
        'citizen' => 'Citizen',
        'type' => 'Type',
        'location' => 'Location',
        'priority' => 'Priority',
        'status' => 'Status',
        'operator' => 'Operator',
        'date' => 'Date',
        'actions' => 'Actions',
        'view' => 'View',
        'assign_to_me' => 'Assign to me',
        'high' => 'HIGH',
        'medium' => 'MEDIUM',
        'low' => 'LOW',
        'medical' => 'Medical',
        'fire' => 'Fire',
        'security' => 'Security',
        'not_assigned' => 'Not assigned',
        'no_alerts' => 'No alerts',
        'total_alerts' => 'Total Alerts',
        'active_now' => 'Active Now',
        'assigned_to_me' => 'Assigned to me',
        'refresh' => 'Refresh'
    ]
][$lang];

// Obtener filtros
$filter = $_GET['filter'] ?? 'all';
$assigned = $_GET['assigned'] ?? '';
$search = $_GET['search'] ?? '';

$where = ["1=1"];
$params = [];

if ($filter != 'all') {
    $where[] = "a.status = ?";
    $params[] = $filter;
}

if ($assigned == 'me') {
    $where[] = "a.assigned_operator_id = ?";
    $params[] = $_SESSION['operator_id'];
}

if (!empty($search)) {
    $where[] = "(a.location LIKE ? OR a.alert_type LIKE ? OR u.full_name LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

// Obtener alertas
try {
    $sql = "SELECT a.*, u.full_name as user_name, u.email as user_email, o.full_name as operator_name 
            FROM alerts a 
            LEFT JOIN users u ON a.user_id = u.id 
            LEFT JOIN operators o ON a.assigned_operator_id = o.id 
            WHERE " . implode(" AND ", $where) . "
            ORDER BY 
                CASE a.status 
                    WHEN 'pending' THEN 1 
                    WHEN 'in_progress' THEN 2 
                    WHEN 'resolved' THEN 3 
                END,
                CASE a.priority 
                    WHEN 'high' THEN 1 
                    WHEN 'medium' THEN 2 
                    WHEN 'low' THEN 3 
                END,
                a.created_at DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    $alerts = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Estadísticas
    $stmt = $conn->query("SELECT COUNT(*) as total FROM alerts");
    $stats['total'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
    
    $stmt = $conn->query("SELECT COUNT(*) as total FROM alerts WHERE status IN ('pending', 'in_progress')");
    $stats['active'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
    
    $stmt = $conn->prepare("SELECT COUNT(*) as total FROM alerts WHERE assigned_operator_id = ? AND status IN ('pending', 'in_progress')");
    $stmt->execute([$_SESSION['operator_id']]);
    $stats['my_assignments'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
    
} catch(PDOException $e) {
    $alerts = [];
    $stats = ['total' => 0, 'active' => 0, 'my_assignments' => 0];
}
?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $t['alerts_management']; ?> - Azuero Safety Services S.A</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary: #667eea;
            --primary-dark: #5568d3;
            --secondary: #764ba2;
            --success: #10b981;
            --warning: #f59e0b;
            --danger: #ef4444;
            --info: #3b82f6;
            --bg-primary: #ffffff;
            --bg-secondary: #f8fafc;
            --bg-card: #ffffff;
            --text-primary: #1e293b;
            --text-secondary: #64748b;
            --border-color: #e2e8f0;
            --shadow-sm: 0 1px 3px rgba(0,0,0,0.1);
            --shadow-md: 0 4px 6px rgba(0,0,0,0.1);
            --shadow-lg: 0 10px 25px rgba(0,0,0,0.15);
            --sidebar-width: 280px;
        }

        [data-theme="dark"] {
            --primary: #818cf8;
            --primary-dark: #6366f1;
            --bg-primary: #0f172a;
            --bg-secondary: #1e293b;
            --bg-card: #1e293b;
            --text-primary: #f1f5f9;
            --text-secondary: #94a3b8;
            --border-color: #334155;
            --shadow-sm: 0 1px 3px rgba(0,0,0,0.3);
            --shadow-md: 0 4px 6px rgba(0,0,0,0.3);
            --shadow-lg: 0 10px 25px rgba(0,0,0,0.5);
        }

        [data-theme="blue-dark"] {
            --primary: #60a5fa;
            --primary-dark: #3b82f6;
            --secondary: #06b6d4;
            --bg-primary: #0c1e35;
            --bg-secondary: #1a2942;
            --bg-card: #1a2942;
            --text-primary: #e0f2fe;
            --text-secondary: #7dd3fc;
            --border-color: #1e3a5f;
            --shadow-sm: 0 1px 3px rgba(14,165,233,0.2);
            --shadow-md: 0 4px 6px rgba(14,165,233,0.2);
            --shadow-lg: 0 10px 25px rgba(14,165,233,0.3);
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: var(--bg-primary);
            color: var(--text-primary);
            transition: all 0.3s ease;
        }

        /* Sidebar */
        .sidebar {
            position: fixed;
            left: 0;
            top: 0;
            bottom: 0;
            width: var(--sidebar-width);
            background: var(--bg-card);
            border-right: 1px solid var(--border-color);
            box-shadow: var(--shadow-md);
            z-index: 1000;
            overflow-y: auto;
        }

        .sidebar-header {
            padding: 30px 20px;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            text-align: center;
        }

        .sidebar-header h2 {
            font-size: 24px;
            margin-bottom: 5px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }

        .sidebar-menu {
            padding: 20px 0;
        }

        .menu-item {
            display: flex;
            align-items: center;
            padding: 15px 25px;
            color: var(--text-primary);
            text-decoration: none;
            transition: all 0.3s ease;
            border-left: 3px solid transparent;
        }

        .menu-item:hover, .menu-item.active {
            background: var(--bg-secondary);
            border-left-color: var(--primary);
            color: var(--primary);
        }

        .menu-item i {
            width: 24px;
            font-size: 20px;
            margin-right: 15px;
        }

        .menu-section {
            padding: 10px 25px;
            font-size: 12px;
            font-weight: 600;
            color: var(--text-secondary);
            text-transform: uppercase;
            letter-spacing: 1px;
            margin-top: 20px;
        }

        /* Main Content */
        .main-content {
            margin-left: var(--sidebar-width);
            min-height: 100vh;
        }

        /* Header */
        .header {
            background: var(--bg-card);
            border-bottom: 1px solid var(--border-color);
            padding: 20px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: var(--shadow-sm);
            position: sticky;
            top: 0;
            z-index: 999;
        }

        .header-left h1 {
            font-size: 28px;
            color: var(--text-primary);
            margin-bottom: 5px;
        }

        .breadcrumb {
            color: var(--text-secondary);
            font-size: 14px;
        }

        .header-right {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .theme-switcher {
            display: flex;
            gap: 10px;
            background: var(--bg-secondary);
            padding: 5px;
            border-radius: 25px;
        }

        .theme-btn {
            padding: 8px 15px;
            border: none;
            background: transparent;
            color: var(--text-secondary);
            cursor: pointer;
            border-radius: 20px;
            font-size: 13px;
            transition: all 0.3s ease;
        }

        .theme-btn.active {
            background: var(--primary);
            color: white;
        }

        .user-profile {
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
            font-size: 16px;
        }

        /* Content */
        .content {
            padding: 30px;
        }

        /* Stats */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 15px;
            padding: 25px;
            box-shadow: var(--shadow-sm);
            transition: all 0.3s ease;
        }

        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, var(--primary), var(--secondary));
        }

        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: var(--shadow-lg);
        }

        .stat-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
        }

        .stat-icon {
            width: 50px;
            height: 50px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: white;
        }

        .stat-icon.blue { background: linear-gradient(135deg, #667eea, #764ba2); }
        .stat-icon.orange { background: linear-gradient(135deg, #f59e0b, #d97706); }
        .stat-icon.purple { background: linear-gradient(135deg, #8b5cf6, #7c3aed); }

        .stat-value {
            font-size: 32px;
            font-weight: bold;
            color: var(--text-primary);
            margin-bottom: 5px;
        }

        .stat-label {
            font-size: 14px;
            color: var(--text-secondary);
        }

        /* Filters */
        .filters-card {
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 15px;
            padding: 25px;
            box-shadow: var(--shadow-sm);
            margin-bottom: 30px;
        }

        .filters-row {
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
            align-items: center;
        }

        .filter-group {
            display: flex;
            gap: 10px;
        }

        .filter-btn {
            padding: 10px 20px;
            border: 2px solid var(--border-color);
            background: var(--bg-secondary);
            color: var(--text-primary);
            border-radius: 8px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 500;
            transition: all 0.3s ease;
            text-decoration: none;
        }

        .filter-btn:hover {
            border-color: var(--primary);
            color: var(--primary);
        }

        .filter-btn.active {
            background: var(--primary);
            border-color: var(--primary);
            color: white;
        }

        .search-box {
            flex: 1;
            min-width: 300px;
        }

        .search-box input {
            width: 100%;
            padding: 12px 20px;
            border: 1px solid var(--border-color);
            border-radius: 8px;
            background: var(--bg-secondary);
            color: var(--text-primary);
            font-size: 14px;
        }

        .search-box input:focus {
            outline: none;
            border-color: var(--primary);
        }

        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 500;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
            box-shadow: var(--shadow-md);
        }

        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
        }

        /* Table */
        .table-card {
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 15px;
            padding: 25px;
            box-shadow: var(--shadow-sm);
        }

        .table-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        .table-header h3 {
            font-size: 18px;
            color: var(--text-primary);
        }

        .table-wrapper {
            overflow-x: auto;
        }

        table {
            width: 100%;
            border-collapse: collapse;
        }

        thead {
            background: var(--bg-secondary);
        }

        th, td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid var(--border-color);
        }

        th {
            font-weight: 600;
            color: var(--text-secondary);
            font-size: 13px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        tbody tr {
            transition: background 0.3s ease;
        }

        tbody tr:hover {
            background: var(--bg-secondary);
        }

        .badge {
            display: inline-block;
            padding: 5px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 500;
        }

        .badge-success { background: rgba(16, 185, 129, 0.1); color: var(--success); }
        .badge-warning { background: rgba(245, 158, 11, 0.1); color: var(--warning); }
        .badge-danger { background: rgba(239, 68, 68, 0.1); color: var(--danger); }
        .badge-info { background: rgba(59, 130, 246, 0.1); color: var(--info); }

        @media (max-width: 1024px) {
            .sidebar {
                transform: translateX(-100%);
            }
            .main-content {
                margin-left: 0;
            }
        }
    </style>
</head>
<body data-theme="light">
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <h2>
                <i class="fas fa-shield-alt"></i>
                Azuero Safety Services S.A
            </h2>
            <p style="font-size: 11px; opacity: 0.85; margin-top: 5px;">Somos tu seguridad</p>
        </div>

        <div class="sidebar-menu">
            <a href="dashboard.php" class="menu-item">
                <i class="fas fa-home"></i>
                <span>Dashboard</span>
            </a>

            <div class="menu-section">Alertas</div>
            
            <a href="alertas.php" class="menu-item active">
                <i class="fas fa-bell"></i>
                <span><?php echo $t['all_alerts']; ?></span>
            </a>

            <a href="alertas.php?assigned=me" class="menu-item">
                <i class="fas fa-tasks"></i>
                <span><?php echo $t['my_alerts']; ?></span>
            </a>

            <?php if ($operator['role'] === 'admin'): ?>
            <div class="menu-section">Administración</div>
            
            <a href="admin-panel.php" class="menu-item">
                <i class="fas fa-user-shield"></i>
                <span>Panel Admin</span>
            </a>
            <?php endif; ?>

            <div class="menu-section">Sistema</div>

            <a href="logout.php" class="menu-item">
                <i class="fas fa-sign-out-alt"></i>
                <span>Cerrar Sesión</span>
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Header -->
        <div class="header">
            <div class="header-left">
                <h1><?php echo $t['alerts_management']; ?></h1>
                <div class="breadcrumb">
                    <i class="fas fa-home"></i> / <?php echo $t['alerts_management']; ?>
                </div>
            </div>

            <div class="header-right">
                <div class="theme-switcher">
                    <button class="theme-btn active" data-theme="light" onclick="changeTheme('light')">☀️</button>
                    <button class="theme-btn" data-theme="dark" onclick="changeTheme('dark')">🌙</button>
                    <button class="theme-btn" data-theme="blue-dark" onclick="changeTheme('blue-dark')">🌊</button>
                </div>

                <div class="user-profile">
                    <div class="user-avatar">
                        <?php echo strtoupper(substr($operator['full_name'], 0, 1)); ?>
                    </div>
                </div>
            </div>
        </div>

        <!-- Content -->
        <div class="content">
            <!-- Stats -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-header">
                        <div>
                            <div class="stat-value"><?php echo $stats['total']; ?></div>
                            <div class="stat-label"><?php echo $t['total_alerts']; ?></div>
                        </div>
                        <div class="stat-icon blue">
                            <i class="fas fa-bell"></i>
                        </div>
                    </div>
                </div>

                <div class="stat-card">
                    <div class="stat-header">
                        <div>
                            <div class="stat-value"><?php echo $stats['active']; ?></div>
                            <div class="stat-label"><?php echo $t['active_now']; ?></div>
                        </div>
                        <div class="stat-icon orange">
                            <i class="fas fa-exclamation-circle"></i>
                        </div>
                    </div>
                </div>

                <div class="stat-card">
                    <div class="stat-header">
                        <div>
                            <div class="stat-value"><?php echo $stats['my_assignments']; ?></div>
                            <div class="stat-label"><?php echo $t['assigned_to_me']; ?></div>
                        </div>
                        <div class="stat-icon purple">
                            <i class="fas fa-user-check"></i>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Filters -->
            <div class="filters-card">
                <div class="filters-row">
                    <div class="filter-group">
                        <a href="?filter=all" class="filter-btn <?php echo $filter == 'all' ? 'active' : ''; ?>">
                            <?php echo $t['all']; ?>
                        </a>
                        <a href="?filter=pending" class="filter-btn <?php echo $filter == 'pending' ? 'active' : ''; ?>">
                            <?php echo $t['pending']; ?>
                        </a>
                        <a href="?filter=in_progress" class="filter-btn <?php echo $filter == 'in_progress' ? 'active' : ''; ?>">
                            <?php echo $t['in_progress']; ?>
                        </a>
                        <a href="?filter=resolved" class="filter-btn <?php echo $filter == 'resolved' ? 'active' : ''; ?>">
                            <?php echo $t['resolved']; ?>
                        </a>
                    </div>

                    <div class="search-box">
                        <form method="GET" style="display: flex; gap: 10px;">
                            <input type="text" name="search" placeholder="<?php echo $t['search']; ?>" value="<?php echo htmlspecialchars($search); ?>">
                            <?php if($filter != 'all'): ?>
                            <input type="hidden" name="filter" value="<?php echo $filter; ?>">
                            <?php endif; ?>
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-search"></i>
                            </button>
                        </form>
                    </div>

                    <button onclick="location.reload()" class="btn btn-primary">
                        <i class="fas fa-sync"></i> <?php echo $t['refresh']; ?>
                    </button>
                </div>
            </div>

            <!-- Table -->
            <div class="table-card">
                <div class="table-header">
                    <h3><?php echo count($alerts); ?> Alertas</h3>
                </div>
                <div class="table-wrapper">
                    <table>
                        <thead>
                            <tr>
                                <th><?php echo $t['id']; ?></th>
                                <th><?php echo $t['citizen']; ?></th>
                                <th><?php echo $t['type']; ?></th>
                                <th><?php echo $t['priority']; ?></th>
                                <th><?php echo $t['status']; ?></th>
                                <th><?php echo $t['operator']; ?></th>
                                <th><?php echo $t['date']; ?></th>
                                <th><?php echo $t['actions']; ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($alerts)): ?>
                            <tr>
                                <td colspan="8" style="text-align: center; padding: 40px;">
                                    <i class="fas fa-inbox" style="font-size: 48px; color: var(--text-secondary); opacity: 0.5;"></i>
                                    <p style="margin-top: 15px; color: var(--text-secondary);"><?php echo $t['no_alerts']; ?></p>
                                </td>
                            </tr>
                            <?php else: ?>
                                <?php foreach($alerts as $alert): ?>
                                <tr>
                                    <td>#<?php echo $alert['id']; ?></td>
                                    <td>
                                        <div style="display: flex; align-items: center; gap: 10px;">
                                            <div class="user-avatar" style="width: 32px; height: 32px; font-size: 12px;">
                                                <?php echo strtoupper(substr($alert['user_name'] ?? 'U', 0, 1)); ?>
                                            </div>
                                            <div>
                                                <div style="font-weight: 500;"><?php echo htmlspecialchars($alert['user_name'] ?? 'Desconocido'); ?></div>
                                                <div style="font-size: 12px; color: var(--text-secondary);"><?php echo htmlspecialchars($alert['user_email'] ?? ''); ?></div>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <?php
                                        $type_badges = [
                                            'medical' => ['icon' => 'fa-heartbeat', 'color' => 'danger', 'text' => $t['medical']],
                                            'fire' => ['icon' => 'fa-fire', 'color' => 'warning', 'text' => $t['fire']],
                                            'security' => ['icon' => 'fa-shield-alt', 'color' => 'info', 'text' => $t['security']]
                                        ];
                                        $badge = $type_badges[$alert['alert_type']] ?? ['icon' => 'fa-bell', 'color' => 'info', 'text' => $alert['alert_type']];
                                        ?>
                                        <span class="badge badge-<?php echo $badge['color']; ?>">
                                            <i class="fas <?php echo $badge['icon']; ?>"></i> <?php echo $badge['text']; ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php
                                        $priority_colors = ['high' => 'danger', 'medium' => 'warning', 'low' => 'info'];
                                        $priority_text = ['high' => $t['high'], 'medium' => $t['medium'], 'low' => $t['low']];
                                        $color = $priority_colors[$alert['priority']] ?? 'info';
                                        $text = $priority_text[$alert['priority']] ?? strtoupper($alert['priority']);
                                        ?>
                                        <span class="badge badge-<?php echo $color; ?>"><?php echo $text; ?></span>
                                    </td>
                                    <td>
                                        <?php
                                        $status_colors = ['pending' => 'warning', 'in_progress' => 'info', 'resolved' => 'success'];
                                        $status_text = ['pending' => $t['pending'], 'in_progress' => $t['in_progress'], 'resolved' => $t['resolved']];
                                        $color = $status_colors[$alert['status']] ?? 'info';
                                        $text = $status_text[$alert['status']] ?? strtoupper($alert['status']);
                                        ?>
                                        <span class="badge badge-<?php echo $color; ?>"><?php echo $text; ?></span>
                                    </td>
                                    <td>
                                        <?php if($alert['operator_name']): ?>
                                            <span style="font-weight: 500;"><?php echo htmlspecialchars($alert['operator_name']); ?></span>
                                        <?php else: ?>
                                            <span style="color: var(--text-secondary);"><?php echo $t['not_assigned']; ?></span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo date('d/m/Y H:i', strtotime($alert['created_at'])); ?></td>
                                    <td>
                                        <a href="ver-alerta.php?id=<?php echo $alert['id']; ?>" class="btn btn-primary btn-sm">
                                            <i class="fas fa-eye"></i> <?php echo $t['view']; ?>
                                        </a>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <script>
        function changeTheme(theme) {
            document.body.setAttribute('data-theme', theme);
            localStorage.setItem('theme', theme);
            
            document.querySelectorAll('.theme-btn').forEach(btn => {
                btn.classList.remove('active');
                if(btn.getAttribute('data-theme') === theme) {
                    btn.classList.add('active');
                }
            });
        }

        const savedTheme = localStorage.getItem('theme') || 'light';
        changeTheme(savedTheme);
    </script>
</body>
</html>
