<?php
session_start();
require_once 'config.php';

// Check if user is citizen and logged in
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] != 'citizen') {
    header('Location: index.php');
    exit;
}

// Get user data
$stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
$stmt->execute([$_SESSION['user_id']]);
$user = $stmt->fetch();

// If profile is already complete, redirect (verificar con las columnas reales)
if (!empty($user['blood_type']) && !empty($user['emergency_contact'])) {
    header('Location: panel-ciudadano.php');
    exit;
}

$lang = $_SESSION['language'] ?? 'es';

// Translations
$translations = [
    'es' => [
        'title' => 'Completar Perfil',
        'welcome' => 'Bienvenido al Sistema de Emergencias',
        'intro' => 'Por favor, completa tu información médica y de contacto de emergencia para activar el acceso completo al sistema.',
        'personal_info' => 'Información Personal',
        'full_name' => 'Nombre Completo',
        'cedula' => 'Cédula',
        'email' => 'Correo Electrónico',
        'phone' => 'Teléfono',
        'medical_info' => 'Información Médica',
        'blood_type' => 'Tipo de Sangre',
        'select_blood' => 'Selecciona tu tipo de sangre',
        'allergies' => 'Alergias',
        'allergies_ph' => 'Ejemplo: Penicilina, Polen, Mariscos...',
        'conditions' => 'Condiciones Médicas',
        'conditions_ph' => 'Ejemplo: Diabetes, Hipertensión, Asma...',
        'medications' => 'Medicamentos Actuales',
        'medications_ph' => 'Ejemplo: Metformina 500mg, Losartán 50mg...',
        'emergency_contact' => 'Contacto de Emergencia',
        'contact_name' => 'Nombre del Contacto',
        'contact_name_ph' => 'Nombre completo',
        'contact_phone' => 'Teléfono del Contacto',
        'contact_phone_ph' => 'Número de teléfono',
        'contact_relation' => 'Relación',
        'contact_relation_ph' => 'Ejemplo: Esposo/a, Hijo/a, Padre/Madre...',
        'save' => 'Guardar y Continuar',
        'required_field' => 'Campo obligatorio',
        'optional' => 'Opcional',
        'logout' => 'Cerrar Sesión'
    ],
    'en' => [
        'title' => 'Complete Profile',
        'welcome' => 'Welcome to the Emergency System',
        'intro' => 'Please complete your medical and emergency contact information to activate full system access.',
        'personal_info' => 'Personal Information',
        'full_name' => 'Full Name',
        'cedula' => 'ID Number',
        'email' => 'Email',
        'phone' => 'Phone',
        'medical_info' => 'Medical Information',
        'blood_type' => 'Blood Type',
        'select_blood' => 'Select your blood type',
        'allergies' => 'Allergies',
        'allergies_ph' => 'Example: Penicillin, Pollen, Seafood...',
        'conditions' => 'Medical Conditions',
        'conditions_ph' => 'Example: Diabetes, Hypertension, Asthma...',
        'medications' => 'Current Medications',
        'medications_ph' => 'Example: Metformin 500mg, Losartan 50mg...',
        'emergency_contact' => 'Emergency Contact',
        'contact_name' => 'Contact Name',
        'contact_name_ph' => 'Full name',
        'contact_phone' => 'Contact Phone',
        'contact_phone_ph' => 'Phone number',
        'contact_relation' => 'Relationship',
        'contact_relation_ph' => 'Example: Spouse, Son/Daughter, Parent...',
        'save' => 'Save and Continue',
        'required_field' => 'Required field',
        'optional' => 'Optional',
        'logout' => 'Logout'
    ]
];

$t = $translations[$lang];

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $blood_type = $_POST['blood_type'] ?? '';
    $allergies = $_POST['allergies'] ?? '';
    $medical_conditions = $_POST['medical_conditions'] ?? '';
    $medications = $_POST['medications'] ?? '';
    $emergency_contact_name = $_POST['emergency_contact_name'] ?? '';
    $emergency_contact_phone = $_POST['emergency_contact_phone'] ?? '';
    $emergency_contact_relation = $_POST['emergency_contact_relation'] ?? '';
    
    // Validate required fields
    if (empty($blood_type) || empty($emergency_contact_name) || empty($emergency_contact_phone)) {
        $error = $lang == 'es' ? 'Por favor completa todos los campos obligatorios.' : 'Please complete all required fields.';
    } else {
        // Update user profile usando las columnas que existen en la BD
        try {
            $stmt = $conn->prepare("UPDATE users SET 
                blood_type = ?,
                allergies = ?,
                medical_conditions = ?,
                emergency_contact = ?,
                emergency_phone = ?
                WHERE id = ?");
            
            // Combinar nombre y relación en emergency_contact
            $emergency_full = $emergency_contact_name;
            if (!empty($emergency_contact_relation)) {
                $emergency_full .= ' (' . $emergency_contact_relation . ')';
            }
            
            if ($stmt->execute([
                $blood_type,
                $allergies,
                $medical_conditions,
                $emergency_full,
                $emergency_contact_phone,
                $_SESSION['user_id']
            ])) {
                header('Location: panel-ciudadano.php');
                exit;
            } else {
                $error = $lang == 'es' ? 'Error al guardar el perfil.' : 'Error saving profile.';
            }
        } catch(PDOException $e) {
            $error = $lang == 'es' ? 'Error: ' . $e->getMessage() : 'Error: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $t['title']; ?> - Azuero Safety</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            justify-content: center;
            align-items: center;
            padding: 20px;
        }
        
        .container {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            padding: 40px;
            max-width: 700px;
            width: 100%;
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .header {
            text-align: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 3px solid #667eea;
        }
        
        .header h1 {
            color: #667eea;
            font-size: 28px;
            margin-bottom: 10px;
        }
        
        .header p {
            color: #666;
            font-size: 14px;
            line-height: 1.5;
        }
        
        .section {
            margin-bottom: 30px;
        }
        
        .section-title {
            font-size: 18px;
            color: #333;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 2px solid #eee;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .section-title::before {
            content: '';
            display: inline-block;
            width: 4px;
            height: 20px;
            background: #667eea;
            border-radius: 2px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            color: #555;
            font-weight: 500;
            margin-bottom: 8px;
            font-size: 14px;
        }
        
        .form-group label .required {
            color: #e74c3c;
            margin-left: 3px;
        }
        
        .form-group label .optional {
            color: #999;
            font-size: 12px;
            font-weight: 400;
            margin-left: 5px;
        }
        
        .form-group input,
        .form-group select,
        .form-group textarea {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #ddd;
            border-radius: 10px;
            font-size: 14px;
            transition: all 0.3s;
            font-family: inherit;
        }
        
        .form-group input:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .form-group input[readonly] {
            background: #f5f5f5;
            color: #999;
            cursor: not-allowed;
        }
        
        .form-group textarea {
            resize: vertical;
            min-height: 80px;
        }
        
        .info-card {
            background: #f8f9fa;
            border-left: 4px solid #17a2b8;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        .info-card p {
            color: #555;
            font-size: 13px;
            line-height: 1.5;
            margin: 0;
        }
        
        .error {
            background: #fee;
            color: #c33;
            padding: 12px 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            border-left: 4px solid #e74c3c;
            font-size: 14px;
        }
        
        .btn-save {
            width: 100%;
            padding: 15px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
            margin-top: 10px;
        }
        
        .btn-save:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(102, 126, 234, 0.3);
        }
        
        .btn-logout {
            display: block;
            text-align: center;
            color: #999;
            text-decoration: none;
            margin-top: 20px;
            font-size: 14px;
            padding: 10px;
            transition: color 0.3s;
        }
        
        .btn-logout:hover {
            color: #667eea;
        }
        
        .grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
        }
        
        /* ========================================
           RESPONSIVE DESIGN PARA MÓVILES
           ======================================== */
        @media (max-width: 768px) {
            body {
                padding: 10px;
            }
            
            .container {
                padding: 25px 20px;
                max-width: 100%;
            }
            
            .header h1 {
                font-size: 22px;
            }
            
            .header p {
                font-size: 13px;
            }
            
            .section-title {
                font-size: 16px;
            }
            
            .form-group label {
                font-size: 13px;
            }
            
            .form-group input,
            .form-group select,
            .form-group textarea {
                padding: 10px 12px;
                font-size: 14px;
            }
            
            .grid {
                grid-template-columns: 1fr;
                gap: 15px;
            }
            
            .btn-save {
                padding: 12px;
                font-size: 15px;
            }
            
            .info-card {
                padding: 12px;
                font-size: 12px;
            }
        }
        
        @media (max-width: 480px) {
            .container {
                padding: 20px 15px;
            }
            
            .header {
                padding: 25px 20px;
            }
            
            .header h1 {
                font-size: 20px;
            }
            
            .section-title {
                font-size: 15px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🏥 <?php echo $t['welcome']; ?></h1>
            <p><?php echo $t['intro']; ?></p>
        </div>
        
        <?php if (isset($error)): ?>
            <div class="error">⚠️ <?php echo $error; ?></div>
        <?php endif; ?>
        
        <form method="POST" action="">
            <!-- Personal Information (Read-only) -->
            <div class="section">
                <div class="section-title">👤 <?php echo $t['personal_info']; ?></div>
                <div class="grid">
                    <div class="form-group">
                        <label><?php echo $t['full_name']; ?></label>
                        <input type="text" value="<?php echo htmlspecialchars($user['full_name']); ?>" readonly>
                    </div>
                    <div class="form-group">
                        <label><?php echo $t['cedula']; ?></label>
                        <input type="text" value="<?php echo htmlspecialchars($user['cedula']); ?>" readonly>
                    </div>
                </div>
                <div class="grid">
                    <div class="form-group">
                        <label><?php echo $t['email']; ?></label>
                        <input type="email" value="<?php echo htmlspecialchars($user['email']); ?>" readonly>
                    </div>
                    <div class="form-group">
                        <label><?php echo $t['phone']; ?></label>
                        <input type="tel" value="<?php echo htmlspecialchars($user['phone']); ?>" readonly>
                    </div>
                </div>
            </div>
            
            <!-- Medical Information -->
            <div class="section">
                <div class="section-title">🩺 <?php echo $t['medical_info']; ?></div>
                
                <div class="form-group">
                    <label>
                        <?php echo $t['blood_type']; ?>
                        <span class="required">*</span>
                    </label>
                    <select name="blood_type" required>
                        <option value=""><?php echo $t['select_blood']; ?></option>
                        <option value="A+">A+</option>
                        <option value="A-">A-</option>
                        <option value="B+">B+</option>
                        <option value="B-">B-</option>
                        <option value="AB+">AB+</option>
                        <option value="AB-">AB-</option>
                        <option value="O+">O+</option>
                        <option value="O-">O-</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label>
                        <?php echo $t['allergies']; ?>
                        <span class="optional">(<?php echo $t['optional']; ?>)</span>
                    </label>
                    <textarea name="allergies" placeholder="<?php echo $t['allergies_ph']; ?>"></textarea>
                </div>
                
                <div class="form-group">
                    <label>
                        <?php echo $t['conditions']; ?>
                        <span class="optional">(<?php echo $t['optional']; ?>)</span>
                    </label>
                    <textarea name="medical_conditions" placeholder="<?php echo $t['conditions_ph']; ?>"></textarea>
                </div>
                
                <div class="form-group">
                    <label>
                        <?php echo $t['medications']; ?>
                        <span class="optional">(<?php echo $t['optional']; ?>)</span>
                    </label>
                    <textarea name="medications" placeholder="<?php echo $t['medications_ph']; ?>"></textarea>
                </div>
            </div>
            
            <!-- Emergency Contact -->
            <div class="section">
                <div class="section-title">📞 <?php echo $t['emergency_contact']; ?></div>
                
                <div class="info-card">
                    <p>ℹ️ <?php echo $lang == 'es' 
                        ? 'Esta persona será contactada en caso de emergencia. Asegúrate de que la información sea correcta.' 
                        : 'This person will be contacted in case of emergency. Make sure the information is correct.'; ?></p>
                </div>
                
                <div class="form-group">
                    <label>
                        <?php echo $t['contact_name']; ?>
                        <span class="required">*</span>
                    </label>
                    <input type="text" name="emergency_contact_name" placeholder="<?php echo $t['contact_name_ph']; ?>" required>
                </div>
                
                <div class="grid">
                    <div class="form-group">
                        <label>
                            <?php echo $t['contact_phone']; ?>
                            <span class="required">*</span>
                        </label>
                        <input type="tel" name="emergency_contact_phone" placeholder="<?php echo $t['contact_phone_ph']; ?>" required>
                    </div>
                    <div class="form-group">
                        <label>
                            <?php echo $t['contact_relation']; ?>
                            <span class="optional">(<?php echo $t['optional']; ?>)</span>
                        </label>
                        <input type="text" name="emergency_contact_relation" placeholder="<?php echo $t['contact_relation_ph']; ?>">
                    </div>
                </div>
            </div>
            
            <button type="submit" class="btn-save">✅ <?php echo $t['save']; ?></button>
        </form>
        
        <a href="logout.php" class="btn-logout">🚪 <?php echo $t['logout']; ?></a>
    </div>
</body>
</html>
