<?php
require_once 'config.php';

// Multi-language support
$lang = $_SESSION['lang'] ?? 'es';

// Guardar idioma en sesión
if (isset($_GET['lang'])) {
    $_SESSION['lang'] = $_GET['lang'];
    header('Location: index.php');
    exit;
}

$translations = [
    'es' => [
        'title' => 'Azuero Safety',
        'subtitle' => 'Sistema de Respuesta a Emergencias',
        'username' => 'Usuario',
        'password' => 'Contraseña',
        'remember' => 'Recordarme',
        'login' => 'Iniciar Sesión',
        'secure' => 'Conexión segura y cifrada',
        'access_note' => 'Ingresa con las credenciales proporcionadas por el administrador',
        'error_credentials' => 'Usuario o contraseña incorrectos',
        'error_empty' => 'Por favor ingresa usuario y contraseña',
        'enter_username' => 'Ingresa tu usuario',
        'enter_password' => 'Ingresa tu contraseña',
        'copyright' => 'Todos los derechos reservados',
        'emergency_system' => 'Sistema de Respuesta a Emergencias'
    ],
    'en' => [
        'title' => 'Azuero Safety',
        'subtitle' => 'Emergency Response System',
        'username' => 'Username',
        'password' => 'Password',
        'remember' => 'Remember me',
        'login' => 'Sign In',
        'secure' => 'Secure and encrypted connection',
        'access_note' => 'Enter with credentials provided by administrator',
        'error_credentials' => 'Incorrect username or password',
        'error_empty' => 'Please enter username and password',
        'enter_username' => 'Enter your username',
        'enter_password' => 'Enter your password',
        'copyright' => 'All rights reserved',
        'emergency_system' => 'Emergency Response System'
    ]
];

$t = $translations[$lang];

$error = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $username = trim($_POST['username'] ?? '');
    $password = trim($_POST['password'] ?? '');
    
    if (empty($username) || empty($password)) {
        $error = $t['error_empty'];
    } else {
        try {
            // Primero buscar en operadores (buscar por username o email)
            $stmt = $conn->prepare("
                SELECT 
                    id as operator_id,
                    username,
                    password,
                    full_name as name,
                    email,
                    role,
                    status,
                    'operator' as user_type 
                FROM operators 
                WHERE (username = ? OR email = ?) 
                AND status = 'active'
            ");
            $stmt->execute([$username, $username]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Si no es operador, buscar en usuarios (ciudadanos)
            if (!$user) {
                $stmt = $conn->prepare("
                    SELECT 
                        id,
                        username,
                        full_name,
                        email,
                        phone,
                        password,
                        status,
                        blood_type,
                        emergency_contact,
                        'citizen' as user_type 
                    FROM users 
                    WHERE (username = ? OR email = ? OR phone = ?) 
                    AND status = 'active'
                ");
                $stmt->execute([$username, $username, $username]);
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
            }
            
            if ($user && password_verify($password, $user['password'])) {
                $_SESSION['username'] = $username;
                $_SESSION['user_type'] = $user['user_type'];
                $_SESSION['lang'] = $lang;
                
                if ($user['user_type'] == 'operator') {
                    $_SESSION['operator_id'] = $user['operator_id'];
                    $_SESSION['user_id'] = $user['operator_id'];
                    $_SESSION['full_name'] = $user['name'];
                    $_SESSION['role'] = $user['role'];
                    $_SESSION['email'] = $user['email'];
                    
                    // Actualizar último login (solo si la columna existe)
                    try {
                        $updateStmt = $conn->prepare("UPDATE operators SET last_login = NOW() WHERE id = ?");
                        $updateStmt->execute([$user['operator_id']]);
                    } catch (PDOException $e) {
                        // Ignorar error si la columna no existe
                    }
                    
                    // Redirigir según el rol
                    if ($user['role'] === 'admin') {
                        header('Location: admin-panel.php');
                        exit();
                    } else {
                        header('Location: dashboard.php');
                        exit();
                    }
                } else {
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['full_name'] = $user['full_name'];
                    
                    // Ciudadano - verificar si completó su perfil
                    if (empty($user['blood_type']) || empty($user['emergency_contact'])) {
                        header('Location: completar-perfil.php');
                        exit();
                    } else {
                        header('Location: panel-ciudadano.php');
                        exit();
                    }
                }
            } else {
                $error = $t['error_credentials'];
            }
        } catch(PDOException $e) {
            $error = 'Error: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $t['title']; ?> - Login</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .login-container {
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.3);
            padding: 40px;
            width: 100%;
            max-width: 400px;
            position: relative;
        }
        
        .lang-switcher {
            position: absolute;
            top: 15px;
            right: 15px;
            display: flex;
            gap: 5px;
        }
        
        .lang-btn {
            padding: 5px 10px;
            border: 2px solid #ddd;
            background: white;
            border-radius: 5px;
            cursor: pointer;
            font-size: 12px;
            font-weight: 600;
            transition: all 0.3s;
            text-decoration: none;
            color: #333;
        }
        
        .lang-btn.active {
            background: #1e3c72;
            color: white;
            border-color: #1e3c72;
        }
        
        .logo { text-align: center; margin-bottom: 30px; }
        .logo-icon { font-size: 60px; margin-bottom: 10px; }
        h1 { color: #1e3c72; text-align: center; margin-bottom: 10px; font-size: 28px; }
        .subtitle { color: #666; text-align: center; margin-bottom: 30px; font-size: 14px; }
        .form-group { margin-bottom: 20px; }
        
        label {
            display: block;
            color: #333;
            font-weight: 600;
            margin-bottom: 8px;
            font-size: 14px;
        }
        
        input[type="text"], input[type="password"] {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #ddd;
            border-radius: 8px;
            font-size: 15px;
            transition: border-color 0.3s;
        }
        
        input[type="text"]:focus, input[type="password"]:focus {
            outline: none;
            border-color: #2a5298;
        }
        
        .btn-login {
            width: 100%;
            padding: 14px;
            background: #1e3c72;
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: background 0.3s;
        }
        
        .btn-login:hover { background: #2a5298; }
        
        .error-message {
            background: #ffebee;
            color: #c62828;
            padding: 12px;
            border-radius: 8px;
            margin-bottom: 20px;
            font-size: 14px;
            text-align: center;
        }
        
        .footer { text-align: center; margin-top: 20px; color: #666; font-size: 12px; }
        
        .register-link {
            background: #f5f5f5;
            padding: 15px;
            border-radius: 8px;
            text-align: center;
            margin-top: 20px;
            border: 2px dashed #ddd;
        }
        
        .register-link p { color: #666; font-size: 14px; margin-bottom: 10px; }
        
        .register-link a {
            display: inline-block;
            background: #2e7d32;
            color: white;
            padding: 10px 20px;
            border-radius: 5px;
            text-decoration: none;
            font-weight: 600;
            transition: background 0.3s;
        }
        
        .register-link a:hover { background: #388e3c; }
    </style>
</head>
<body>
    <div class="login-container">
        <div class="lang-switcher">
            <a href="?lang=es" class="lang-btn <?php echo $lang == 'es' ? 'active' : ''; ?>">🇪🇸 ES</a>
            <a href="?lang=en" class="lang-btn <?php echo $lang == 'en' ? 'active' : ''; ?>">🇺🇸 EN</a>
        </div>
        
        <div class="logo">
            <div class="logo-icon">🚨</div>
            <h1><?php echo $t['title']; ?></h1>
            <p class="subtitle"><?php echo $t['subtitle']; ?></p>
        </div>
        
        <?php if ($error): ?>
        <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <form method="POST" action="">
            <div class="form-group">
                <label for="username">👤 <?php echo $t['username']; ?></label>
                <input type="text" id="username" name="username" placeholder="<?php echo $t['enter_username']; ?>" required autofocus>
            </div>
            
            <div class="form-group">
                <label for="password">🔒 <?php echo $t['password']; ?></label>
                <input type="password" id="password" name="password" placeholder="<?php echo $t['enter_password']; ?>" required>
            </div>
            
            <div class="form-group">
                <label style="display: flex; align-items: center; font-weight: normal; cursor: pointer;">
                    <input type="checkbox" name="remember" style="margin-right: 8px;">
                    <span style="font-size: 14px;"><?php echo $t['remember']; ?></span>
                </label>
            </div>
            
            <button type="submit" class="btn-login">🔐 <?php echo $t['login']; ?></button>
        </form>
        
        <div class="footer">
            <p style="margin-bottom: 10px;">ℹ️ <?php echo $t['access_note']; ?></p>
            <p style="margin-top: 5px;">🔒 <?php echo $t['secure']; ?></p>
            <p style="margin-top: 15px; color: #999;">&copy; <?php echo date('Y'); ?> <?php echo $t['title']; ?>. <?php echo $t['copyright']; ?>.</p>
            <p style="margin-top: 5px; font-size: 11px; color: #999;"><?php echo $t['emergency_system']; ?></p>
        </div>
    </div>
</body>
</html>
