<?php
require_once 'config.php';

// Headers CORS y JSON
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Manejar preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Verificar que sea operador
if (!isset($_SESSION['operator_id'])) {
    http_response_code(403);
    echo json_encode(['success' => false, 'error' => 'No autorizado - No hay sesión activa']);
    exit();
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

try {
    switch($action) {
        case 'list':
            listUsers();
            break;
        case 'create':
            createUser();
            break;
        case 'toggle_status':
            toggleUserStatus();
            break;
        case 'reset_password':
            resetUserPassword();
            break;
        case 'get_user_details':
            getUserDetails();
            break;
        case 'get_user_history':
            getUserHistory();
            break;
        case 'delete_user':
            deleteUser();
            break;
        default:
            throw new Exception('Acción no válida');
    }
} catch(Exception $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}

function createUser() {
    global $conn;
    
    $name = trim($_POST['name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $cedula = trim($_POST['cedula'] ?? '');
    $password = $_POST['password'] ?? '';
    
    if (empty($name) || empty($email)) {
        throw new Exception('Nombre y email son obligatorios');
    }
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        throw new Exception('Email inválido');
    }
    
    // Verificar email único
    $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$email]);
    if ($stmt->fetch()) {
        throw new Exception('El email ya está registrado');
    }
    
    // Verificar username único si se proporciona
    $username = !empty($_POST['username']) ? trim($_POST['username']) : null;
    if ($username) {
        $stmt = $conn->prepare("SELECT id FROM users WHERE username = ?");
        $stmt->execute([$username]);
        if ($stmt->fetch()) {
            throw new Exception('El username ya está registrado');
        }
    }
    
    // Generar contraseña si no viene
    if (empty($password)) {
        $password = 'User' . rand(1000, 9999) . '!';
        $generated = true;
    } else {
        $generated = false;
    }
    
    $hashed = password_hash($password, PASSWORD_DEFAULT);
    
    $stmt = $conn->prepare("INSERT INTO users (username, full_name, email, phone, cedula, password, status, created_at) VALUES (?, ?, ?, ?, ?, ?, 'active', NOW())");
    $stmt->execute([$username, $name, $email, $phone, $cedula, $hashed]);
    $newId = $conn->lastInsertId();
    
    // Log
    $operatorId = $_SESSION['operator_id'];
    error_log("Operator {$operatorId} creó usuario {$newId} ({$email})");
    
    echo json_encode([
        'success' => true,
        'message' => 'Usuario creado correctamente',
        'user_id' => $newId,
        'temp_password' => $generated ? $password : null
    ]);
}

function listUsers() {
    global $conn;
    
    $search = $_GET['search'] ?? '';
    $status = $_GET['status'] ?? '';
    
    // Query base
    $sql = "
        SELECT 
            u.id as user_id,
            u.full_name as name,
            u.email,
            u.phone,
            u.status,
            u.created_at,
            COUNT(DISTINCT a.id) as alert_count,
            MAX(a.created_at) as last_alert
        FROM users u
        LEFT JOIN alerts a ON u.id = a.user_id
        WHERE 1=1
    ";
    
    $params = [];
    
    // Filtro de búsqueda
    if (!empty($search)) {
        $sql .= " AND (u.full_name LIKE ? OR u.email LIKE ? OR u.phone LIKE ?)";
        $searchParam = "%{$search}%";
        $params[] = $searchParam;
        $params[] = $searchParam;
        $params[] = $searchParam;
    }
    
    // Filtro de estado
    if (!empty($status)) {
        $sql .= " AND u.status = ?";
        $params[] = $status;
    }
    
    $sql .= " GROUP BY u.id ORDER BY u.created_at DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Estadísticas
    $statsQuery = "
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active,
            SUM(CASE WHEN status = 'inactive' THEN 1 ELSE 0 END) as inactive,
            SUM(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) THEN 1 ELSE 0 END) as new_users
        FROM users
    ";
    $stmt = $conn->query($statsQuery);
    $stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'users' => $users,
        'stats' => $stats
    ]);
}

function toggleUserStatus() {
    global $conn;
    
    $userId = $_POST['user_id'] ?? 0;
    $newStatus = $_POST['status'] ?? '';
    
    if (!$userId || !in_array($newStatus, ['active', 'inactive'])) {
        throw new Exception('Parámetros inválidos');
    }
    
    $stmt = $conn->prepare("UPDATE users SET status = ? WHERE id = ?");
    $stmt->execute([$newStatus, $userId]);
    
    // Log de la acción
    $operatorId = $_SESSION['operator_id'];
    error_log("Operator {$operatorId} cambió estado de usuario {$userId} a {$newStatus}");
    
    echo json_encode([
        'success' => true,
        'message' => 'Estado actualizado correctamente',
        'new_status' => $newStatus
    ]);
}

function resetUserPassword() {
    global $conn;
    
    $userId = $_POST['user_id'] ?? 0;
    $newPassword = trim($_POST['new_password'] ?? '');
    
    if (!$userId) {
        throw new Exception('ID de usuario inválido');
    }
    
    // Si se proporciona contraseña específica, usarla. Si no, generar automática
    if (!empty($newPassword)) {
        if (strlen($newPassword) < 4) {
            throw new Exception('La contraseña debe tener al menos 4 caracteres');
        }
        $tempPassword = $newPassword;
        $isCustomPassword = true;
    } else {
        $tempPassword = 'Temp' . rand(1000, 9999) . '!';
        $isCustomPassword = false;
    }
    
    $hashedPassword = password_hash($tempPassword, PASSWORD_DEFAULT);
    
    $stmt = $conn->prepare("UPDATE users SET password = ? WHERE id = ?");
    $stmt->execute([$hashedPassword, $userId]);
    
    // Obtener email del usuario
    $stmt = $conn->prepare("SELECT email, full_name as name FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Log de la acción
    $operatorId = $_SESSION['operator_id'];
    $logType = $isCustomPassword ? 'cambió' : 'reseteo';
    error_log("Operator {$operatorId} {$logType} contraseña del usuario {$userId} ({$user['email']})");
    
    // Aquí podrías enviar un email con la nueva contraseña
    // mail($user['email'], 'Contraseña Restablecida', "Tu nueva contraseña es: {$tempPassword}");
    
    echo json_encode([
        'success' => true,
        'message' => $isCustomPassword ? 'Contraseña cambiada correctamente' : 'Contraseña restablecida correctamente',
        'temp_password' => $tempPassword,
        'user_email' => $user['email'],
        'is_custom' => $isCustomPassword
    ]);
}

function getUserDetails() {
    global $conn;
    
    $userId = $_GET['user_id'] ?? 0;
    
    if (!$userId) {
        throw new Exception('ID de usuario inválido');
    }
    
    $stmt = $conn->prepare("
        SELECT 
            u.*,
            COUNT(DISTINCT a.alert_id) as total_alerts,
            SUM(CASE WHEN a.alert_type = 'medical' THEN 1 ELSE 0 END) as medical_alerts,
            SUM(CASE WHEN a.alert_type = 'fire' THEN 1 ELSE 0 END) as fire_alerts,
            SUM(CASE WHEN a.alert_type = 'security' THEN 1 ELSE 0 END) as security_alerts,
            SUM(CASE WHEN a.status = 'resolved' THEN 1 ELSE 0 END) as resolved_alerts,
            MAX(a.created_at) as last_alert_date
        FROM users u
        LEFT JOIN alerts a ON u.id = a.user_id
        WHERE u.id = ?
        GROUP BY u.id
    ");
    $stmt->execute([$userId]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        throw new Exception('Usuario no encontrado');
    }
    
    echo json_encode([
        'success' => true,
        'user' => $user
    ]);
}

function getUserHistory() {
    global $conn;
    
    $userId = $_GET['user_id'] ?? 0;
    $limit = $_GET['limit'] ?? 50;
    
    if (!$userId) {
        throw new Exception('ID de usuario inválido');
    }
    
    $stmt = $conn->prepare("
        SELECT 
            a.*,
            o.name as operator_name
        FROM alerts a
        LEFT JOIN operators o ON a.operator_id = o.operator_id
        WHERE a.user_id = ?
        ORDER BY a.created_at DESC
        LIMIT ?
    ");
    $stmt->execute([$userId, $limit]);
    $alerts = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'alerts' => $alerts
    ]);
}

function deleteUser() {
    global $conn;
    
    $userId = $_POST['user_id'] ?? 0;
    
    if (!$userId) {
        throw new Exception('ID de usuario inválido');
    }
    
    // Verificar si tiene alertas activas
    $stmt = $conn->prepare("SELECT COUNT(*) as active FROM alerts WHERE user_id = ? AND status IN ('pending', 'in_progress')");
    $stmt->execute([$userId]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($result['active'] > 0) {
        throw new Exception('No se puede eliminar: el usuario tiene alertas activas');
    }
    
    // Eliminar usuario (las alertas se mantienen por registro histórico)
    $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    
    // Log de la acción
    $operatorId = $_SESSION['operator_id'];
    error_log("Operator {$operatorId} eliminó al usuario {$userId}");
    
    echo json_encode([
        'success' => true,
        'message' => 'Usuario eliminado correctamente'
    ]);
}
?>
