const CACHE_NAME = 'azuero-safety-v1.0.0';
const DYNAMIC_CACHE = 'azuero-dynamic-v1.0.0';

// Archivos esenciales para cachear
const STATIC_ASSETS = [
  '/panel-simple/',
  '/panel-simple/panel-ciudadano.php',
  '/panel-simple/dashboard.php',
  '/panel-simple/admin-panel.php',
  '/panel-simple/manifest.json',
  '/panel-simple/icons/icon-192x192.png',
  '/panel-simple/icons/icon-512x512.png'
];

// Instalación del Service Worker
self.addEventListener('install', (event) => {
  console.log('[SW] Instalando Service Worker...');
  event.waitUntil(
    caches.open(CACHE_NAME)
      .then((cache) => {
        console.log('[SW] Cacheando archivos estáticos');
        return cache.addAll(STATIC_ASSETS.map(url => new Request(url, {cache: 'reload'})));
      })
      .catch((err) => {
        console.error('[SW] Error al cachear:', err);
      })
  );
  self.skipWaiting();
});

// Activación del Service Worker
self.addEventListener('activate', (event) => {
  console.log('[SW] Activando Service Worker...');
  event.waitUntil(
    caches.keys().then((cacheNames) => {
      return Promise.all(
        cacheNames.map((cacheName) => {
          if (cacheName !== CACHE_NAME && cacheName !== DYNAMIC_CACHE) {
            console.log('[SW] Eliminando caché antigua:', cacheName);
            return caches.delete(cacheName);
          }
        })
      );
    })
  );
  return self.clients.claim();
});

// Estrategia de caché: Network First con fallback a Cache
self.addEventListener('fetch', (event) => {
  const { request } = event;
  const url = new URL(request.url);

  // Solo cachear requests del mismo origen
  if (url.origin !== location.origin) {
    return;
  }

  // No cachear requests POST (formularios, APIs que modifican datos)
  if (request.method !== 'GET') {
    return;
  }

  // No cachear check-new-alerts.php y otras APIs en tiempo real
  if (url.pathname.includes('check-new-alerts.php') || 
      url.pathname.includes('check-my-alerts.php') ||
      url.pathname.includes('procesar-alerta.php') ||
      url.pathname.includes('marcar-recibida.php')) {
    return;
  }

  event.respondWith(
    fetch(request)
      .then((response) => {
        // Si la respuesta es válida, guardarla en caché dinámica
        if (response && response.status === 200) {
          const responseClone = response.clone();
          caches.open(DYNAMIC_CACHE).then((cache) => {
            cache.put(request, responseClone);
          });
        }
        return response;
      })
      .catch(() => {
        // Si falla la red, intentar servir desde caché
        return caches.match(request).then((cachedResponse) => {
          if (cachedResponse) {
            return cachedResponse;
          }
          
          // Si no hay caché, mostrar página offline para navegación
          if (request.headers.get('accept').includes('text/html')) {
            return caches.match('/panel-simple/');
          }
        });
      })
  );
});

// Push Notifications
self.addEventListener('push', (event) => {
  console.log('[SW] Push recibido:', event);
  
  let notificationData = {
    title: 'Azuero Safety Services',
    body: 'Nueva notificación',
    icon: '/panel-simple/icons/icon-192x192.png',
    badge: '/panel-simple/icons/icon-72x72.png',
    tag: 'azuero-notification',
    requireInteraction: true,
    vibrate: [200, 100, 200, 100, 200]
  };

  if (event.data) {
    try {
      const data = event.data.json();
      notificationData = {
        ...notificationData,
        ...data
      };
    } catch (e) {
      notificationData.body = event.data.text();
    }
  }

  event.waitUntil(
    self.registration.showNotification(notificationData.title, notificationData)
  );
});

// Click en notificación
self.addEventListener('notificationclick', (event) => {
  console.log('[SW] Notificación clickeada:', event);
  event.notification.close();

  event.waitUntil(
    clients.openWindow(event.notification.data?.url || '/panel-simple/dashboard.php')
  );
});

// Sincronización en segundo plano (para enviar alertas cuando vuelva la conexión)
self.addEventListener('sync', (event) => {
  console.log('[SW] Sincronización en segundo plano:', event.tag);
  
  if (event.tag === 'sync-alerts') {
    event.waitUntil(syncPendingAlerts());
  }
});

// Función para sincronizar alertas pendientes
async function syncPendingAlerts() {
  try {
    const cache = await caches.open(DYNAMIC_CACHE);
    const requests = await cache.keys();
    
    for (const request of requests) {
      if (request.url.includes('procesar-alerta.php')) {
        try {
          await fetch(request);
          await cache.delete(request);
          console.log('[SW] Alerta sincronizada:', request.url);
        } catch (err) {
          console.error('[SW] Error al sincronizar alerta:', err);
        }
      }
    }
  } catch (err) {
    console.error('[SW] Error en syncPendingAlerts:', err);
  }
}

console.log('[SW] Service Worker cargado correctamente');
