<?php
require_once 'config.php';

// Verificar si está logueado como operador
if (!isset($_SESSION['operator_id'])) {
    header('Location: index.php');
    exit;
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Test de Notificaciones</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
        }
        h1 {
            color: #667eea;
            margin-bottom: 30px;
            text-align: center;
        }
        .test-section {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            border-left: 4px solid #667eea;
        }
        .test-section h3 {
            color: #333;
            margin-bottom: 15px;
        }
        .btn {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 8px;
            cursor: pointer;
            font-size: 16px;
            margin: 5px;
            transition: transform 0.2s;
        }
        .btn:hover {
            transform: translateY(-2px);
        }
        .btn-success {
            background: linear-gradient(135deg, #10b981 0%, #059669 100%);
        }
        .btn-danger {
            background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
        }
        .log {
            background: #1e293b;
            color: #22c55e;
            padding: 15px;
            border-radius: 8px;
            font-family: 'Courier New', monospace;
            font-size: 14px;
            max-height: 400px;
            overflow-y: auto;
            margin-top: 15px;
        }
        .log-entry {
            margin: 5px 0;
        }
        .log-error { color: #ef4444; }
        .log-success { color: #22c55e; }
        .log-info { color: #3b82f6; }
        .log-warning { color: #f59e0b; }
        .status {
            display: inline-block;
            padding: 5px 10px;
            border-radius: 5px;
            font-size: 12px;
            font-weight: bold;
            margin-left: 10px;
        }
        .status-ok { background: #22c55e; color: white; }
        .status-error { background: #ef4444; color: white; }
        .status-pending { background: #f59e0b; color: white; }
    </style>
</head>
<body>
    <div class="container">
        <h1><i class="fas fa-vial"></i> Diagnóstico de Notificaciones</h1>
        
        <!-- Test 1: Conexión a API -->
        <div class="test-section">
            <h3>1️⃣ Test de Conexión a API</h3>
            <button class="btn" onclick="testAPI()">
                <i class="fas fa-plug"></i> Probar check-new-alerts.php
            </button>
            <span id="api-status"></span>
            <div id="api-log" class="log" style="display:none;"></div>
        </div>

        <!-- Test 2: Sonido de Alarma -->
        <div class="test-section">
            <h3>2️⃣ Test de Sonido</h3>
            <button class="btn btn-danger" onclick="testSound()">
                <i class="fas fa-volume-up"></i> Probar Alarma
            </button>
            <button class="btn" onclick="stopSound()">
                <i class="fas fa-stop"></i> Detener
            </button>
            <span id="sound-status"></span>
        </div>

        <!-- Test 3: Sistema de Polling -->
        <div class="test-section">
            <h3>3️⃣ Test de Polling Automático</h3>
            <button class="btn btn-success" onclick="startPolling()">
                <i class="fas fa-play"></i> Iniciar Polling (cada 3s)
            </button>
            <button class="btn btn-danger" onclick="stopPolling()">
                <i class="fas fa-stop"></i> Detener Polling
            </button>
            <span id="polling-status"></span>
            <div id="polling-log" class="log"></div>
        </div>

        <!-- Test 4: Simular Notificación -->
        <div class="test-section">
            <h3>4️⃣ Test de Toast Notification</h3>
            <button class="btn btn-danger" onclick="testToast()">
                <i class="fas fa-bell"></i> Mostrar Notificación de Prueba
            </button>
        </div>

        <!-- Test 5: Verificar Base de Datos -->
        <div class="test-section">
            <h3>5️⃣ Verificar Alertas en BD</h3>
            <button class="btn" onclick="checkDB()">
                <i class="fas fa-database"></i> Ver Últimas Alertas
            </button>
            <div id="db-log" class="log" style="display:none;"></div>
        </div>

        <div style="margin-top: 30px; text-align: center;">
            <a href="dashboard.php" class="btn"><i class="fas fa-arrow-left"></i> Volver al Dashboard</a>
        </div>
    </div>

    <!-- Toast Container -->
    <div id="toastContainer" style="position: fixed; top: 20px; right: 20px; z-index: 9999;"></div>

    <!-- Audio embebido -->
    <audio id="alarmSound" preload="auto">
        <source src="data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmwhBTGJ0fPTgjMGHm7A7+OZSA0PVKrl7K1aFgpBmeHyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgo=" type="audio/wav">
    </audio>

    <script>
        let pollingInterval = null;
        let lastCheckTs = Math.floor(Date.now()/1000);

        function log(containerId, message, type = 'info') {
            const container = document.getElementById(containerId);
            if (!container) return;
            
            container.style.display = 'block';
            const timestamp = new Date().toLocaleTimeString();
            const entry = document.createElement('div');
            entry.className = 'log-entry log-' + type;
            entry.innerHTML = `[${timestamp}] ${message}`;
            container.appendChild(entry);
            container.scrollTop = container.scrollHeight;
        }

        function setStatus(id, text, ok = true) {
            const el = document.getElementById(id);
            if (el) {
                el.innerHTML = `<span class="status status-${ok ? 'ok' : 'error'}">${text}</span>`;
            }
        }

        // Test 1: API
        function testAPI() {
            log('api-log', '🔌 Conectando a check-new-alerts.php...', 'info');
            
            fetch('check-new-alerts.php?last_check=0')
                .then(response => {
                    log('api-log', `✅ Respuesta recibida: ${response.status}`, 'success');
                    return response.json();
                })
                .then(data => {
                    log('api-log', '📦 Datos recibidos:', 'success');
                    log('api-log', JSON.stringify(data, null, 2), 'info');
                    
                    if (data.error) {
                        log('api-log', `❌ ERROR: ${data.error}`, 'error');
                        setStatus('api-status', 'ERROR', false);
                    } else {
                        log('api-log', `✅ Nuevas alertas: ${data.new_count || 0}`, 'success');
                        log('api-log', `📊 Total alertas en respuesta: ${data.alerts ? data.alerts.length : 0}`, 'info');
                        setStatus('api-status', 'OK', true);
                    }
                })
                .catch(error => {
                    log('api-log', `❌ Error de conexión: ${error.message}`, 'error');
                    setStatus('api-status', 'ERROR', false);
                });
        }

        // Test 2: Sonido
        function testSound() {
            const audio = document.getElementById('alarmSound');
            if (audio) {
                try {
                    audio.currentTime = 0;
                    audio.play()
                        .then(() => {
                            log('polling-log', '🔊 Sonido reproducido correctamente', 'success');
                            setStatus('sound-status', 'SONANDO', true);
                        })
                        .catch(err => {
                            log('polling-log', `❌ Error al reproducir: ${err.message}`, 'error');
                            setStatus('sound-status', 'ERROR', false);
                        });
                } catch(e) {
                    log('polling-log', `❌ Excepción: ${e.message}`, 'error');
                    setStatus('sound-status', 'ERROR', false);
                }
            } else {
                log('polling-log', '❌ Elemento de audio no encontrado', 'error');
                setStatus('sound-status', 'NO ENCONTRADO', false);
            }
        }

        function stopSound() {
            const audio = document.getElementById('alarmSound');
            if (audio) {
                audio.pause();
                audio.currentTime = 0;
                setStatus('sound-status', 'DETENIDO', true);
            }
        }

        // Test 3: Polling
        function startPolling() {
            if (pollingInterval) {
                log('polling-log', '⚠️ Polling ya está activo', 'warning');
                return;
            }

            log('polling-log', '▶️ Iniciando polling cada 3 segundos...', 'info');
            setStatus('polling-status', 'ACTIVO', true);
            
            pollingInterval = setInterval(() => {
                const url = 'check-new-alerts.php?last_check=' + lastCheckTs;
                log('polling-log', `🔄 Consultando: last_check=${lastCheckTs}`, 'info');
                
                fetch(url)
                    .then(response => response.json())
                    .then(data => {
                        if (data.error) {
                            log('polling-log', `❌ Error: ${data.error}`, 'error');
                            return;
                        }

                        if (data.has_new) {
                            log('polling-log', `🚨 ¡${data.new_count} NUEVA(S) ALERTA(S) DETECTADA(S)!`, 'success');
                            testSound();
                            
                            // Mostrar detalles de la primera
                            if (data.alerts && data.alerts.length > 0) {
                                const alert = data.alerts[0];
                                log('polling-log', `📋 ID: ${alert.id} | Tipo: ${alert.alert_type} | GPS: ${alert.latitude},${alert.longitude}`, 'info');
                            }
                        } else {
                            log('polling-log', `✓ Sin nuevas alertas (total: ${data.alerts ? data.alerts.length : 0})`, 'info');
                        }

                        lastCheckTs = data.timestamp || Math.floor(Date.now()/1000);
                    })
                    .catch(error => {
                        log('polling-log', `❌ Error de red: ${error.message}`, 'error');
                    });
            }, 3000);
        }

        function stopPolling() {
            if (pollingInterval) {
                clearInterval(pollingInterval);
                pollingInterval = null;
                log('polling-log', '⏸️ Polling detenido', 'warning');
                setStatus('polling-status', 'DETENIDO', false);
            }
        }

        // Test 4: Toast
        function testToast() {
            const toast = document.createElement('div');
            toast.style.cssText = 'background: linear-gradient(135deg, #ef4444, #dc2626); color: white; padding: 20px; border-radius: 10px; box-shadow: 0 5px 15px rgba(0,0,0,0.3); min-width: 300px; animation: slideIn 0.3s;';
            toast.innerHTML = `
                <div style="display: flex; align-items: center;">
                    <i class="fas fa-exclamation-circle" style="font-size: 24px; margin-right: 15px;"></i>
                    <div>
                        <h4 style="margin: 0 0 5px 0;">¡Alerta de Prueba!</h4>
                        <p style="margin: 0; font-size: 14px;">Esta es una notificación de emergencia de prueba.</p>
                        <p style="margin: 5px 0 0 0; font-size: 12px;">📍 GPS: 8.123456, -80.654321</p>
                    </div>
                </div>
            `;
            
            document.getElementById('toastContainer').appendChild(toast);
            testSound();
            
            setTimeout(() => {
                toast.style.animation = 'slideOut 0.3s';
                setTimeout(() => toast.remove(), 300);
            }, 5000);
        }

        // Test 5: Base de Datos
        function checkDB() {
            log('db-log', '🔍 Consultando últimas alertas...', 'info');
            
            fetch('check-new-alerts.php?last_check=0')
                .then(response => response.json())
                .then(data => {
                    if (data.alerts && data.alerts.length > 0) {
                        log('db-log', `✅ ${data.alerts.length} alertas encontradas:`, 'success');
                        data.alerts.forEach((alert, i) => {
                            log('db-log', `${i+1}. ID:${alert.id} | ${alert.alert_type} | ${alert.status} | GPS:(${alert.latitude},${alert.longitude})`, 'info');
                        });
                    } else {
                        log('db-log', '⚠️ No hay alertas en la base de datos', 'warning');
                    }
                })
                .catch(error => {
                    log('db-log', `❌ Error: ${error.message}`, 'error');
                });
        }
    </script>

    <style>
        @keyframes slideIn {
            from { transform: translateX(400px); opacity: 0; }
            to { transform: translateX(0); opacity: 1; }
        }
        @keyframes slideOut {
            from { transform: translateX(0); opacity: 1; }
            to { transform: translateX(400px); opacity: 0; }
        }
    </style>
</body>
</html>
