<?php
require_once 'config.php';

if (!isset($_SESSION['operator_id'])) {
    header('Location: index.php');
    exit;
}

// Obtener información del operador
try {
    $stmt = $conn->prepare("SELECT * FROM operators WHERE id = ?");
    $stmt->execute([$_SESSION['operator_id']]);
    $operator = $stmt->fetch(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    die("Error");
}

$alert_id = $_GET['id'] ?? 0;
$message = '';
$message_type = '';

// Procesar acciones
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $action = $_POST['action'] ?? '';
    
    try {
        if ($action == 'assign') {
            $stmt = $conn->prepare("UPDATE alerts SET status = 'in_progress', assigned_operator_id = ?, response_time = NOW(), updated_at = NOW() WHERE id = ?");
            $stmt->execute([$_SESSION['operator_id'], $alert_id]);
            $message = "Alerta asignada correctamente";
            $message_type = "success";
        } elseif ($action == 'resolve') {
            $notes = $_POST['notes'] ?? '';
            $stmt = $conn->prepare("UPDATE alerts SET status = 'resolved', notes = ?, resolved_at = NOW(), updated_at = NOW() WHERE id = ?");
            $stmt->execute([$notes, $alert_id]);
            $message = "Alerta resuelta correctamente";
            $message_type = "success";
        } elseif ($action == 'update_notes') {
            $notes = $_POST['notes'] ?? '';
            $stmt = $conn->prepare("UPDATE alerts SET notes = ?, updated_at = NOW() WHERE id = ?");
            $stmt->execute([$notes, $alert_id]);
            $message = "Notas actualizadas";
            $message_type = "success";
        }
    } catch(PDOException $e) {
        $message = "Error: " . $e->getMessage();
        $message_type = "error";
    }
}

// Obtener datos de la alerta
try {
    $stmt = $conn->prepare("
        SELECT a.*, u.full_name as user_name, u.phone as user_phone, u.email as user_email,
               u.blood_type, u.allergies, u.medical_conditions, u.emergency_contact, u.emergency_phone,
               o.full_name as operator_name
        FROM alerts a 
        LEFT JOIN users u ON a.user_id = u.id
        LEFT JOIN operators o ON a.assigned_operator_id = o.id 
        WHERE a.id = ?
    ");
    $stmt->execute([$alert_id]);
    $alert = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$alert) {
        header('Location: alertas.php');
        exit;
    }
} catch(PDOException $e) {
    die("Error: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Alerta #<?php echo $alert['id']; ?> - Azuero Safety Services S.A</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <!-- Leaflet CSS (OpenStreetMap - No requiere API key) -->
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" 
          integrity="sha256-p4NxAoJBhIIN+hmNHrzRCf9tD/miZyoHS5obTRR9BMY=" crossorigin=""/>
    
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary: #667eea;
            --primary-dark: #5568d3;
            --secondary: #764ba2;
            --success: #10b981;
            --warning: #f59e0b;
            --danger: #ef4444;
            --info: #3b82f6;
            --bg-primary: #ffffff;
            --bg-secondary: #f8fafc;
            --bg-card: #ffffff;
            --text-primary: #1e293b;
            --text-secondary: #64748b;
            --border-color: #e2e8f0;
            --shadow-sm: 0 1px 3px rgba(0,0,0,0.1);
            --shadow-md: 0 4px 6px rgba(0,0,0,0.1);
            --shadow-lg: 0 10px 25px rgba(0,0,0,0.15);
            --sidebar-width: 280px;
        }

        [data-theme="dark"] {
            --primary: #818cf8;
            --primary-dark: #6366f1;
            --bg-primary: #0f172a;
            --bg-secondary: #1e293b;
            --bg-card: #1e293b;
            --text-primary: #f1f5f9;
            --text-secondary: #94a3b8;
            --border-color: #334155;
        }

        [data-theme="blue-dark"] {
            --primary: #60a5fa;
            --primary-dark: #3b82f6;
            --secondary: #06b6d4;
            --bg-primary: #0c1e35;
            --bg-secondary: #1a2942;
            --bg-card: #1a2942;
            --text-primary: #e0f2fe;
            --text-secondary: #7dd3fc;
            --border-color: #1e3a5f;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: var(--bg-primary);
            color: var(--text-primary);
        }

        /* Sidebar */
        .sidebar {
            position: fixed;
            left: 0;
            top: 0;
            bottom: 0;
            width: var(--sidebar-width);
            background: var(--bg-card);
            border-right: 1px solid var(--border-color);
            box-shadow: var(--shadow-md);
            z-index: 1000;
            overflow-y: auto;
        }

        .sidebar-header {
            padding: 30px 20px;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            text-align: center;
        }

        .sidebar-header h2 {
            font-size: 24px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }

        .sidebar-menu {
            padding: 20px 0;
        }

        .menu-item {
            display: flex;
            align-items: center;
            padding: 15px 25px;
            color: var(--text-primary);
            text-decoration: none;
            transition: all 0.3s ease;
            border-left: 3px solid transparent;
        }

        .menu-item:hover {
            background: var(--bg-secondary);
            border-left-color: var(--primary);
            color: var(--primary);
        }

        .menu-item i {
            width: 24px;
            font-size: 20px;
            margin-right: 15px;
        }

        .menu-section {
            padding: 10px 25px;
            font-size: 12px;
            font-weight: 600;
            color: var(--text-secondary);
            text-transform: uppercase;
            margin-top: 20px;
        }

        /* Main Content */
        .main-content {
            margin-left: var(--sidebar-width);
            min-height: 100vh;
        }

        /* Header */
        .header {
            background: var(--bg-card);
            border-bottom: 1px solid var(--border-color);
            padding: 20px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: var(--shadow-sm);
            position: sticky;
            top: 0;
            z-index: 999;
        }

        .header-left h1 {
            font-size: 28px;
            color: var(--text-primary);
        }

        .breadcrumb {
            color: var(--text-secondary);
            font-size: 14px;
            margin-top: 5px;
        }

        .header-right {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .theme-switcher {
            display: flex;
            gap: 10px;
            background: var(--bg-secondary);
            padding: 5px;
            border-radius: 25px;
        }

        .theme-btn {
            padding: 8px 15px;
            border: none;
            background: transparent;
            color: var(--text-secondary);
            cursor: pointer;
            border-radius: 20px;
            font-size: 13px;
            transition: all 0.3s ease;
        }

        .theme-btn.active {
            background: var(--primary);
            color: white;
        }

        /* Content */
        .content {
            padding: 30px;
        }

        .alert-message {
            padding: 15px 20px;
            border-radius: 12px;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 12px;
            font-size: 14px;
            font-weight: 500;
        }

        .alert-message.success {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
            border-left: 4px solid var(--success);
        }

        .alert-message.error {
            background: rgba(239, 68, 68, 0.1);
            color: var(--danger);
            border-left: 4px solid var(--danger);
        }

        .alert-header {
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 15px;
            padding: 30px;
            box-shadow: var(--shadow-sm);
            margin-bottom: 30px;
        }

        .alert-status-badge {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
            margin-bottom: 20px;
        }

        .status-pending {
            background: rgba(245, 158, 11, 0.1);
            color: var(--warning);
        }

        .status-in_progress {
            background: rgba(59, 130, 246, 0.1);
            color: var(--info);
        }

        .status-resolved {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
        }

        .alert-title {
            font-size: 32px;
            margin-bottom: 10px;
            color: var(--text-primary);
        }

        .alert-meta {
            display: flex;
            gap: 30px;
            flex-wrap: wrap;
            color: var(--text-secondary);
            font-size: 14px;
        }

        .alert-meta-item {
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .grid-2 {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
            gap: 30px;
            margin-bottom: 30px;
        }

        .card {
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 15px;
            padding: 25px;
            box-shadow: var(--shadow-sm);
        }

        .card-header {
            display: flex;
            align-items: center;
            gap: 12px;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 1px solid var(--border-color);
        }

        .card-icon {
            width: 40px;
            height: 40px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 20px;
            color: white;
        }

        .card-icon.blue { background: linear-gradient(135deg, #667eea, #764ba2); }
        .card-icon.green { background: linear-gradient(135deg, #10b981, #059669); }
        .card-icon.orange { background: linear-gradient(135deg, #f59e0b, #d97706); }
        .card-icon.red { background: linear-gradient(135deg, #ef4444, #dc2626); }

        .card-title {
            font-size: 18px;
            font-weight: 600;
            color: var(--text-primary);
        }

        .info-row {
            display: flex;
            justify-content: space-between;
            padding: 12px 0;
            border-bottom: 1px solid var(--border-color);
        }

        .info-row:last-child {
            border-bottom: none;
        }

        .info-label {
            color: var(--text-secondary);
            font-size: 14px;
        }

        .info-value {
            color: var(--text-primary);
            font-weight: 500;
            font-size: 14px;
        }

        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 500;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-success {
            background: var(--success);
            color: white;
        }

        .btn-danger {
            background: var(--danger);
            color: white;
        }

        .btn:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-md);
        }

        .btn-group {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }

        textarea {
            width: 100%;
            padding: 12px;
            border: 1px solid var(--border-color);
            border-radius: 8px;
            background: var(--bg-secondary);
            color: var(--text-primary);
            font-family: inherit;
            font-size: 14px;
            resize: vertical;
            min-height: 120px;
        }

        textarea:focus {
            outline: none;
            border-color: var(--primary);
        }

        .badge {
            display: inline-block;
            padding: 5px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 500;
        }

        .badge-danger { background: rgba(239, 68, 68, 0.1); color: var(--danger); }
        .badge-warning { background: rgba(245, 158, 11, 0.1); color: var(--warning); }
        .badge-info { background: rgba(59, 130, 246, 0.1); color: var(--info); }

        #map {
            width: 100%;
            height: 450px;
            border-radius: 12px;
            background: var(--bg-secondary);
            overflow: hidden;
            box-shadow: var(--shadow-md);
        }

        .map-fallback {
            display: flex;
            align-items: center;
            justify-content: center;
            flex-direction: column;
            height: 100%;
            color: var(--text-secondary);
            padding: 40px;
            text-align: center;
        }

        @media (max-width: 1024px) {
            .sidebar {
                transform: translateX(-100%);
            }
            .main-content {
                margin-left: 0;
            }
            .grid-2 {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body data-theme="light">
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <h2>
                <i class="fas fa-shield-alt"></i>
                Azuero Safety Services S.A
            </h2>
            <p style="font-size: 11px; opacity: 0.85; margin-top: 5px;">Somos tu seguridad</p>
        </div>

        <div class="sidebar-menu">
            <a href="dashboard.php" class="menu-item">
                <i class="fas fa-home"></i>
                <span>Dashboard</span>
            </a>

            <div class="menu-section">Alertas</div>
            
            <a href="alertas.php" class="menu-item">
                <i class="fas fa-bell"></i>
                <span>Todas las Alertas</span>
            </a>

            <?php if ($operator['role'] === 'admin'): ?>
            <div class="menu-section">Administración</div>
            
            <a href="admin-panel.php" class="menu-item">
                <i class="fas fa-user-shield"></i>
                <span>Panel Admin</span>
            </a>
            <?php endif; ?>

            <div class="menu-section">Sistema</div>

            <a href="logout.php" class="menu-item">
                <i class="fas fa-sign-out-alt"></i>
                <span>Cerrar Sesión</span>
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Header -->
        <div class="header">
            <div class="header-left">
                <h1>Alerta #<?php echo $alert['id']; ?></h1>
                <div class="breadcrumb">
                    <a href="dashboard.php" style="color: var(--text-secondary); text-decoration: none;"><i class="fas fa-home"></i></a> / 
                    <a href="alertas.php" style="color: var(--text-secondary); text-decoration: none;">Alertas</a> / 
                    #<?php echo $alert['id']; ?>
                </div>
            </div>

            <div class="header-right">
                <div class="theme-switcher">
                    <button class="theme-btn active" data-theme="light" onclick="changeTheme('light')">☀️</button>
                    <button class="theme-btn" data-theme="dark" onclick="changeTheme('dark')">🌙</button>
                    <button class="theme-btn" data-theme="blue-dark" onclick="changeTheme('blue-dark')">🌊</button>
                </div>

                <a href="alertas.php" class="btn btn-primary">
                    <i class="fas fa-arrow-left"></i> Volver
                </a>
            </div>
        </div>

        <!-- Content -->
        <div class="content">
            <?php if($message): ?>
            <div class="alert-message <?php echo $message_type; ?>">
                <i class="fas <?php echo $message_type == 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'; ?>"></i>
                <?php echo $message; ?>
            </div>
            <?php endif; ?>

            <!-- Alert Header -->
            <div class="alert-header">
                <div class="alert-status-badge status-<?php echo $alert['status']; ?>">
                    <i class="fas fa-circle" style="font-size: 8px;"></i>
                    <?php 
                    $status_text = [
                        'pending' => 'PENDIENTE',
                        'in_progress' => 'EN PROGRESO',
                        'resolved' => 'RESUELTA'
                    ];
                    echo $status_text[$alert['status']] ?? strtoupper($alert['status']);
                    ?>
                </div>

                <h2 class="alert-title">
                    <?php
                    $type_text = [
                        'medical' => '🚑 Emergencia Médica',
                        'fire' => '🔥 Incendio',
                        'security' => '🚨 Seguridad'
                    ];
                    echo $type_text[$alert['alert_type']] ?? '⚠️ Alerta';
                    ?>
                </h2>

                <div class="alert-meta">
                    <div class="alert-meta-item">
                        <i class="fas fa-calendar"></i>
                        <?php echo date('d/m/Y H:i:s', strtotime($alert['created_at'])); ?>
                    </div>
                    <div class="alert-meta-item">
                        <i class="fas fa-map-marker-alt"></i>
                        <?php echo htmlspecialchars($alert['location'] ?? 'Sin ubicación'); ?>
                    </div>
                    <div class="alert-meta-item">
                        <i class="fas fa-flag"></i>
                        <?php
                        $priority_text = ['high' => 'ALTA', 'medium' => 'MEDIA', 'low' => 'BAJA'];
                        $priority_colors = ['high' => 'danger', 'medium' => 'warning', 'low' => 'info'];
                        ?>
                        <span class="badge badge-<?php echo $priority_colors[$alert['priority']]; ?>">
                            <?php echo $priority_text[$alert['priority']]; ?>
                        </span>
                    </div>
                </div>
            </div>

            <!-- Content Grid -->
            <div class="grid-2">
                <!-- Citizen Info -->
                <div class="card">
                    <div class="card-header">
                        <div class="card-icon blue">
                            <i class="fas fa-user"></i>
                        </div>
                        <h3 class="card-title">Información del Ciudadano</h3>
                    </div>
                    
                    <div class="info-row">
                        <span class="info-label">Nombre</span>
                        <span class="info-value"><?php echo htmlspecialchars($alert['user_name'] ?? 'Desconocido'); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Teléfono</span>
                        <span class="info-value"><?php echo htmlspecialchars($alert['user_phone'] ?? 'No especificado'); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Email</span>
                        <span class="info-value"><?php echo htmlspecialchars($alert['user_email'] ?? 'No especificado'); ?></span>
                    </div>
                </div>

                <!-- Alert Details -->
                <div class="card">
                    <div class="card-header">
                        <div class="card-icon orange">
                            <i class="fas fa-info-circle"></i>
                        </div>
                        <h3 class="card-title">Detalles de la Alerta</h3>
                    </div>
                    
                    <div class="info-row">
                        <span class="info-label">ID</span>
                        <span class="info-value">#<?php echo $alert['id']; ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Tipo</span>
                        <span class="info-value"><?php echo ucfirst($alert['alert_type']); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Prioridad</span>
                        <span class="info-value"><?php echo $priority_text[$alert['priority']]; ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Operador Asignado</span>
                        <span class="info-value"><?php echo htmlspecialchars($alert['operator_name'] ?? 'Sin asignar'); ?></span>
                    </div>
                    
                    <?php if (!empty($alert['description'])): ?>
                    <div class="info-row" style="border-top: 2px solid var(--border-color); margin-top: 15px; padding-top: 15px;">
                        <span class="info-label" style="color: var(--danger); font-weight: 600;">
                            <i class="fas fa-comment-dots"></i> Mensaje del Ciudadano
                        </span>
                        <span class="info-value" style="background: var(--bg-secondary); padding: 15px; border-radius: 8px; border-left: 4px solid var(--danger); font-style: italic; line-height: 1.6;">
                            "<?php echo nl2br(htmlspecialchars($alert['description'])); ?>"
                        </span>
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Map -->
            <div class="card" style="margin-bottom: 30px;">
                <div class="card-header">
                    <div class="card-icon red">
                        <i class="fas fa-map-marked-alt"></i>
                    </div>
                    <h3 class="card-title">Ubicación GPS</h3>
                </div>
                
                <?php if (!empty($alert['latitude']) && !empty($alert['longitude'])): ?>
                <!-- Información de Coordenadas -->
                <div style="padding: 20px; background: linear-gradient(135deg, #ef4444, #dc2626); color: white; text-align: center; border-radius: 12px 12px 0 0;">
                    <div style="display: inline-flex; align-items: center; gap: 15px; background: rgba(255,255,255,0.2); padding: 15px 25px; border-radius: 50px; backdrop-filter: blur(10px);">
                        <i class="fas fa-map-marker-alt" style="font-size: 24px;"></i>
                        <span style="font-size: 18px; font-weight: 600; font-family: 'Courier New', monospace;">
                            <?php echo number_format($alert['latitude'], 6); ?>, <?php echo number_format($alert['longitude'], 6); ?>
                        </span>
                    </div>
                </div>
                
                <!-- Mapa Interactivo -->
                <div id="map"></div>
                
                <!-- Botones de Acción -->
                <div style="padding: 20px; display: flex; gap: 10px; justify-content: center; flex-wrap: wrap; background: var(--bg-secondary); border-radius: 0 0 12px 12px;">
                    <a href="https://www.google.com/maps?q=<?php echo $alert['latitude']; ?>,<?php echo $alert['longitude']; ?>" 
                       target="_blank" 
                       class="btn btn-success" 
                       style="text-decoration: none; display: inline-flex; align-items: center; gap: 8px;">
                        <i class="fab fa-google"></i> Google Maps
                    </a>
                    
                    <a href="https://www.waze.com/ul?ll=<?php echo $alert['latitude']; ?>,<?php echo $alert['longitude']; ?>&navigate=yes" 
                       target="_blank" 
                       class="btn btn-primary" 
                       style="background: #33CCFF; text-decoration: none; display: inline-flex; align-items: center; gap: 8px;">
                        <i class="fab fa-waze"></i> Waze
                    </a>
                    
                    <button onclick="copyCoordinates('<?php echo $alert['latitude']; ?>', '<?php echo $alert['longitude']; ?>')" 
                            class="btn btn-primary" 
                            style="display: inline-flex; align-items: center; gap: 8px;">
                        <i class="fas fa-copy"></i> Copiar
                    </button>
                </div>
                
                <?php if (!empty($alert['location'])): ?>
                <div style="padding: 15px; text-align: center; background: var(--bg-secondary); border-top: 1px solid var(--border-color);">
                    <p style="font-size: 14px; color: var(--text-secondary); font-style: italic;">
                        <i class="fas fa-info-circle"></i> <?php echo htmlspecialchars($alert['location']); ?>
                    </p>
                </div>
                <?php endif; ?>
                
                <?php else: ?>
                <!-- Sin GPS -->
                <div class="map-fallback">
                    <i class="fas fa-map-marker-alt" style="font-size: 64px; margin-bottom: 20px; opacity: 0.2;"></i>
                    <h4 style="font-size: 18px; font-weight: 600; margin-bottom: 10px;">📍 Ubicación GPS no disponible</h4>
                    <p style="font-size: 14px; opacity: 0.7;">El usuario no compartió su ubicación durante la emergencia</p>
                </div>
                <?php endif; ?>
            </div>

            <!-- Notes -->
            <div class="card" style="margin-bottom: 30px;">
                <div class="card-header">
                    <div class="card-icon green">
                        <i class="fas fa-sticky-note"></i>
                    </div>
                    <h3 class="card-title">Notas del Operador</h3>
                </div>
                <form method="POST">
                    <input type="hidden" name="action" value="update_notes">
                    <textarea name="notes" placeholder="Agregar notas sobre la alerta..."><?php echo htmlspecialchars($alert['notes'] ?? ''); ?></textarea>
                    <div style="margin-top: 15px;">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i> Guardar Notas
                        </button>
                    </div>
                </form>
            </div>

            <!-- Actions -->
            <div class="card">
                <div class="card-header">
                    <div class="card-icon blue">
                        <i class="fas fa-tasks"></i>
                    </div>
                    <h3 class="card-title">Acciones</h3>
                </div>
                
                <div class="btn-group">
                    <?php if ($alert['status'] == 'pending'): ?>
                    <form method="POST" style="display: inline;">
                        <input type="hidden" name="action" value="assign">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-hand-paper"></i> Asignar a mí
                        </button>
                    </form>
                    <?php endif; ?>

                    <?php if ($alert['status'] == 'in_progress' && $alert['assigned_operator_id'] == $_SESSION['operator_id']): ?>
                    <form method="POST" style="display: inline;">
                        <input type="hidden" name="action" value="resolve">
                        <input type="hidden" name="notes" value="<?php echo htmlspecialchars($alert['notes'] ?? ''); ?>">
                        <button type="submit" class="btn btn-success">
                            <i class="fas fa-check"></i> Marcar como Resuelta
                        </button>
                    </form>
                    <?php endif; ?>

                    <a href="alertas.php" class="btn btn-primary">
                        <i class="fas fa-arrow-left"></i> Volver a Alertas
                    </a>
                </div>
            </div>
        </div>
    </div>

    <script>
        function changeTheme(theme) {
            document.body.setAttribute('data-theme', theme);
            localStorage.setItem('theme', theme);
            
            document.querySelectorAll('.theme-btn').forEach(btn => {
                btn.classList.remove('active');
                if(btn.getAttribute('data-theme') === theme) {
                    btn.classList.add('active');
                }
            });
        }

        const savedTheme = localStorage.getItem('theme') || 'light';
        changeTheme(savedTheme);

        // Función para copiar coordenadas al portapapeles
        function copyCoordinates(lat, lng) {
            const coords = lat + ', ' + lng;
            
            // Método moderno (si está disponible)
            if (navigator.clipboard && navigator.clipboard.writeText) {
                navigator.clipboard.writeText(coords).then(function() {
                    showCopyNotification('✅ Coordenadas copiadas: ' + coords);
                }).catch(function(err) {
                    fallbackCopyCoordinates(coords);
                });
            } else {
                fallbackCopyCoordinates(coords);
            }
        }

        // Método alternativo para navegadores antiguos
        function fallbackCopyCoordinates(coords) {
            const textArea = document.createElement('textarea');
            textArea.value = coords;
            textArea.style.position = 'fixed';
            textArea.style.left = '-999999px';
            document.body.appendChild(textArea);
            textArea.select();
            try {
                document.execCommand('copy');
                showCopyNotification('✅ Coordenadas copiadas: ' + coords);
            } catch (err) {
                showCopyNotification('❌ No se pudo copiar. Coordenadas: ' + coords);
            }
            document.body.removeChild(textArea);
        }

        // Mostrar notificación temporal
        function showCopyNotification(message) {
            const notification = document.createElement('div');
            notification.style.cssText = 'position: fixed; top: 20px; right: 20px; background: linear-gradient(135deg, #10b981, #059669); color: white; padding: 15px 25px; border-radius: 10px; font-weight: 600; box-shadow: 0 4px 20px rgba(0,0,0,0.3); z-index: 10000; animation: slideIn 0.3s;';
            notification.textContent = message;
            document.body.appendChild(notification);
            
            setTimeout(function() {
                notification.style.animation = 'slideOut 0.3s';
                setTimeout(function() {
                    document.body.removeChild(notification);
                }, 300);
            }, 3000);
        }
    </script>

    <!-- Leaflet JS -->
    <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js" 
            integrity="sha256-20nQCchB9co0qIjJZRGuk2/Z9VM+kNiyxNV1lvTlZBo=" crossorigin=""></script>
    
    <script>
        // Inicializar mapa si hay coordenadas
        <?php if (!empty($alert['latitude']) && !empty($alert['longitude'])): ?>
        document.addEventListener('DOMContentLoaded', function() {
            try {
                // Crear mapa centrado en la ubicación de la alerta
                var map = L.map('map', {
                    center: [<?php echo $alert['latitude']; ?>, <?php echo $alert['longitude']; ?>],
                    zoom: 16,
                    zoomControl: true,
                    scrollWheelZoom: true
                });

                // Agregar capa de OpenStreetMap
                L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                    attribution: '© <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a>',
                    maxZoom: 19
                }).addTo(map);

                // Crear icono personalizado rojo para emergencia
                var emergencyIcon = L.divIcon({
                    className: 'custom-marker',
                    html: '<div style="background: linear-gradient(135deg, #ef4444, #dc2626); width: 40px; height: 40px; border-radius: 50% 50% 50% 0; transform: rotate(-45deg); border: 3px solid white; box-shadow: 0 4px 8px rgba(0,0,0,0.3); display: flex; align-items: center; justify-content: center;"><i class="fas fa-exclamation" style="color: white; font-size: 20px; transform: rotate(45deg);"></i></div>',
                    iconSize: [40, 40],
                    iconAnchor: [20, 40],
                    popupAnchor: [0, -40]
                });

                // Agregar marcador en la ubicación
                var marker = L.marker([<?php echo $alert['latitude']; ?>, <?php echo $alert['longitude']; ?>], {
                    icon: emergencyIcon
                }).addTo(map);

                // Contenido del popup
                var popupContent = '<div style="text-align: center; padding: 10px; min-width: 200px;">' +
                    '<h4 style="margin: 0 0 10px 0; color: #ef4444; font-size: 16px;">' +
                    '<i class="fas fa-exclamation-triangle"></i> Ubicación de Emergencia</h4>' +
                    '<p style="margin: 5px 0; font-size: 14px;"><strong>Tipo:</strong> <?php echo ucfirst($alert['alert_type']); ?></p>' +
                    '<p style="margin: 5px 0; font-size: 14px;"><strong>Prioridad:</strong> <?php echo strtoupper($alert['priority']); ?></p>' +
                    '<p style="margin: 5px 0; font-size: 12px; font-family: monospace; background: #f3f4f6; padding: 5px; border-radius: 5px;">' +
                    '<?php echo number_format($alert['latitude'], 6); ?>, <?php echo number_format($alert['longitude'], 6); ?></p>' +
                    '</div>';

                marker.bindPopup(popupContent).openPopup();

                // Agregar círculo de área de emergencia (500m de radio)
                L.circle([<?php echo $alert['latitude']; ?>, <?php echo $alert['longitude']; ?>], {
                    color: '#ef4444',
                    fillColor: '#ef4444',
                    fillOpacity: 0.15,
                    radius: 500
                }).addTo(map);

                console.log('✅ Mapa inicializado correctamente');
            } catch (error) {
                console.error('❌ Error inicializando mapa:', error);
                document.getElementById('map').innerHTML = '<div class="map-fallback">' +
                    '<i class="fas fa-exclamation-triangle" style="font-size: 48px; color: #f59e0b; margin-bottom: 15px;"></i>' +
                    '<p style="font-size: 16px; color: var(--text-primary);">Error al cargar el mapa</p>' +
                    '<p style="font-size: 14px; color: var(--text-secondary); margin-top: 10px;">Las coordenadas GPS están disponibles arriba</p>' +
                    '</div>';
            }
        });
        <?php endif; ?>

        // Marcar alerta como recibida automáticamente al abrir esta página
        (function() {
            var alertId = <?php echo $alert_id; ?>;
            var formData = new FormData();
            formData.append('alert_id', alertId);
            
            fetch('marcar-recibida.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    console.log('✅ Alerta marcada como recibida:', data.message);
                    // Opcionalmente mostrar indicador visual
                    if (!data.already_received) {
                        var badge = document.querySelector('.badge.badge-warning, .badge.badge-danger');
                        if (badge && badge.textContent.toLowerCase().includes('pendiente')) {
                            badge.textContent = 'RECIBIDA';
                            badge.className = 'badge badge-info';
                        }
                    }
                } else {
                    console.warn('⚠️ No se pudo marcar como recibida:', data.error);
                }
            })
            .catch(error => {
                console.error('❌ Error al marcar como recibida:', error);
            });
        })();
    </script>
</body>
</html>
