<?php
session_start();
header('Content-Type: application/json');

// Versión móvil - SOLO para ciudadanos
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'citizen') {
    echo json_encode(['success' => false, 'error' => 'No autorizado - Solo ciudadanos']);
    exit;
}

require_once 'config.php';

try {
    $pdo = new PDO(
        "mysql:host=".DB_HOST.";dbname=".DB_NAME.";charset=utf8mb4",
        DB_USER,
        DB_PASS,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
} catch(PDOException $e) {
    echo json_encode(['success' => false, 'error' => 'Error de conexión']);
    exit;
}

$action = $_GET['action'] ?? $_POST['action'] ?? '';
$userId = $_SESSION['user_id'];

switch($action) {
    case 'recent':
        // Obtener alertas recientes del ciudadano
        $stmt = $pdo->prepare("
            SELECT id, alert_type, status, location_name as location, latitude, longitude, created_at, updated_at
            FROM alerts 
            WHERE user_id = ? 
            ORDER BY created_at DESC 
            LIMIT 5
        ");
        $stmt->execute([$userId]);
        $alerts = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode(['success' => true, 'data' => $alerts]);
        break;

    case 'all':
        // Obtener todas las alertas del ciudadano con paginación
        $page = $_GET['page'] ?? 1;
        $limit = $_GET['limit'] ?? 10;
        $offset = ($page - 1) * $limit;

        $stmt = $pdo->prepare("
            SELECT id, alert_type, status, location_name as location, latitude, longitude, created_at, updated_at
            FROM alerts 
            WHERE user_id = ? 
            ORDER BY created_at DESC 
            LIMIT ? OFFSET ?
        ");
        $stmt->execute([$userId, $limit, $offset]);
        $alerts = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Contar total
        $stmt2 = $pdo->prepare("SELECT COUNT(*) FROM alerts WHERE user_id = ?");
        $stmt2->execute([$userId]);
        $total = $stmt2->fetchColumn();

        echo json_encode([
            'success' => true,
            'data' => $alerts,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'pages' => ceil($total / $limit)
            ]
        ]);
        break;

    case 'detail':
        // Obtener detalles de una alerta
        $alertId = $_GET['id'] ?? 0;

        $stmt = $pdo->prepare("
            SELECT a.*, u.full_name as user_name, u.phone as user_phone
            FROM alerts a
            LEFT JOIN users u ON a.user_id = u.id
            WHERE a.id = ? AND a.user_id = ?
        ");
        $stmt->execute([$alertId, $userId]);
        $alert = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($alert) {
            echo json_encode(['success' => true, 'data' => $alert]);
        } else {
            echo json_encode(['success' => false, 'error' => 'Alerta no encontrada']);
        }
        break;

    case 'create_panic':
        // Crear alerta de pánico
        $latitude = $_POST['latitude'] ?? null;
        $longitude = $_POST['longitude'] ?? null;
        $location = "Ubicación GPS: $latitude, $longitude";

        // Prevenir duplicados
        $stmt = $pdo->prepare("
            SELECT id, created_at FROM alerts
            WHERE user_id = ?
            AND alert_type = 'panic'
            AND created_at >= DATE_SUB(NOW(), INTERVAL 10 SECOND)
            ORDER BY created_at DESC
            LIMIT 1
        ");
        $stmt->execute([$userId]);
        if ($existing = $stmt->fetch(PDO::FETCH_ASSOC)) {
            error_log("⚠️ ALERTA PANIC DUPLICADA - User: $userId");
            echo json_encode([
                'success' => false,
                'error' => 'Ya enviaste una alerta hace poco. Espera 10 segundos.',
                'duplicate' => true
            ]);
            exit;
        }

        $stmt = $pdo->prepare("
            INSERT INTO alerts (user_id, alert_type, status, location_name, latitude, longitude, priority, created_at)
            VALUES (?, 'panic', 'pending', ?, ?, ?, 'high', NOW())
        ");

        if ($stmt->execute([$userId, $location, $latitude, $longitude])) {
            $alertId = $pdo->lastInsertId();
            error_log("✅ ALERTA PANIC - ID: $alertId, User: $userId");

            echo json_encode([
                'success' => true,
                'message' => 'Alerta de pánico enviada',
                'alert_id' => $alertId
            ]);
        } else {
            echo json_encode(['success' => false, 'error' => 'Error al crear alerta']);
        }
        break;

    case 'create':
        // Crear alerta personalizada
        $alertType = $_POST['alert_type'] ?? 'other';
        $description = $_POST['description'] ?? '';
        $latitude = $_POST['latitude'] ?? null;
        $longitude = $_POST['longitude'] ?? null;
        $location = $_POST['location'] ?? '';

        if (empty($description)) {
            echo json_encode(['success' => false, 'error' => 'Descripción requerida']);
            exit;
        }

        // Prevenir duplicados
        $stmt = $pdo->prepare("
            SELECT id FROM alerts
            WHERE user_id = ?
            AND alert_type = ?
            AND created_at >= DATE_SUB(NOW(), INTERVAL 10 SECOND)
            LIMIT 1
        ");
        $stmt->execute([$userId, $alertType]);
        if ($stmt->fetch()) {
            echo json_encode([
                'success' => false,
                'error' => 'Ya enviaste una alerta hace poco. Espera 10 segundos.',
                'duplicate' => true
            ]);
            exit;
        }

        $stmt = $pdo->prepare("
            INSERT INTO alerts (user_id, alert_type, status, location_name, latitude, longitude, description, created_at)
            VALUES (?, ?, 'pending', ?, ?, ?, ?, NOW())
        ");

        if ($stmt->execute([$userId, $alertType, $location, $latitude, $longitude, $description])) {
            $alertId = $pdo->lastInsertId();
            error_log("✅ ALERTA - ID: $alertId, Tipo: $alertType, User: $userId");

            echo json_encode([
                'success' => true,
                'message' => 'Alerta creada exitosamente',
                'alert_id' => $alertId
            ]);
        } else {
            echo json_encode(['success' => false, 'error' => 'Error al crear alerta']);
        }
        break;

    case 'cancel':
        // Cancelar alerta pendiente
        $alertId = $_POST['id'] ?? 0;

        $stmt = $pdo->prepare("
            UPDATE alerts 
            SET status = 'cancelled', updated_at = NOW()
            WHERE id = ? AND user_id = ? AND status = 'pending'
        ");
        
        if ($stmt->execute([$alertId, $userId])) {
            if ($stmt->rowCount() > 0) {
                echo json_encode(['success' => true, 'message' => 'Alerta cancelada']);
            } else {
                echo json_encode(['success' => false, 'error' => 'No se puede cancelar esta alerta']);
            }
        } else {
            echo json_encode(['success' => false, 'error' => 'Error al cancelar']);
        }
        break;

    default:
        echo json_encode(['success' => false, 'error' => 'Acción no válida']);
}
?>
