<?php
session_start();

// Check if user is logged in
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'citizen') {
    header('Location: index.php');
    exit;
}

require_once 'config.php';

try {
    $pdo = new PDO(
        "mysql:host=".DB_HOST.";dbname=".DB_NAME.";charset=utf8mb4",
        DB_USER,
        DB_PASS,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );

    // Get user info
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    // Get alert counts
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM alerts WHERE user_id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $totalAlerts = $stmt->fetchColumn();

    $stmt = $pdo->prepare("SELECT COUNT(*) FROM alerts WHERE user_id = ? AND status = 'pending'");
    $stmt->execute([$_SESSION['user_id']]);
    $pendingAlerts = $stmt->fetchColumn();

    $stmt = $pdo->prepare("SELECT COUNT(*) FROM alerts WHERE user_id = ? AND status = 'resolved'");
    $stmt->execute([$_SESSION['user_id']]);
    $resolvedAlerts = $stmt->fetchColumn();

} catch(PDOException $e) {
    die("Error de conexión");
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="theme-color" content="#1a1a2e">
    <title>Citizen Panel - Azuero Safety</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            -webkit-tap-highlight-color: transparent;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #0f0f1e;
            padding-bottom: 80px;
        }

        /* Header */
        .header {
            background: linear-gradient(135deg, #0f3460, #16213e);
            color: white;
            padding: 20px;
            position: sticky;
            top: 0;
            z-index: 100;
            box-shadow: 0 2px 10px rgba(0,0,0,0.3);
        }

        .header-top {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
        }

        .user-info {
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .avatar {
            width: 45px;
            height: 45px;
            background: rgba(255,255,255,0.3);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 20px;
        }

        .user-details h2 {
            font-size: 18px;
            font-weight: 600;
        }

        .user-details p {
            font-size: 13px;
            opacity: 0.9;
        }

        .btn-logout {
            background: rgba(255,255,255,0.2);
            border: 1px solid rgba(255,255,255,0.3);
            color: white;
            padding: 8px 15px;
            border-radius: 8px;
            font-size: 14px;
            cursor: pointer;
        }

        .stats-quick {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 10px;
        }

        .stat-quick {
            background: rgba(255,255,255,0.2);
            backdrop-filter: blur(10px);
            padding: 12px;
            border-radius: 12px;
            text-align: center;
        }

        .stat-quick-value {
            font-size: 24px;
            font-weight: 700;
            display: block;
        }

        .stat-quick-label {
            font-size: 11px;
            opacity: 0.9;
            margin-top: 4px;
        }

        /* Main Content */
        .container {
            padding: 20px;
        }

        /* Panic Button */
        .panic-section {
            text-align: center;
            margin-bottom: 30px;
        }

        .panic-button {
            width: 200px;
            height: 200px;
            margin: 0 auto;
            background: linear-gradient(135deg, #ff4757, #d63031);
            border: 8px solid white;
            border-radius: 50%;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 24px;
            font-weight: 700;
            box-shadow: 0 10px 40px rgba(214, 48, 49, 0.5);
            cursor: pointer;
            transition: all 0.3s;
            position: relative;
            overflow: hidden;
        }

        .panic-button::before {
            content: '';
            position: absolute;
            width: 100%;
            height: 100%;
            background: radial-gradient(circle, rgba(255,255,255,0.3) 0%, transparent 70%);
            animation: pulse 2s infinite;
        }

        @keyframes pulse {
            0%, 100% { transform: scale(1); opacity: 1; }
            50% { transform: scale(1.1); opacity: 0.5; }
        }

        .panic-button:active {
            transform: scale(0.95);
        }

        .panic-button i {
            font-size: 60px;
            margin-bottom: 10px;
        }

        .panic-info {
            margin-top: 15px;
            font-size: 14px;
            color: #666;
        }

        /* Quick Actions */
        .section-title {
            font-size: 18px;
            font-weight: 700;
            color: #333;
            margin-bottom: 15px;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .quick-actions {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 12px;
            margin-bottom: 30px;
        }

        .action-card {
            background: #1a1a2e;
            padding: 20px;
            border-radius: 16px;
            text-align: center;
            box-shadow: 0 2px 8px rgba(0,0,0,0.3);
            cursor: pointer;
            transition: all 0.3s;
            border: 1px solid #2d2d44;
            color: #fff;
        }

        .action-card:active {
            transform: scale(0.98);
        }

        .action-icon {
            width: 60px;
            height: 60px;
            margin: 0 auto 12px;
            background: linear-gradient(135deg, #667eea, #764ba2);
            border-radius: 16px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 28px;
        }

        .action-card.secondary .action-icon {
            background: linear-gradient(135deg, #f093fb, #f5576c);
        }

        .action-card.success .action-icon {
            background: linear-gradient(135deg, #4facfe, #00f2fe);
        }

        .action-card.warning .action-icon {
            background: linear-gradient(135deg, #fa709a, #fee140);
        }

        .action-title {
            font-size: 15px;
            font-weight: 600;
            color: #333;
        }

        .action-subtitle {
            font-size: 12px;
            color: #999;
            margin-top: 4px;
        }

        /* Recent Alerts */
        .alerts-list {
            background: white;
            border-radius: 16px;
            overflow: hidden;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }

        .alert-item {
            padding: 16px;
            border-bottom: 1px solid #f0f0f0;
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .alert-item:last-child {
            border-bottom: none;
        }

        .alert-icon {
            width: 50px;
            height: 50px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            flex-shrink: 0;
        }

        .alert-icon.pending {
            background: #fff3cd;
            color: #ffc107;
        }

        .alert-icon.in_progress {
            background: #d1ecf1;
            color: #0dcaf0;
        }

        .alert-icon.resolved {
            background: #d1e7dd;
            color: #198754;
        }

        .alert-details {
            flex: 1;
        }

        .alert-type {
            font-size: 15px;
            font-weight: 600;
            color: #333;
            text-transform: capitalize;
        }

        .alert-date {
            font-size: 12px;
            color: #999;
            margin-top: 4px;
        }

        .alert-status {
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
        }

        .alert-status.pending {
            background: #fff3cd;
            color: #856404;
        }

        .alert-status.in_progress {
            background: #d1ecf1;
            color: #055160;
        }

        .alert-status.resolved {
            background: #d1e7dd;
            color: #0f5132;
        }

        .empty-state {
            text-align: center;
            padding: 40px 20px;
            color: #999;
        }

        .empty-state i {
            font-size: 48px;
            margin-bottom: 15px;
            opacity: 0.5;
        }

        /* Bottom Navigation */
        .bottom-nav {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: white;
            padding: 10px 0;
            box-shadow: 0 -2px 10px rgba(0,0,0,0.1);
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            z-index: 100;
        }

        .nav-item {
            text-align: center;
            padding: 8px;
            color: #999;
            text-decoration: none;
            transition: all 0.3s;
        }

        .nav-item.active {
            color: #667eea;
        }

        .nav-item i {
            display: block;
            font-size: 24px;
            margin-bottom: 4px;
        }

        .nav-item span {
            font-size: 11px;
            font-weight: 600;
        }

        /* Modal */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.7);
            z-index: 1000;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }

        .modal.show {
            display: flex;
        }

        .modal-content {
            background: white;
            border-radius: 20px;
            padding: 30px;
            max-width: 400px;
            width: 100%;
            text-align: center;
            animation: slideUp 0.3s ease;
        }

        .modal-icon {
            width: 80px;
            height: 80px;
            margin: 0 auto 20px;
            background: linear-gradient(135deg, #ff4757, #d63031);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 40px;
        }

        .modal-title {
            font-size: 22px;
            font-weight: 700;
            margin-bottom: 10px;
        }

        .modal-text {
            color: #666;
            margin-bottom: 25px;
            line-height: 1.6;
        }

        .modal-buttons {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 12px;
        }

        .btn-modal {
            padding: 14px;
            border: none;
            border-radius: 12px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
        }

        .btn-confirm {
            background: linear-gradient(135deg, #ff4757, #d63031);
            color: white;
        }

        .btn-cancel {
            background: #2d2d44;
            color: #fff;
        }

        /* Global Dark Theme Overrides */
        .section-title, h3, h4, p {
            color: #fff !important;
        }
        
        .card, .recent-alerts, .alert-item {
            background: #1a1a2e !important;
            border: 1px solid #2d2d44 !important;
            color: #fff !important;
        }

        .bottom-nav {
            background: #16213e !important;
            border-top: 1px solid #2d2d44 !important;
        }

        .nav-item {
            color: rgba(255,255,255,0.6) !important;
        }

        .nav-item.active {
            color: #4facfe !important;
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="header">
        <div class="header-top">
            <div class="user-info">
                <div class="avatar">
                    <i class="fas fa-user"></i>
                </div>
                <div class="user-details">
                    <h2><?php echo htmlspecialchars($user['full_name']); ?></h2>
                    <p>Verified Citizen</p>
                </div>
            </div>
            <button class="btn-logout" onclick="logout()">
                <i class="fas fa-sign-out-alt"></i>
            </button>
        </div>

        <div class="stats-quick">
            <div class="stat-quick">
                <span class="stat-quick-value"><?php echo $totalAlerts; ?></span>
                <span class="stat-quick-label">Total</span>
            </div>
            <div class="stat-quick">
                <span class="stat-quick-value"><?php echo $pendingAlerts; ?></span>
                <span class="stat-quick-label">Activas</span>
            </div>
            <div class="stat-quick">
                <span class="stat-quick-value"><?php echo $resolvedAlerts; ?></span>
                <span class="stat-quick-label">Resueltas</span>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="container">
        <!-- Panic Button -->
        <div class="panic-section">
            <div class="panic-button" onclick="showPanicModal()">
                <i class="fas fa-exclamation-triangle"></i>
                <span>SOS</span>
                <small style="font-size: 12px; font-weight: 400;">Presiona para emergencia</small>
            </div>
            <p class="panic-info">
                <i class="fas fa-shield-alt"></i> Tu ubicación será enviada automáticamente
            </p>
        </div>

        <!-- Quick Actions -->
        <h3 class="section-title">
            <i class="fas fa-bolt"></i> Acciones Rápidas
        </h3>
        <div class="quick-actions">
            <div class="action-card" onclick="location.href='my-alerts.php'">
                <div class="action-icon">
                    <i class="fas fa-bell"></i>
                </div>
                <div class="action-title">Mis Alertas</div>
                <div class="action-subtitle"><?php echo $totalAlerts; ?> alertas</div>
            </div>

            <div class="action-card secondary" onclick="location.href='profile.php'">
                <div class="action-icon">
                    <i class="fas fa-user-edit"></i>
                </div>
                <div class="action-title">Mi Perfil</div>
                <div class="action-subtitle">Editar datos</div>
            </div>

            <div class="action-card success" onclick="location.href='emergency-contacts.php'">
                <div class="action-icon">
                    <i class="fas fa-phone"></i>
                </div>
                <div class="action-title">Contactos</div>
                <div class="action-subtitle">Emergencia</div>
            </div>

            <div class="action-card warning" onclick="location.href='settings.php'">
                <div class="action-icon">
                    <i class="fas fa-cog"></i>
                </div>
                <div class="action-title">Configuración</div>
                <div class="action-subtitle">Ajustes</div>
            </div>
        </div>

        <!-- Recent Alerts -->
        <h3 class="section-title">
            <i class="fas fa-history"></i> Alertas Recientes
        </h3>
        <div class="alerts-list" id="alertsList">
            <div class="empty-state">
                <i class="fas fa-clipboard-list"></i>
                <p>Cargando alertas...</p>
            </div>
        </div>
    </div>

    <!-- Bottom Navigation -->
    <div class="bottom-nav">
        <a href="dashboard.php" class="nav-item active">
            <i class="fas fa-home"></i>
            <span>Inicio</span>
        </a>
        <a href="my-alerts.php" class="nav-item">
            <i class="fas fa-bell"></i>
            <span>Alertas</span>
        </a>
        <a href="profile.php" class="nav-item">
            <i class="fas fa-user"></i>
            <span>Perfil</span>
        </a>
        <a href="settings.php" class="nav-item">
            <i class="fas fa-cog"></i>
            <span>Ajustes</span>
        </a>
    </div>

    <!-- Panic Modal -->
    <div class="modal" id="panicModal">
        <div class="modal-content">
            <div class="modal-icon">
                <i class="fas fa-exclamation-triangle"></i>
            </div>
            <h3 class="modal-title">¿Enviar Alerta de Emergencia?</h3>
            <p class="modal-text">
                Se notificará inmediatamente al centro de operaciones. 
                Tu ubicación actual será compartida.
            </p>
            <div class="modal-buttons">
                <button class="btn-modal btn-cancel" onclick="closePanicModal()">Cancelar</button>
                <button class="btn-modal btn-confirm" onclick="sendPanicAlert()">Confirmar SOS</button>
            </div>
        </div>
    </div>

    <script>
        // Load recent alerts
        async function loadRecentAlerts() {
            try {
                const response = await fetch('api-alerts.php?action=recent');
                const result = await response.json();

                const alertsList = document.getElementById('alertsList');

                if (result.success && result.data.length > 0) {
                    alertsList.innerHTML = result.data.map(alert => `
                        <div class="alert-item" onclick="viewAlert(${alert.id})">
                            <div class="alert-icon ${alert.status}">
                                <i class="fas fa-${getAlertIcon(alert.alert_type)}"></i>
                            </div>
                            <div class="alert-details">
                                <div class="alert-type">${alert.alert_type}</div>
                                <div class="alert-date">${formatDate(alert.created_at)}</div>
                            </div>
                            <span class="alert-status ${alert.status}">${getStatusText(alert.status)}</span>
                        </div>
                    `).join('');
                } else {
                    alertsList.innerHTML = `
                        <div class="empty-state">
                            <i class="fas fa-clipboard-list"></i>
                            <p>No hay alertas registradas</p>
                        </div>
                    `;
                }
            } catch (error) {
                console.error('Error loading alerts:', error);
            }
        }

        function getAlertIcon(type) {
            const icons = {
                'medical': 'ambulance',
                'security': 'shield-alt',
                'fire': 'fire',
                'other': 'exclamation-circle'
            };
            return icons[type] || 'bell';
        }

        function getStatusText(status) {
            const texts = {
                'pending': 'Pendiente',
                'in_progress': 'En proceso',
                'resolved': 'Resuelta'
            };
            return texts[status] || status;
        }

        function formatDate(dateString) {
            const date = new Date(dateString);
            const now = new Date();
            const diff = Math.floor((now - date) / 1000);

            if (diff < 60) return 'Hace unos segundos';
            if (diff < 3600) return `Hace ${Math.floor(diff / 60)} minutos`;
            if (diff < 86400) return `Hace ${Math.floor(diff / 3600)} horas`;
            return date.toLocaleDateString('es-ES');
        }

        function viewAlert(id) {
            window.location.href = `alert-detail.php?id=${id}`;
        }

        // Panic button
        function showPanicModal() {
            document.getElementById('panicModal').classList.add('show');
        }

        function closePanicModal() {
            document.getElementById('panicModal').classList.remove('show');
        }

        // Bandera para prevenir envíos duplicados
        let isSendingPanic = false;

        async function sendPanicAlert() {
            // Prevenir doble envío
            if (isSendingPanic) {
                console.log('❌ Ya hay un envío en progreso, ignorando clic');
                return;
            }

            isSendingPanic = true;
            console.log('🚨 Enviando alerta de pánico...');

            // Get geolocation
            if (navigator.geolocation) {
                navigator.geolocation.getCurrentPosition(async (position) => {
                    try {
                        const formData = new FormData();
                        formData.append('action', 'create_panic');
                        formData.append('latitude', position.coords.latitude);
                        formData.append('longitude', position.coords.longitude);

                        const response = await fetch('api-alerts.php', {
                            method: 'POST',
                            body: formData
                        });

                        const result = await response.json();
                        console.log('✅ Respuesta del servidor:', result);

                        if (result.success) {
                            closePanicModal();
                            alert('✓ Alerta de emergencia enviada. Ayuda en camino.');
                            loadRecentAlerts();

                            // Esperar 10 segundos antes de permitir otro envío
                            setTimeout(() => {
                                isSendingPanic = false;
                                console.log('✅ Cooldown completado, se puede enviar otra alerta');
                            }, 10000);
                        } else {
                            // Liberar inmediatamente si hay error
                            isSendingPanic = false;

                            // Mensaje especial para alertas duplicadas
                            if (result.duplicate) {
                                alert('⚠️ Ya enviaste una alerta hace poco. Por favor espera 10 segundos entre alertas.');
                            } else {
                                alert('Error al enviar alerta: ' + (result.error || 'Error desconocido'));
                            }
                        }
                    } catch (error) {
                        alert('Error de conexión');
                        isSendingPanic = false; // Liberar en caso de error
                    }
                }, () => {
                    alert('No se pudo obtener tu ubicación. Activa el GPS.');
                    isSendingPanic = false; // Liberar en caso de error
                });
            } else {
                alert('Tu dispositivo no soporta geolocalización');
                isSendingPanic = false; // Liberar en caso de error
            }
        }

        // Logout
        function logout() {
            if (confirm('¿Cerrar sesión?')) {
                window.location.href = 'logout.php';
            }
        }

        // Load data on page load
        window.addEventListener('load', () => {
            loadRecentAlerts();
            
            // Auto refresh every 30 seconds
            setInterval(loadRecentAlerts, 30000);

            // Solicitar permisos de notificación
            requestNotificationPermission();
        });

        // Sistema de Notificaciones Push
        async function requestNotificationPermission() {
            if (!('Notification' in window)) {
                console.log('Este navegador no soporta notificaciones');
                return;
            }

            if (Notification.permission === 'granted') {
                console.log('Notificaciones ya autorizadas');
                subscribeToNotifications();
                return;
            }

            if (Notification.permission !== 'denied') {
                const permission = await Notification.requestPermission();
                if (permission === 'granted') {
                    console.log('Permiso de notificación concedido');
                    subscribeToNotifications();
                    showLocalNotification('Notificaciones activadas', 'Recibirás alertas en tiempo real');
                }
            }
        }

        async function subscribeToNotifications() {
            try {
                const registration = await navigator.serviceWorker.ready;
                
                // Verificar si ya hay suscripción
                let subscription = await registration.pushManager.getSubscription();
                
                if (!subscription) {
                    // Crear nueva suscripción (requiere VAPID keys en producción)
                    subscription = await registration.pushManager.subscribe({
                        userVisibleOnly: true,
                        applicationServerKey: urlBase64ToUint8Array('YOUR_PUBLIC_VAPID_KEY_HERE')
                    });
                }

                // Enviar suscripción al servidor
                await fetch('api-notifications.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({
                        action: 'subscribe',
                        subscription: subscription
                    })
                });

                console.log('Suscripción a push notifications exitosa');
            } catch (error) {
                console.error('Error al suscribirse a notificaciones:', error);
            }
        }

        function showLocalNotification(title, body) {
            if (Notification.permission === 'granted') {
                new Notification(title, {
                    body: body,
                    icon: 'icons/icon-192x192.png',
                    badge: 'icons/icon-72x72.png',
                    vibrate: [200, 100, 200],
                    tag: 'azuero-safety',
                    requireInteraction: false
                });
            }
        }

        function urlBase64ToUint8Array(base64String) {
            const padding = '='.repeat((4 - base64String.length % 4) % 4);
            const base64 = (base64String + padding).replace(/\-/g, '+').replace(/_/g, '/');
            const rawData = window.atob(base64);
            const outputArray = new Uint8Array(rawData.length);
            for (let i = 0; i < rawData.length; ++i) {
                outputArray[i] = rawData.charCodeAt(i);
            }
            return outputArray;
        }

        // Verificar nuevas alertas periódicamente y notificar
        let lastAlertCount = 0;
        setInterval(async () => {
            try {
                const response = await fetch('api-alerts.php?action=recent');
                const data = await response.json();
                
                if (data.success && data.data.length > lastAlertCount && lastAlertCount > 0) {
                    const newAlerts = data.data.length - lastAlertCount;
                    showLocalNotification(
                        'Nuevas Alertas',
                        `Tienes ${newAlerts} alerta(s) nueva(s)`
                    );
                }
                
                lastAlertCount = data.data.length;
            } catch (error) {
                console.error('Error verificando alertas:', error);
            }
        }, 60000); // Cada minuto

        // PWA Features
        if ('serviceWorker' in navigator) {
            navigator.serviceWorker.register('service-worker.js')
                .then(registration => {
                    console.log('Service Worker registrado:', registration);
                })
                .catch(error => {
                    console.error('Error registrando Service Worker:', error);
                });
        }
    </script>
</body>
</html>
