<?php
session_start();

if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'citizen') {
    header('Location: index.php');
    exit;
}

require_once 'config.php';

try {
    $pdo = new PDO(
        "mysql:host=".DB_HOST.";dbname=".DB_NAME.";charset=utf8mb4",
        DB_USER,
        DB_PASS,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );

    $stmt = $pdo->prepare("SELECT full_name FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
} catch(PDOException $e) {
    die("Error de conexión");
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="theme-color" content="#667eea">
    <title>Mis Alertas - Azuero Safety</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            -webkit-tap-highlight-color: transparent;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #f5f7fa;
            padding-bottom: 70px;
        }

        .header {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            padding: 15px 20px;
            position: sticky;
            top: 0;
            z-index: 100;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }

        .header-content {
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .btn-back {
            background: rgba(255,255,255,0.2);
            border: none;
            color: white;
            width: 38px;
            height: 38px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 18px;
            cursor: pointer;
        }

        .header-title {
            flex: 1;
        }

        .header-title h1 {
            font-size: 20px;
            font-weight: 700;
        }

        .header-title p {
            font-size: 12px;
            opacity: 0.9;
            margin-top: 2px;
        }

        .container {
            padding: 15px;
        }

        .filter-tabs {
            background: white;
            border-radius: 12px;
            padding: 6px;
            display: flex;
            gap: 6px;
            margin-bottom: 15px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
        }

        .filter-tab {
            flex: 1;
            padding: 10px;
            border: none;
            background: transparent;
            border-radius: 8px;
            font-size: 13px;
            font-weight: 600;
            color: #666;
            cursor: pointer;
            transition: all 0.3s;
        }

        .filter-tab.active {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
        }

        .alerts-list {
            display: flex;
            flex-direction: column;
            gap: 12px;
        }

        .alert-card {
            background: white;
            border-radius: 14px;
            padding: 15px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
            cursor: pointer;
            transition: all 0.3s;
            position: relative;
        }

        .alert-card:active {
            transform: scale(0.98);
        }

        .alert-header {
            display: flex;
            align-items: flex-start;
            gap: 12px;
            margin-bottom: 10px;
        }

        .alert-icon {
            width: 45px;
            height: 45px;
            border-radius: 11px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 20px;
            flex-shrink: 0;
        }

        .alert-icon.panic {
            background: #fee;
            color: #d63031;
        }

        .alert-icon.medical {
            background: #fff3cd;
            color: #ffc107;
        }

        .alert-icon.security {
            background: #d1ecf1;
            color: #0dcaf0;
        }

        .alert-icon.fire {
            background: #f8d7da;
            color: #dc3545;
        }

        .alert-details {
            flex: 1;
        }

        .alert-type {
            font-size: 15px;
            font-weight: 600;
            color: #333;
            text-transform: capitalize;
            margin-bottom: 4px;
        }

        .alert-date {
            font-size: 12px;
            color: #999;
        }

        .alert-status {
            position: absolute;
            top: 15px;
            right: 15px;
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
        }

        .alert-status.pending {
            background: #fff3cd;
            color: #856404;
        }

        .alert-status.in_progress {
            background: #d1ecf1;
            color: #055160;
        }

        .alert-status.resolved {
            background: #d1e7dd;
            color: #0f5132;
        }

        .alert-status.cancelled {
            background: #f8d7da;
            color: #842029;
        }

        .alert-location {
            display: flex;
            align-items: center;
            gap: 6px;
            font-size: 12px;
            color: #666;
            margin-top: 8px;
        }

        .alert-location i {
            font-size: 11px;
        }

        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #999;
        }

        .empty-state i {
            font-size: 60px;
            margin-bottom: 15px;
            opacity: 0.3;
        }

        .empty-state h3 {
            font-size: 18px;
            margin-bottom: 8px;
            color: #666;
        }

        .empty-state p {
            font-size: 14px;
        }

        .loading {
            text-align: center;
            padding: 40px;
            color: #667eea;
        }

        .loading i {
            font-size: 40px;
            animation: spin 1s linear infinite;
        }

        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }

        .bottom-nav {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: white;
            padding: 8px 0;
            box-shadow: 0 -2px 10px rgba(0,0,0,0.1);
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            z-index: 100;
        }

        .nav-item {
            text-align: center;
            padding: 8px;
            color: #999;
            text-decoration: none;
            transition: all 0.3s;
        }

        .nav-item.active {
            color: #667eea;
        }

        .nav-item i {
            display: block;
            font-size: 22px;
            margin-bottom: 4px;
        }

        .nav-item span {
            font-size: 11px;
            font-weight: 600;
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="header-content">
            <button class="btn-back" onclick="location.href='dashboard.php'">
                <i class="fas fa-arrow-left"></i>
            </button>
            <div class="header-title">
                <h1>Mis Alertas</h1>
                <p>Historial completo</p>
            </div>
        </div>
    </div>

    <div class="container">
        <div class="filter-tabs">
            <button class="filter-tab active" data-filter="all">Todas</button>
            <button class="filter-tab" data-filter="pending">Activas</button>
            <button class="filter-tab" data-filter="resolved">Resueltas</button>
        </div>

        <div id="alertsList" class="alerts-list">
            <div class="loading">
                <i class="fas fa-spinner fa-spin"></i>
                <p>Cargando alertas...</p>
            </div>
        </div>
    </div>

    <div class="bottom-nav">
        <a href="dashboard.php" class="nav-item">
            <i class="fas fa-home"></i>
            <span>Inicio</span>
        </a>
        <a href="my-alerts.php" class="nav-item active">
            <i class="fas fa-bell"></i>
            <span>Alertas</span>
        </a>
        <a href="profile.php" class="nav-item">
            <i class="fas fa-user"></i>
            <span>Perfil</span>
        </a>
        <a href="settings.php" class="nav-item">
            <i class="fas fa-cog"></i>
            <span>Ajustes</span>
        </a>
    </div>

    <script>
        let currentFilter = 'all';

        // Filter tabs
        document.querySelectorAll('.filter-tab').forEach(tab => {
            tab.addEventListener('click', function() {
                document.querySelectorAll('.filter-tab').forEach(t => t.classList.remove('active'));
                this.classList.add('active');
                currentFilter = this.dataset.filter;
                loadAlerts();
            });
        });

        async function loadAlerts() {
            const alertsList = document.getElementById('alertsList');
            alertsList.innerHTML = '<div class="loading"><i class="fas fa-spinner fa-spin"></i><p>Cargando...</p></div>';

            try {
                const url = currentFilter === 'all' 
                    ? 'api-alerts.php?action=all&page=1' 
                    : `api-alerts.php?action=all&page=1&status=${currentFilter}`;
                
                const response = await fetch(url);
                const result = await response.json();

                if (result.success && result.data.length > 0) {
                    alertsList.innerHTML = result.data.map(alert => `
                        <div class="alert-card" onclick="viewAlert(${alert.id})">
                            <div class="alert-header">
                                <div class="alert-icon ${alert.alert_type}">
                                    <i class="fas fa-${getAlertIcon(alert.alert_type)}"></i>
                                </div>
                                <div class="alert-details">
                                    <div class="alert-type">${getAlertTypeName(alert.alert_type)}</div>
                                    <div class="alert-date">${formatDate(alert.created_at)}</div>
                                </div>
                            </div>
                            <span class="alert-status ${alert.status}">${getStatusText(alert.status)}</span>
                            ${alert.location ? `
                                <div class="alert-location">
                                    <i class="fas fa-map-marker-alt"></i>
                                    <span>${alert.location.substring(0, 50)}${alert.location.length > 50 ? '...' : ''}</span>
                                </div>
                            ` : ''}
                        </div>
                    `).join('');
                } else {
                    alertsList.innerHTML = `
                        <div class="empty-state">
                            <i class="fas fa-clipboard-list"></i>
                            <h3>No hay alertas</h3>
                            <p>No se encontraron alertas ${currentFilter === 'all' ? '' : 'con este filtro'}</p>
                        </div>
                    `;
                }
            } catch (error) {
                alertsList.innerHTML = `
                    <div class="empty-state">
                        <i class="fas fa-exclamation-circle"></i>
                        <h3>Error al cargar</h3>
                        <p>No se pudieron cargar las alertas</p>
                    </div>
                `;
            }
        }

        function getAlertIcon(type) {
            const icons = {
                'panic': 'exclamation-triangle',
                'medical': 'ambulance',
                'security': 'shield-alt',
                'fire': 'fire',
                'other': 'bell'
            };
            return icons[type] || 'bell';
        }

        function getAlertTypeName(type) {
            const names = {
                'panic': 'Alerta de Pánico',
                'medical': 'Emergencia Médica',
                'security': 'Seguridad',
                'fire': 'Incendio',
                'other': 'Otra Emergencia'
            };
            return names[type] || type;
        }

        function getStatusText(status) {
            const texts = {
                'pending': 'Pendiente',
                'in_progress': 'En Proceso',
                'resolved': 'Resuelta',
                'cancelled': 'Cancelada'
            };
            return texts[status] || status;
        }

        function formatDate(dateString) {
            const date = new Date(dateString);
            const now = new Date();
            const diff = Math.floor((now - date) / 1000);

            if (diff < 60) return 'Hace unos segundos';
            if (diff < 3600) return `Hace ${Math.floor(diff / 60)} minutos`;
            if (diff < 86400) return `Hace ${Math.floor(diff / 3600)} horas`;
            if (diff < 172800) return 'Ayer';
            return date.toLocaleDateString('es-ES', { day: '2-digit', month: 'short', year: 'numeric' });
        }

        function viewAlert(id) {
            window.location.href = `alert-detail.php?id=${id}`;
        }

        // Load alerts on page load
        window.addEventListener('load', loadAlerts);
    </script>
</body>
</html>
