<?php
session_start();

// Si ya hay sesión activa, redirigir
if (isset($_SESSION['user_id'])) {
    header('Location: dashboard.php');
    exit;
}

require_once 'config.php';

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $full_name = trim($_POST['full_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $cedula = trim($_POST['cedula'] ?? '');
    $password = $_POST['password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    $emergency_contact_name = trim($_POST['emergency_contact_name'] ?? '');
    $emergency_contact_phone = trim($_POST['emergency_contact_phone'] ?? '');
    $terms = isset($_POST['terms']);

    // Validaciones
    if (empty($full_name) || empty($email) || empty($password)) {
        $error = 'Por favor completa todos los campos obligatorios';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Email inválido';
    } elseif (strlen($password) < 6) {
        $error = 'La contraseña debe tener al menos 6 caracteres';
    } elseif ($password !== $confirm_password) {
        $error = 'Las contraseñas no coinciden';
    } elseif (!$terms) {
        $error = 'Debes aceptar los términos y condiciones';
    } else {
        try {
            $pdo = new PDO(
                "mysql:host=".DB_HOST.";dbname=".DB_NAME.";charset=utf8mb4",
                DB_USER,
                DB_PASS,
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );

            // Verificar si el email ya existe
            $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
            $stmt->execute([$email]);
            
            if ($stmt->fetch()) {
                $error = 'Este email ya está registrado';
            } else {
                // Crear usuario
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                
                $stmt = $pdo->prepare("
                    INSERT INTO users (
                        full_name, email, phone, cedula, password,
                        emergency_contact_name, emergency_contact_phone,
                        created_at
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
                ");
                
                if ($stmt->execute([
                    $full_name, $email, $phone, $cedula, $hashed_password,
                    $emergency_contact_name, $emergency_contact_phone
                ])) {
                    $success = 'Cuenta creada exitosamente. Redirigiendo...';
                    header('refresh:2;url=index.php');
                } else {
                    $error = 'Error al crear la cuenta';
                }
            }
        } catch(PDOException $e) {
            $error = 'Error de conexión a la base de datos';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="theme-color" content="#667eea">
    <title>Registro - Azuero Safety</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            -webkit-tap-highlight-color: transparent;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .register-container {
            width: 100%;
            max-width: 420px;
            background: white;
            border-radius: 20px;
            overflow: hidden;
            box-shadow: 0 10px 40px rgba(0,0,0,0.15);
        }

        .register-header {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            padding: 30px 25px 25px;
            text-align: center;
        }

        .logo {
            width: 70px;
            height: 70px;
            margin: 0 auto 15px;
            background: rgba(255,255,255,0.2);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 35px;
        }

        .register-header h1 {
            font-size: 24px;
            font-weight: 700;
            margin-bottom: 5px;
        }

        .register-header p {
            font-size: 13px;
            opacity: 0.9;
        }

        .register-body {
            padding: 25px 25px 30px;
        }

        .form-group {
            margin-bottom: 15px;
        }

        .form-label {
            display: block;
            font-size: 13px;
            font-weight: 600;
            color: #555;
            margin-bottom: 6px;
        }

        .required {
            color: #dc3545;
        }

        .input-wrapper {
            position: relative;
        }

        .input-icon {
            position: absolute;
            left: 14px;
            top: 50%;
            transform: translateY(-50%);
            color: #999;
            font-size: 16px;
        }

        .form-input {
            width: 100%;
            padding: 14px 14px 14px 45px;
            border: 1px solid #e0e0e0;
            border-radius: 10px;
            font-size: 14px;
            font-family: inherit;
            transition: all 0.3s;
        }

        .form-input:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }

        .checkbox-group {
            display: flex;
            align-items: start;
            gap: 10px;
            margin: 15px 0;
        }

        .checkbox-group input[type="checkbox"] {
            width: 18px;
            height: 18px;
            margin-top: 2px;
            cursor: pointer;
        }

        .checkbox-group label {
            font-size: 13px;
            color: #666;
            line-height: 1.5;
            cursor: pointer;
        }

        .btn-register {
            width: 100%;
            padding: 15px;
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            border: none;
            border-radius: 10px;
            font-size: 15px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
            margin-top: 10px;
        }

        .btn-register:active {
            transform: scale(0.98);
        }

        .btn-register:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }

        .divider {
            text-align: center;
            margin: 20px 0;
            position: relative;
        }

        .divider::before {
            content: '';
            position: absolute;
            top: 50%;
            left: 0;
            right: 0;
            height: 1px;
            background: #e0e0e0;
        }

        .divider span {
            position: relative;
            background: white;
            padding: 0 15px;
            color: #999;
            font-size: 13px;
        }

        .login-link {
            text-align: center;
        }

        .btn-login {
            display: inline-block;
            padding: 14px;
            width: 100%;
            background: white;
            color: #667eea;
            border: 1px solid #667eea;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            text-decoration: none;
            transition: all 0.3s;
        }

        .alert {
            padding: 12px;
            border-radius: 10px;
            margin-bottom: 15px;
            font-size: 13px;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .alert-error {
            background: #f8d7da;
            color: #842029;
        }

        .alert-success {
            background: #d1e7dd;
            color: #0f5132;
        }

        .section-divider {
            font-size: 14px;
            font-weight: 700;
            color: #667eea;
            margin: 20px 0 15px;
            padding-bottom: 8px;
            border-bottom: 2px solid #f0f0f0;
        }
    </style>
</head>
<body>
    <div class="register-container">
        <div class="register-header">
            <div class="logo">
                <i class="fas fa-shield-alt"></i>
            </div>
            <h1>Crear Cuenta</h1>
            <p>Únete a Azuero Safety</p>
        </div>

        <div class="register-body">
            <?php if ($error): ?>
                <div class="alert alert-error">
                    <i class="fas fa-exclamation-circle"></i>
                    <?php echo $error; ?>
                </div>
            <?php endif; ?>

            <?php if ($success): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i>
                    <?php echo $success; ?>
                </div>
            <?php endif; ?>

            <form method="POST" action="">
                <div class="section-divider">
                    <i class="fas fa-user"></i> Información Personal
                </div>

                <div class="form-group">
                    <label class="form-label">Nombre Completo <span class="required">*</span></label>
                    <div class="input-wrapper">
                        <i class="fas fa-user input-icon"></i>
                        <input type="text" name="full_name" class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['full_name'] ?? ''); ?>" 
                               placeholder="Juan Pérez" required>
                    </div>
                </div>

                <div class="form-group">
                    <label class="form-label">Email <span class="required">*</span></label>
                    <div class="input-wrapper">
                        <i class="fas fa-envelope input-icon"></i>
                        <input type="email" name="email" class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>" 
                               placeholder="correo@ejemplo.com" required>
                    </div>
                </div>

                <div class="form-group">
                    <label class="form-label">Teléfono</label>
                    <div class="input-wrapper">
                        <i class="fas fa-phone input-icon"></i>
                        <input type="tel" name="phone" class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['phone'] ?? ''); ?>" 
                               placeholder="+507 6000-0000">
                    </div>
                </div>

                <div class="form-group">
                    <label class="form-label">Cédula</label>
                    <div class="input-wrapper">
                        <i class="fas fa-id-card input-icon"></i>
                        <input type="text" name="cedula" class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['cedula'] ?? ''); ?>" 
                               placeholder="8-888-8888">
                    </div>
                </div>

                <div class="section-divider">
                    <i class="fas fa-lock"></i> Seguridad
                </div>

                <div class="form-group">
                    <label class="form-label">Contraseña <span class="required">*</span></label>
                    <div class="input-wrapper">
                        <i class="fas fa-lock input-icon"></i>
                        <input type="password" name="password" class="form-input" 
                               placeholder="Mínimo 6 caracteres" required>
                    </div>
                </div>

                <div class="form-group">
                    <label class="form-label">Confirmar Contraseña <span class="required">*</span></label>
                    <div class="input-wrapper">
                        <i class="fas fa-lock input-icon"></i>
                        <input type="password" name="confirm_password" class="form-input" 
                               placeholder="Repetir contraseña" required>
                    </div>
                </div>

                <div class="section-divider">
                    <i class="fas fa-phone-alt"></i> Contacto de Emergencia
                </div>

                <div class="form-group">
                    <label class="form-label">Nombre del Contacto</label>
                    <div class="input-wrapper">
                        <i class="fas fa-user-friends input-icon"></i>
                        <input type="text" name="emergency_contact_name" class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['emergency_contact_name'] ?? ''); ?>" 
                               placeholder="Familiar o amigo">
                    </div>
                </div>

                <div class="form-group">
                    <label class="form-label">Teléfono del Contacto</label>
                    <div class="input-wrapper">
                        <i class="fas fa-phone input-icon"></i>
                        <input type="tel" name="emergency_contact_phone" class="form-input" 
                               value="<?php echo htmlspecialchars($_POST['emergency_contact_phone'] ?? ''); ?>" 
                               placeholder="+507 6000-0000">
                    </div>
                </div>

                <div class="checkbox-group">
                    <input type="checkbox" name="terms" id="terms" required>
                    <label for="terms">
                        Acepto los <a href="#" style="color: #667eea;">Términos y Condiciones</a> 
                        y la <a href="#" style="color: #667eea;">Política de Privacidad</a>
                    </label>
                </div>

                <button type="submit" class="btn-register">
                    <i class="fas fa-user-plus"></i> Crear Cuenta
                </button>
            </form>

            <div class="divider">
                <span>¿Ya tienes cuenta?</span>
            </div>

            <div class="login-link">
                <a href="index.php" class="btn-login">
                    <i class="fas fa-sign-in-alt"></i> Iniciar Sesión
                </a>
            </div>
        </div>
    </div>
</body>
</html>
