<?php
require_once 'config.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit;
}

// Obtener todas las alertas
$filter = $_GET['filter'] ?? 'all';
$search = $_GET['search'] ?? '';

$where = "WHERE 1=1";
if ($filter != 'all') {
    $where .= " AND a.status = :status";
}
if (!empty($search)) {
    $where .= " AND (a.location_name LIKE :search OR a.alert_type LIKE :search OR u.full_name LIKE :search)";
}

try {
    $sql = "SELECT a.*, u.full_name as citizen_name, o.full_name as operator_name 
            FROM alerts a 
            LEFT JOIN users u ON a.user_id = u.id 
            LEFT JOIN operators o ON a.assigned_operator_id = o.id 
            $where 
            ORDER BY 
                CASE a.status 
                    WHEN 'pending' THEN 1 
                    WHEN 'in_progress' THEN 2 
                    WHEN 'resolved' THEN 3 
                    WHEN 'cancelled' THEN 4 
                END,
                CASE a.priority 
                    WHEN 'high' THEN 1 
                    WHEN 'medium' THEN 2 
                    WHEN 'low' THEN 3 
                END,
                a.created_at DESC";
    
    $stmt = $conn->prepare($sql);
    if ($filter != 'all') {
        $stmt->bindValue(':status', $filter);
    }
    if (!empty($search)) {
        $stmt->bindValue(':search', "%$search%");
    }
    $stmt->execute();
    $alerts = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch(PDOException $e) {
    $alerts = [];
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestión de Alertas - Azuero Safety</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; background: #f5f5f5; }
        
        .header {
            background: #1e3c72;
            color: white;
            padding: 15px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        
        .header h1 { font-size: 24px; }
        .nav { display: flex; gap: 20px; align-items: center; }
        .nav a { color: white; text-decoration: none; padding: 8px 15px; border-radius: 5px; transition: background 0.3s; }
        .nav a:hover { background: rgba(255,255,255,0.1); }
        .nav a.active { background: rgba(255,255,255,0.2); }
        .logout-btn { background: #c62828; padding: 8px 20px; border-radius: 5px; }
        
        .container { max-width: 1400px; margin: 30px auto; padding: 0 20px; }
        
        .filters {
            background: white;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 20px;
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
            align-items: center;
        }
        
        .filter-group { display: flex; gap: 10px; align-items: center; }
        .filter-btn {
            padding: 8px 20px;
            border: 2px solid #ddd;
            background: white;
            border-radius: 5px;
            cursor: pointer;
            transition: all 0.3s;
            text-decoration: none;
            color: #333;
        }
        .filter-btn:hover { border-color: #1e3c72; color: #1e3c72; }
        .filter-btn.active { background: #1e3c72; color: white; border-color: #1e3c72; }
        
        .search-box {
            flex: 1;
            min-width: 250px;
            position: relative;
        }
        .search-box input {
            width: 100%;
            padding: 10px 15px;
            border: 2px solid #ddd;
            border-radius: 5px;
            font-size: 14px;
        }
        
        .alerts-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 20px;
        }
        
        .alert-card {
            background: white;
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            border-left: 5px solid #ddd;
            transition: transform 0.2s;
        }
        .alert-card:hover { transform: translateY(-5px); box-shadow: 0 5px 20px rgba(0,0,0,0.15); }
        
        .alert-card.high { border-left-color: #c62828; }
        .alert-card.medium { border-left-color: #f57c00; }
        .alert-card.low { border-left-color: #1976d2; }
        
        .alert-header {
            display: flex;
            justify-content: space-between;
            align-items: start;
            margin-bottom: 15px;
        }
        
        .alert-id {
            font-size: 18px;
            font-weight: bold;
            color: #1e3c72;
        }
        
        .alert-type {
            background: #e3f2fd;
            color: #1976d2;
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
        }
        
        .alert-info { margin: 10px 0; }
        .alert-info div { margin: 5px 0; color: #666; font-size: 14px; }
        .alert-info strong { color: #333; }
        
        .alert-description {
            background: #f9f9f9;
            padding: 10px;
            border-radius: 5px;
            margin: 10px 0;
            font-size: 14px;
            color: #555;
        }
        
        .alert-footer {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 15px;
            padding-top: 15px;
            border-top: 1px solid #eee;
        }
        
        .badge {
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
        }
        .badge.pending { background: #ffebee; color: #c62828; }
        .badge.in-progress { background: #fff3e0; color: #f57c00; }
        .badge.resolved { background: #e8f5e9; color: #2e7d32; }
        .badge.cancelled { background: #f5f5f5; color: #666; }
        
        .btn {
            padding: 8px 15px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            text-decoration: none;
            font-size: 13px;
            font-weight: 600;
            transition: all 0.3s;
        }
        .btn-primary { background: #1e3c72; color: white; }
        .btn-primary:hover { background: #2a5298; }
        
        .no-alerts {
            text-align: center;
            padding: 60px;
            color: #999;
            grid-column: 1 / -1;
        }
        
        .stats-mini {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
        }
        .stat-mini {
            background: white;
            padding: 15px 20px;
            border-radius: 8px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
            flex: 1;
            text-align: center;
        }
        .stat-mini strong {
            display: block;
            font-size: 24px;
            color: #1e3c72;
            margin-bottom: 5px;
        }
        .stat-mini span { font-size: 12px; color: #666; }
    </style>
</head>
<body>
    <div class="header">
        <h1>🚨 Azuero Safety</h1>
        <div class="nav">
            <a href="dashboard.php">📊 Dashboard</a>
            <a href="alertas.php" class="active">🚨 Alertas</a>
            <a href="operadores.php">👥 Operadores</a>
            <span style="opacity: 0.7;">👤 <?php echo htmlspecialchars($_SESSION['full_name']); ?></span>
            <a href="logout.php" class="logout-btn">Salir</a>
        </div>
    </div>
    
    <div class="container">
        <div class="stats-mini">
            <div class="stat-mini">
                <strong><?php echo count(array_filter($alerts, fn($a) => $a['status'] == 'pending')); ?></strong>
                <span>Pendientes</span>
            </div>
            <div class="stat-mini">
                <strong><?php echo count(array_filter($alerts, fn($a) => $a['status'] == 'in_progress')); ?></strong>
                <span>En Progreso</span>
            </div>
            <div class="stat-mini">
                <strong><?php echo count(array_filter($alerts, fn($a) => $a['priority'] == 'high')); ?></strong>
                <span>Alta Prioridad</span>
            </div>
            <div class="stat-mini">
                <strong><?php echo count($alerts); ?></strong>
                <span>Total</span>
            </div>
        </div>
        
        <div class="filters">
            <div class="filter-group">
                <strong>Filtrar:</strong>
                <a href="?filter=all" class="filter-btn <?php echo $filter == 'all' ? 'active' : ''; ?>">Todas</a>
                <a href="?filter=pending" class="filter-btn <?php echo $filter == 'pending' ? 'active' : ''; ?>">Pendientes</a>
                <a href="?filter=in_progress" class="filter-btn <?php echo $filter == 'in_progress' ? 'active' : ''; ?>">En Progreso</a>
                <a href="?filter=resolved" class="filter-btn <?php echo $filter == 'resolved' ? 'active' : ''; ?>">Resueltas</a>
            </div>
            
            <form method="GET" class="search-box">
                <input type="hidden" name="filter" value="<?php echo htmlspecialchars($filter); ?>">
                <input type="text" name="search" placeholder="🔍 Buscar por ubicación, tipo o ciudadano..." value="<?php echo htmlspecialchars($search); ?>">
            </form>
        </div>
        
        <div class="alerts-grid">
            <?php if (count($alerts) > 0): ?>
                <?php foreach ($alerts as $alert): ?>
                <div class="alert-card <?php echo $alert['priority']; ?>">
                    <div class="alert-header">
                        <div class="alert-id">#<?php echo $alert['id']; ?></div>
                        <div class="alert-type"><?php echo htmlspecialchars($alert['alert_type']); ?></div>
                    </div>
                    
                    <div class="alert-info">
                        <div>📍 <strong>Ubicación:</strong> <?php echo htmlspecialchars($alert['location_name'] ?? 'No especificada'); ?></div>
                        <div>👤 <strong>Ciudadano:</strong> <?php echo htmlspecialchars($alert['citizen_name'] ?? 'Desconocido'); ?></div>
                        <div>⏰ <strong>Hora:</strong> <?php echo date('d/m/Y H:i', strtotime($alert['created_at'])); ?></div>
                        <div>🎯 <strong>Prioridad:</strong> <span style="color: <?php echo $alert['priority'] == 'high' ? '#c62828' : ($alert['priority'] == 'medium' ? '#f57c00' : '#1976d2'); ?>; font-weight: bold;"><?php echo strtoupper($alert['priority']); ?></span></div>
                        <?php if ($alert['operator_name']): ?>
                        <div>👨‍💼 <strong>Operador:</strong> <?php echo htmlspecialchars($alert['operator_name']); ?></div>
                        <?php endif; ?>
                    </div>
                    
                    <?php if ($alert['description']): ?>
                    <div class="alert-description">
                        💬 <?php echo htmlspecialchars($alert['description']); ?>
                    </div>
                    <?php endif; ?>
                    
                    <div class="alert-footer">
                        <span class="badge <?php echo $alert['status']; ?>">
                            <?php echo str_replace('_', ' ', strtoupper($alert['status'])); ?>
                        </span>
                        <a href="ver-alerta.php?id=<?php echo $alert['id']; ?>" class="btn btn-primary">Ver Detalles</a>
                    </div>
                </div>
                <?php endforeach; ?>
            <?php else: ?>
                <div class="no-alerts">
                    <h2>📭 No hay alertas</h2>
                    <p>No se encontraron alertas con los filtros seleccionados</p>
                </div>
            <?php endif; ?>
        </div>
    </div>
    
    <script>
        // Auto-refresh cada 30 segundos
        setTimeout(() => location.reload(), 30000);
    </script>
</body>
</html>
