<?php
require_once 'config.php';

// Verificar si está logueado
if (!isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit;
}

// Obtener idioma de la sesión
$lang = $_SESSION['lang'] ?? 'es';

// Si viene por parámetro GET, actualizar
if (isset($_GET['lang']) && in_array($_GET['lang'], ['es', 'en'])) {
    $lang = $_GET['lang'];
    $_SESSION['lang'] = $lang;
}

// Traducciones
$translations = [
    'es' => [
        'dashboard' => 'Panel de Control',
        'alerts' => 'Alertas',
        'operators' => 'Operadores',
        'codes' => 'Códigos',
        'logout' => 'Cerrar Sesión',
        'active_alerts' => 'Alertas Activas',
        'pending' => 'Pendientes',
        'in_progress' => 'En Progreso',
        'resolved_today' => 'Resueltas Hoy',
        'recent_alerts' => 'Alertas Recientes',
        'no_alerts' => 'No hay alertas registradas',
        'id' => 'ID',
        'type' => 'Tipo',
        'location' => 'Ubicación',
        'citizen' => 'Ciudadano',
        'priority' => 'Prioridad',
        'status' => 'Estado',
        'date' => 'Fecha',
        'not_specified' => 'No especificada',
        'unknown' => 'Desconocido',
        'high' => 'ALTA',
        'medium' => 'MEDIA',
        'low' => 'BAJA',
        'pending_status' => 'PENDIENTE',
        'in_progress_status' => 'EN PROGRESO',
        'resolved' => 'RESUELTA'
    ],
    'en' => [
        'dashboard' => 'Dashboard',
        'alerts' => 'Alerts',
        'operators' => 'Operators',
        'codes' => 'Codes',
        'logout' => 'Logout',
        'active_alerts' => 'Active Alerts',
        'pending' => 'Pending',
        'in_progress' => 'In Progress',
        'resolved_today' => 'Resolved Today',
        'recent_alerts' => 'Recent Alerts',
        'no_alerts' => 'No alerts registered',
        'id' => 'ID',
        'type' => 'Type',
        'location' => 'Location',
        'citizen' => 'Citizen',
        'priority' => 'Priority',
        'status' => 'Status',
        'date' => 'Date',
        'not_specified' => 'Not specified',
        'unknown' => 'Unknown',
        'high' => 'HIGH',
        'medium' => 'MEDIUM',
        'low' => 'LOW',
        'pending_status' => 'PENDING',
        'in_progress_status' => 'IN PROGRESS',
        'resolved' => 'RESOLVED'
    ]
];

$t = $translations[$lang];

// Obtener estadísticas
$stats = [];

try {
    // Total de alertas activas
    $stmt = $conn->query("SELECT COUNT(*) as total FROM alerts WHERE status IN ('pending', 'in_progress')");
    $stats['active_alerts'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Alertas pendientes
    $stmt = $conn->query("SELECT COUNT(*) as total FROM alerts WHERE status = 'pending'");
    $stats['pending_alerts'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Alertas en progreso
    $stmt = $conn->query("SELECT COUNT(*) as total FROM alerts WHERE status = 'in_progress'");
    $stats['in_progress_alerts'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Alertas resueltas hoy
    $stmt = $conn->query("SELECT COUNT(*) as total FROM alerts WHERE status = 'resolved' AND DATE(created_at) = CURDATE()");
    $stats['resolved_today'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Obtener últimas alertas
    $stmt = $conn->query("
        SELECT a.*, u.full_name as citizen_name 
        FROM alerts a 
        LEFT JOIN users u ON a.user_id = u.id 
        ORDER BY a.created_at DESC 
        LIMIT 10
    ");
    $recent_alerts = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch(PDOException $e) {
    $stats = ['active_alerts' => 0, 'pending_alerts' => 0, 'in_progress_alerts' => 0, 'resolved_today' => 0];
    $recent_alerts = [];
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Azuero Safety</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #f5f5f5;
        }
        
        /* Header */
        .header {
            background: #1e3c72;
            color: white;
            padding: 15px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        
        .header h1 {
            font-size: 24px;
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .logout-btn {
            background: #c62828;
            color: white;
            padding: 8px 20px;
            border-radius: 5px;
            text-decoration: none;
            font-size: 14px;
            transition: background 0.3s;
        }
        
        .logout-btn:hover {
            background: #d32f2f;
        }
        
        /* Container */
        .container {
            max-width: 1400px;
            margin: 30px auto;
            padding: 0 20px;
        }
        
        /* Stats */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            padding: 25px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .stat-card h3 {
            color: #666;
            font-size: 14px;
            font-weight: 600;
            margin-bottom: 10px;
            text-transform: uppercase;
        }
        
        .stat-value {
            font-size: 36px;
            font-weight: bold;
            color: #1e3c72;
            transition: all 0.3s ease;
        }
        
        .stat-value.updated {
            animation: pulse 0.5s ease;
        }
        
        @keyframes pulse {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.1); color: #4CAF50; }
        }
        
        .stat-card.urgent .stat-value { color: #c62828; }
        .stat-card.warning .stat-value { color: #f57c00; }
        .stat-card.success .stat-value { color: #2e7d32; }
        
        /* Alerts Table */
        .section {
            background: white;
            padding: 25px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .section h2 {
            color: #1e3c72;
            margin-bottom: 20px;
            font-size: 20px;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        th {
            background: #f5f5f5;
            padding: 12px;
            text-align: left;
            font-weight: 600;
            color: #333;
            border-bottom: 2px solid #ddd;
        }
        
        td {
            padding: 12px;
            border-bottom: 1px solid #eee;
        }
        
        tr:hover {
            background: #f9f9f9;
        }
        
        .badge {
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
        }
        
        .badge.pending { background: #ffebee; color: #c62828; }
        .badge.in-progress { background: #fff3e0; color: #f57c00; }
        .badge.resolved { background: #e8f5e9; color: #2e7d32; }
        .badge.high { background: #ffebee; color: #c62828; }
        .badge.medium { background: #fff3e0; color: #f57c00; }
        .badge.low { background: #e3f2fd; color: #1976d2; }
        
        .no-data {
            text-align: center;
            padding: 40px;
            color: #999;
        }
        
        /* ========================================
           RESPONSIVE DESIGN PARA MÓVILES
           ======================================== */
        @media (max-width: 768px) {
            .header {
                flex-direction: column;
                padding: 15px;
                gap: 10px;
            }
            
            .header h1 {
                font-size: 20px;
                text-align: center;
            }
            
            .user-info {
                flex-direction: column;
                gap: 8px;
                font-size: 13px;
            }
            
            .user-info a {
                font-size: 13px !important;
            }
            
            .logout-btn {
                padding: 8px 15px;
                font-size: 13px;
            }
            
            .container {
                margin: 15px auto;
                padding: 0 10px;
            }
            
            /* Stats en columna en móvil */
            .stats-grid {
                grid-template-columns: 1fr;
                gap: 15px;
                margin-bottom: 20px;
            }
            
            .stat-card {
                padding: 20px;
            }
            
            .stat-card h3 {
                font-size: 13px;
            }
            
            .stat-value {
                font-size: 28px;
            }
            
            /* Tabla responsive */
            .section {
                padding: 15px;
                overflow-x: auto;
            }
            
            .section h2 {
                font-size: 18px;
                margin-bottom: 15px;
            }
            
            table {
                font-size: 12px;
            }
            
            th, td {
                padding: 8px 6px;
            }
            
            th {
                font-size: 11px;
            }
            
            .badge {
                font-size: 10px;
                padding: 4px 8px;
            }
        }
        
        @media (max-width: 480px) {
            .header h1 {
                font-size: 18px;
            }
            
            .user-info {
                font-size: 11px;
            }
            
            .stat-value {
                font-size: 24px;
            }
            
            /* Ocultar columnas específicas con clase */
            .hide-mobile {
                display: none !important;
            }
        }
        
        /* Mejoras para notificaciones en móvil */
        @media (max-width: 600px) {
            .notification-popup {
                left: 10px !important;
                right: 10px !important;
                top: 10px !important;
                width: auto !important;
                max-width: none !important;
            }
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>🚨 Azuero Safety</h1>
        <div class="user-info">
            <a href="dashboard.php" style="color: white; text-decoration: none; margin-right: 20px; opacity: 0.9;">📊 <?php echo $t['dashboard']; ?></a>
            <a href="alertas.php" style="color: white; text-decoration: none; margin-right: 20px; opacity: 0.9;">🚨 <?php echo $t['alerts']; ?></a>
            <a href="operadores.php" style="color: white; text-decoration: none; margin-right: 20px; opacity: 0.9;">👥 <?php echo $t['operators']; ?></a>
            <?php if ($_SESSION['role'] == 'admin'): ?>
            <a href="admin-codigos.php" style="color: white; text-decoration: none; margin-right: 20px; opacity: 0.9;">🔑 <?php echo $t['codes']; ?></a>
            <?php endif; ?>
            <span>👤 <?php echo htmlspecialchars($_SESSION['full_name']); ?> (<?php echo ucfirst($_SESSION['role']); ?>)</span>
            <a href="?lang=<?php echo $lang == 'es' ? 'en' : 'es'; ?>" style="color: white; text-decoration: none; margin-right: 10px; opacity: 0.9;">🌐 <?php echo $lang == 'es' ? 'EN' : 'ES'; ?></a>
            <a href="logout.php" class="logout-btn"><?php echo $t['logout']; ?></a>
        </div>
    </div>
    
    <div class="container">
        <!-- Estadísticas -->
        <div class="stats-grid">
            <div class="stat-card urgent">
                <h3>⚠️ <?php echo $t['active_alerts']; ?></h3>
                <div class="stat-value"><?php echo $stats['active_alerts']; ?></div>
            </div>
            
            <div class="stat-card warning">
                <h3>🔔 <?php echo $t['pending']; ?></h3>
                <div class="stat-value"><?php echo $stats['pending_alerts']; ?></div>
            </div>
            
            <div class="stat-card">
                <h3>⏳ <?php echo $t['in_progress']; ?></h3>
                <div class="stat-value"><?php echo $stats['in_progress_alerts']; ?></div>
            </div>
            
            <div class="stat-card success">
                <h3>✅ <?php echo $t['resolved_today']; ?></h3>
                <div class="stat-value"><?php echo $stats['resolved_today']; ?></div>
            </div>
        </div>
        
        <!-- Tabla de alertas recientes -->
        <div class="section recent-alerts">
            <h2>📋 <?php echo $t['recent_alerts']; ?></h2>
            
            <?php if (count($recent_alerts) > 0): ?>
            <table>
                <thead>
                    <tr>
                        <th><?php echo $t['id']; ?></th>
                        <th><?php echo $t['type']; ?></th>
                        <th><?php echo $t['location']; ?></th>
                        <th><?php echo $t['citizen']; ?></th>
                        <th><?php echo $t['priority']; ?></th>
                        <th><?php echo $t['status']; ?></th>
                        <th><?php echo $t['date']; ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php 
                    $priority_trans = ['high' => $t['high'], 'medium' => $t['medium'], 'low' => $t['low']];
                    $status_trans = ['pending' => $t['pending_status'], 'in_progress' => $t['in_progress_status'], 'resolved' => $t['resolved']];
                    foreach ($recent_alerts as $alert): 
                    ?>
                    <tr>
                        <td><strong>#<?php echo $alert['id']; ?></strong></td>
                        <td><?php echo htmlspecialchars($alert['alert_type']); ?></td>
                        <td><?php echo htmlspecialchars($alert['location_name'] ?? $t['not_specified']); ?></td>
                        <td><?php echo htmlspecialchars($alert['citizen_name'] ?? $t['unknown']); ?></td>
                        <td><span class="badge <?php echo $alert['priority']; ?>"><?php echo $priority_trans[$alert['priority']] ?? strtoupper($alert['priority']); ?></span></td>
                        <td><span class="badge <?php echo $alert['status']; ?>"><?php echo $status_trans[$alert['status']] ?? ucfirst($alert['status']); ?></span></td>
                        <td><?php echo date('d/m/Y H:i', strtotime($alert['created_at'])); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php else: ?>
            <div class="no-data">
                <p>📭 <?php echo $t['no_alerts']; ?></p>
            </div>
            <?php endif; ?>
        </div>
    </div>
    
    <script>
        // Version: <?php echo time(); ?> - Sistema de notificaciones en tiempo real sin reload
        let lastCheck = Math.floor(Date.now() / 1000);
        let notificationSound = null;
        
        // Inicializar sonido de notificación (opcional)
        function initSound() {
            // Audio simple de alerta (puedes reemplazar con un archivo .mp3)
            const audioContext = new (window.AudioContext || window.webkitAudioContext)();
            notificationSound = function() {
                const oscillator = audioContext.createOscillator();
                const gainNode = audioContext.createGain();
                oscillator.connect(gainNode);
                gainNode.connect(audioContext.destination);
                oscillator.frequency.value = 800;
                oscillator.type = 'sine';
                gainNode.gain.setValueAtTime(0.3, audioContext.currentTime);
                gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.5);
                oscillator.start(audioContext.currentTime);
                oscillator.stop(audioContext.currentTime + 0.5);
            };
        }
        
        // Verificar nuevas alertas cada 3 segundos
        function checkNewAlerts() {
            fetch('check-new-alerts.php?last_check=' + lastCheck)
                .then(response => response.json())
                .then(data => {
                    console.log('Datos recibidos del servidor:', data);
                    
                    if (data.success) {
                        // Actualizar timestamp
                        lastCheck = data.timestamp;
                        
                        // Siempre actualizar la tabla y estadísticas
                        if (data.alerts) {
                            console.log('Actualizando tabla con', data.alerts.length, 'alertas');
                            updateAlertsList(data.alerts);
                        }
                        
                        if (data.stats) {
                            console.log('Actualizando estadísticas:', data.stats);
                            updateStats(data.stats);
                            updateBadge(data.stats.pending);
                        }
                        
                        // Si hay nuevas alertas, notificar
                        if (data.has_new && data.new_count > 0) {
                            console.log('Nuevas alertas detectadas:', data.new_count);
                            showNotification(data.new_count);
                            if (notificationSound) notificationSound();
                        }
                    } else if (data.error) {
                        console.error('Error del servidor:', data.error);
                    }
                })
                .catch(error => console.error('Error checking alerts:', error));
        }
        
        // Mostrar notificación del navegador
        function showNotification(count) {
            if ('Notification' in window && Notification.permission === 'granted') {
                new Notification('🚨 Nueva Alerta de Emergencia', {
                    body: `${count} nueva${count > 1 ? 's' : ''} alerta${count > 1 ? 's' : ''} pendiente${count > 1 ? 's' : ''}`,
                    icon: 'https://cdn-icons-png.flaticon.com/512/564/564619.png',
                    badge: 'https://cdn-icons-png.flaticon.com/512/564/564619.png',
                    tag: 'emergency-alert',
                    requireInteraction: true
                });
            } else {
                // Notificación visual en la página
                const notification = document.createElement('div');
                notification.style.cssText = `
                    position: fixed; top: 20px; right: 20px; 
                    background: #d32f2f; color: white; 
                    padding: 20px 30px; border-radius: 10px; 
                    font-weight: bold; font-size: 16px;
                    box-shadow: 0 4px 20px rgba(0,0,0,0.3);
                    z-index: 10000; animation: slideIn 0.3s;
                `;
                notification.innerHTML = `🚨 ${count} Nueva${count > 1 ? 's' : ''} Alerta${count > 1 ? 's' : ''}`;
                document.body.appendChild(notification);
                
                setTimeout(() => {
                    notification.style.animation = 'slideOut 0.3s';
                    setTimeout(() => notification.remove(), 300);
                }, 5000);
            }
        }
        
        // Actualizar lista de alertas dinámicamente
        function updateAlertsList(alerts) {
            const container = document.querySelector('.recent-alerts');
            if (!container) {
                console.error('Contenedor .recent-alerts no encontrado');
                return;
            }
            
            console.log('Actualizando tabla con', alerts.length, 'alertas');
            
            // Limpiar y crear nuevo contenido
            container.innerHTML = '<h2>📋 Alertas Recientes</h2>';
            
            if (alerts && alerts.length > 0) {
                const table = document.createElement('table');
                table.innerHTML = `
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Tipo</th>
                            <th class="hide-mobile">Ubicación</th>
                            <th>Ciudadano</th>
                            <th>Prioridad</th>
                            <th>Estado</th>
                            <th class="hide-mobile">Fecha</th>
                        </tr>
                    </thead>
                    <tbody>
                    </tbody>
                `;
                
                const tbody = table.querySelector('tbody');
                alerts.forEach(alert => {
                    const row = document.createElement('tr');
                    const statusMap = {
                        'pending': 'PENDIENTE',
                        'in_progress': 'EN PROGRESO',
                        'resolved': 'RESUELTA'
                    };
                    
                    row.innerHTML = `
                        <td><strong>#${alert.id}</strong></td>
                        <td>${alert.alert_type || 'N/A'}</td>
                        <td class="hide-mobile">${alert.location_name || 'No especificada'}</td>
                        <td>${alert.full_name || 'Desconocido'}</td>
                        <td><span class="badge ${alert.priority}">${(alert.priority || 'medium').toUpperCase()}</span></td>
                        <td><span class="badge ${alert.status}">${statusMap[alert.status] || alert.status}</span></td>
                        <td class="hide-mobile">${formatDateTime(alert.created_at)}</td>
                    `;
                    tbody.appendChild(row);
                });
                
                container.appendChild(table);
            } else {
                container.innerHTML += '<div class="no-data"><p>📭 No hay alertas registradas</p></div>';
            }
        }
        
        // Formatear fecha y hora
        function formatDateTime(dateString) {
            if (!dateString) return 'N/A';
            const date = new Date(dateString);
            if (isNaN(date.getTime())) return 'Fecha inválida';
            
            const day = String(date.getDate()).padStart(2, '0');
            const month = String(date.getMonth() + 1).padStart(2, '0');
            const year = date.getFullYear();
            const hours = String(date.getHours()).padStart(2, '0');
            const minutes = String(date.getMinutes()).padStart(2, '0');
            return `${day}/${month}/${year} ${hours}:${minutes}`;
        }
        
        // Actualizar estadísticas con animación
        function updateStats(stats) {
            if (!stats) {
                console.warn('No hay estadísticas para actualizar');
                return;
            }
            
            // Convertir a números seguros (0 si es null/undefined/NaN)
            const pending = parseInt(stats.pending) || 0;
            const inProgress = parseInt(stats.in_progress) || 0;
            const resolved = parseInt(stats.resolved) || 0;
            const total = parseInt(stats.total) || 0;
            
            console.log('Actualizando stats:', { pending, inProgress, resolved, total });
            
            const elements = [
                { el: document.querySelector('.stat-card:nth-child(1) .stat-value'), value: pending + inProgress },
                { el: document.querySelector('.stat-card:nth-child(2) .stat-value'), value: pending },
                { el: document.querySelector('.stat-card:nth-child(3) .stat-value'), value: inProgress },
                { el: document.querySelector('.stat-card:nth-child(4) .stat-value'), value: resolved }
            ];
            
            elements.forEach(item => {
                if (item.el) {
                    const newValue = item.value || 0;
                    if (item.el.textContent != newValue) {
                        item.el.textContent = newValue;
                        item.el.classList.add('updated');
                        setTimeout(() => item.el.classList.remove('updated'), 500);
                    }
                }
            });
        }
        
        // Actualizar badge en el menú
        function updateBadge(count) {
            let badge = document.querySelector('.alert-badge');
            if (count > 0) {
                if (!badge) {
                    badge = document.createElement('span');
                    badge.className = 'alert-badge';
                    badge.style.cssText = `
                        background: #d32f2f; color: white; 
                        border-radius: 50%; width: 20px; height: 20px; 
                        display: inline-flex; align-items: center; 
                        justify-content: center; font-size: 11px; 
                        font-weight: bold; margin-left: 5px;
                    `;
                    const alertLink = document.querySelector('a[href="alertas.php"]');
                    if (alertLink) alertLink.appendChild(badge);
                }
                badge.textContent = count > 99 ? '99+' : count;
            }
        }
        
        // Pedir permiso para notificaciones
        if ('Notification' in window && Notification.permission === 'default') {
            Notification.requestPermission();
        }
        
        // Inicializar sonido al hacer clic (por políticas del navegador)
        document.addEventListener('click', function initSoundOnce() {
            initSound();
            document.removeEventListener('click', initSoundOnce);
        }, { once: true });
        
        // Iniciar verificación cada 3 segundos
        setInterval(checkNewAlerts, 3000);
        checkNewAlerts(); // Primera verificación inmediata
        
        // Agregar animaciones CSS
        const style = document.createElement('style');
        style.textContent = `
            @keyframes slideIn {
                from { transform: translateX(400px); opacity: 0; }
                to { transform: translateX(0); opacity: 1; }
            }
            @keyframes slideOut {
                from { transform: translateX(0); opacity: 1; }
                to { transform: translateX(400px); opacity: 0; }
            }
        `;
        document.head.appendChild(style);
    </script>
</body>
</html>
