<?php
require_once 'config.php';

// Verificar si está logueado como operador
if (!isset($_SESSION['operator_id'])) {
    header('Location: index.php');
    exit;
}

// Obtener información del operador
try {
    $stmt = $conn->prepare("SELECT * FROM operators WHERE id = ?");
    $stmt->execute([$_SESSION['operator_id']]);
    $operator = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$operator) {
        session_destroy();
        header('Location: index.php');
        exit();
    }
} catch (PDOException $e) {
    die("Error al obtener información del operador: " . $e->getMessage());
}

// Configuración de idioma - Persistente en BD
$lang = $operator['language'] ?? 'es';

if (isset($_GET['lang']) && in_array($_GET['lang'], ['es', 'en'])) {
    $lang = $_GET['lang'];
    $_SESSION['lang'] = $lang;
    
    // Guardar en la base de datos para persistencia
    try {
        $stmt = $conn->prepare("UPDATE operators SET language = ? WHERE id = ?");
        $stmt->execute([$lang, $_SESSION['operator_id']]);
    } catch(PDOException $e) {
        // Si falla, continuamos con el idioma en sesión
    }
}

$translations = [
    'es' => [
        'operator_panel' => 'Panel de Operador',
        'dashboard' => 'Dashboard',
        'alerts' => 'Alertas',
        'my_alerts' => 'Mis Alertas',
        'all_alerts' => 'Todas las Alertas',
        'profile' => 'Perfil',
        'logout' => 'Cerrar Sesión',
        'welcome' => 'Bienvenido',
        'search' => 'Buscar...',
        'active_alerts' => 'Alertas Activas',
        'pending' => 'Pendientes',
        'in_progress' => 'En Progreso',
        'resolved_today' => 'Resueltas Hoy',
        'my_assignments' => 'Mis Asignaciones',
        'recent_alerts' => 'Alertas Recientes',
        'view_all' => 'Ver Todas',
        'medical' => 'Médica',
        'fire' => 'Incendio',
        'security' => 'Seguridad',
        'theme' => 'Tema',
        'light' => 'Claro',
        'dark' => 'Oscuro',
        'blue_dark' => 'Azul Oscuro',
        'no_alerts' => 'No hay alertas',
        'id' => 'ID',
        'type' => 'Tipo',
        'location' => 'Ubicación',
        'citizen' => 'Ciudadano',
        'priority' => 'Prioridad',
        'status' => 'Estado',
        'date' => 'Fecha',
        'actions' => 'Acciones',
        'view_details' => 'Ver Detalles',
        'assign_to_me' => 'Asignar a mí',
        'high' => 'ALTA',
        'medium' => 'MEDIA',
        'low' => 'BAJA',
        'pending_status' => 'PENDIENTE',
        'in_progress_status' => 'EN PROGRESO',
        'resolved' => 'RESUELTA',
        'not_specified' => 'No especificada',
        'unknown' => 'Desconocido',
        'emergency_response' => 'Sistema de Respuesta a Emergencias'
    ],
    'en' => [
        'operator_panel' => 'Operator Panel',
        'dashboard' => 'Dashboard',
        'alerts' => 'Alerts',
        'my_alerts' => 'My Alerts',
        'all_alerts' => 'All Alerts',
        'profile' => 'Profile',
        'logout' => 'Logout',
        'welcome' => 'Welcome',
        'search' => 'Search...',
        'active_alerts' => 'Active Alerts',
        'pending' => 'Pending',
        'in_progress' => 'In Progress',
        'resolved_today' => 'Resolved Today',
        'my_assignments' => 'My Assignments',
        'recent_alerts' => 'Recent Alerts',
        'view_all' => 'View All',
        'medical' => 'Medical',
        'fire' => 'Fire',
        'security' => 'Security',
        'theme' => 'Theme',
        'light' => 'Light',
        'dark' => 'Dark',
        'blue_dark' => 'Blue Dark',
        'no_alerts' => 'No alerts',
        'id' => 'ID',
        'type' => 'Type',
        'location' => 'Location',
        'citizen' => 'Citizen',
        'priority' => 'Priority',
        'status' => 'Status',
        'date' => 'Date',
        'actions' => 'Actions',
        'view_details' => 'View Details',
        'assign_to_me' => 'Assign to me',
        'high' => 'HIGH',
        'medium' => 'MEDIUM',
        'low' => 'LOW',
        'pending_status' => 'PENDING',
        'in_progress_status' => 'IN PROGRESS',
        'resolved' => 'RESOLVED',
        'not_specified' => 'Not specified',
        'unknown' => 'Unknown',
        'emergency_response' => 'Emergency Response System'
    ]
];

$t = $translations[$lang];

// Obtener estadísticas (con manejo de errores)
$stats = [];

try {
    // Alertas activas
    $stmt = $conn->query("SELECT COUNT(*) as total FROM alerts WHERE status IN ('pending', 'in_progress')");
    $stats['active_alerts'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
} catch (PDOException $e) {
    $stats['active_alerts'] = 0;
}

try {
    // Alertas pendientes
    $stmt = $conn->query("SELECT COUNT(*) as total FROM alerts WHERE status = 'pending'");
    $stats['pending'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
} catch (PDOException $e) {
    $stats['pending'] = 0;
}

try {
    // Alertas en progreso
    $stmt = $conn->query("SELECT COUNT(*) as total FROM alerts WHERE status = 'in_progress'");
    $stats['in_progress'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
} catch (PDOException $e) {
    $stats['in_progress'] = 0;
}

try {
    // Resueltas hoy
    $stmt = $conn->query("SELECT COUNT(*) as total FROM alerts WHERE status = 'resolved' AND DATE(updated_at) = CURDATE()");
    $stats['resolved_today'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
} catch (PDOException $e) {
    $stats['resolved_today'] = 0;
}

try {
    // Mis asignaciones
    $stmt = $conn->prepare("SELECT COUNT(*) as total FROM alerts WHERE assigned_operator_id = ? AND status IN ('pending', 'in_progress')");
    $stmt->execute([$_SESSION['operator_id']]);
    $stats['my_assignments'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;
} catch (PDOException $e) {
    $stats['my_assignments'] = 0;
}

try {
    // Alertas recientes
    $stmt = $conn->query("
        SELECT a.*, u.full_name as user_name, u.email as user_email, o.full_name as operator_name
        FROM alerts a
        LEFT JOIN users u ON a.user_id = u.id
        LEFT JOIN operators o ON a.assigned_operator_id = o.id
        ORDER BY a.created_at DESC
        LIMIT 20
    ");
    $recent_alerts = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $recent_alerts = [];
}
?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title><?php echo $t['operator_panel']; ?> - Azuero Safety Services S.A</title>

    <!-- PWA Meta Tags -->
    <meta name="theme-color" content="#667eea">
    <meta name="mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="black-translucent">
    <meta name="apple-mobile-web-app-title" content="AzueroSafe">
    <meta name="description" content="Panel de Operadores - Azuero Safety Services S.A">

    <!-- PWA Manifest -->
    <link rel="manifest" href="manifest.json">

    <!-- Icons -->
    <link rel="icon" type="image/png" sizes="192x192" href="icon-192x192.png">
    <link rel="apple-touch-icon" href="icon-192x192.png">
    
    <!-- PWA Manifest -->
    <link rel="manifest" href="manifest.json">
    
    <!-- Icons -->
    <link rel="icon" type="image/png" sizes="192x192" href="icons/icon-192x192.png">
    <link rel="apple-touch-icon" sizes="192x192" href="icons/icon-192x192.png">
    
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary: #667eea;
            --primary-dark: #5568d3;
            --secondary: #764ba2;
            --success: #10b981;
            --warning: #f59e0b;
            --danger: #ef4444;
            --info: #3b82f6;
            --bg-primary: #ffffff;
            --bg-secondary: #f8fafc;
            --bg-card: #ffffff;
            --text-primary: #1e293b;
            --text-secondary: #64748b;
            --border-color: #e2e8f0;
            --shadow-sm: 0 1px 3px rgba(0,0,0,0.1);
            --shadow-md: 0 4px 6px rgba(0,0,0,0.1);
            --shadow-lg: 0 10px 25px rgba(0,0,0,0.15);
            --sidebar-width: 280px;
        }

        [data-theme="dark"] {
            --primary: #818cf8;
            --primary-dark: #6366f1;
            --bg-primary: #0f172a;
            --bg-secondary: #1e293b;
            --bg-card: #1e293b;
            --text-primary: #f1f5f9;
            --text-secondary: #94a3b8;
            --border-color: #334155;
            --shadow-sm: 0 1px 3px rgba(0,0,0,0.3);
            --shadow-md: 0 4px 6px rgba(0,0,0,0.3);
            --shadow-lg: 0 10px 25px rgba(0,0,0,0.5);
        }

        [data-theme="blue-dark"] {
            --primary: #60a5fa;
            --primary-dark: #3b82f6;
            --secondary: #06b6d4;
            --bg-primary: #0c1e35;
            --bg-secondary: #1a2942;
            --bg-card: #1a2942;
            --text-primary: #e0f2fe;
            --text-secondary: #7dd3fc;
            --border-color: #1e3a5f;
            --shadow-sm: 0 1px 3px rgba(14,165,233,0.2);
            --shadow-md: 0 4px 6px rgba(14,165,233,0.2);
            --shadow-lg: 0 10px 25px rgba(14,165,233,0.3);
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: var(--bg-primary);
            color: var(--text-primary);
            transition: all 0.3s ease;
        }

        /* Sidebar */
        .sidebar {
            position: fixed;
            left: 0;
            top: 0;
            bottom: 0;
            width: var(--sidebar-width);
            background: var(--bg-card);
            border-right: 1px solid var(--border-color);
            box-shadow: var(--shadow-md);
            z-index: 1000;
            transition: all 0.3s ease;
            overflow-y: auto;
        }

        .sidebar-header {
            padding: 30px 20px;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            text-align: center;
        }

        .sidebar-header h2 {
            font-size: 24px;
            margin-bottom: 5px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }

        .sidebar-header p {
            font-size: 14px;
            opacity: 0.9;
        }

        .sidebar-menu {
            padding: 20px 0;
        }

        .menu-item {
            display: flex;
            align-items: center;
            padding: 15px 25px;
            color: var(--text-primary);
            text-decoration: none;
            transition: all 0.3s ease;
            cursor: pointer;
            border-left: 3px solid transparent;
        }

        .menu-item:hover, .menu-item.active {
            background: var(--bg-secondary);
            border-left-color: var(--primary);
            color: var(--primary);
        }

        .menu-item i {
            width: 24px;
            font-size: 20px;
            margin-right: 15px;
        }

        .menu-section {
            padding: 10px 25px;
            font-size: 12px;
            font-weight: 600;
            color: var(--text-secondary);
            text-transform: uppercase;
            letter-spacing: 1px;
            margin-top: 20px;
        }

        /* Main Content */
        .main-content {
            margin-left: var(--sidebar-width);
            min-height: 100vh;
            transition: margin-left 0.3s ease;
        }

        /* Header */
        .header {
            background: var(--bg-card);
            border-bottom: 1px solid var(--border-color);
            padding: 20px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: var(--shadow-sm);
            position: sticky;
            top: 0;
            z-index: 999;
        }

        .header-left h1 {
            font-size: 28px;
            color: var(--text-primary);
            margin-bottom: 5px;
        }

        .breadcrumb {
            color: var(--text-secondary);
            font-size: 14px;
        }

        .header-right {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .search-box {
            position: relative;
        }

        .search-box input {
            padding: 10px 40px 10px 15px;
            border: 1px solid var(--border-color);
            border-radius: 25px;
            background: var(--bg-secondary);
            color: var(--text-primary);
            width: 300px;
            transition: all 0.3s ease;
        }

        .search-box input:focus {
            outline: none;
            border-color: var(--primary);
            width: 350px;
        }

        .search-box i {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--text-secondary);
        }

        .notification-bell {
            position: relative;
            cursor: pointer;
            font-size: 24px;
            color: var(--text-primary);
            transition: color 0.3s ease;
        }

        .notification-bell:hover {
            color: var(--primary);
        }

        .notification-badge {
            position: absolute;
            top: -8px;
            right: -8px;
            background: var(--danger);
            color: white;
            border-radius: 50%;
            width: 20px;
            height: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 11px;
            font-weight: bold;
            animation: pulse 2s infinite;
        }

        @keyframes pulse {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.1); }
        }

        .theme-switcher {
            display: flex;
            gap: 10px;
            background: var(--bg-secondary);
            padding: 5px;
            border-radius: 25px;
        }

        .theme-btn, .lang-btn {
            padding: 8px 15px;
            border: none;
            background: transparent;
            color: var(--text-secondary);
            cursor: pointer;
            border-radius: 20px;
            font-size: 13px;
            transition: all 0.3s ease;
        }

        .theme-btn.active, .lang-btn.active {
            background: var(--primary);
            color: white;
        }

        .lang-switcher {
            display: flex;
            gap: 10px;
            align-items: center;
            background: var(--bg-secondary);
            padding: 5px;
            border-radius: 25px;
        }

        .user-profile {
            display: flex;
            align-items: center;
            gap: 12px;
            cursor: pointer;
            padding: 8px 15px;
            border-radius: 25px;
            transition: all 0.3s ease;
        }

        .user-profile:hover {
            background: var(--bg-secondary);
        }

        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
            font-size: 16px;
        }

        .user-info h4 {
            font-size: 14px;
            margin-bottom: 2px;
        }

        .user-info p {
            font-size: 12px;
            color: var(--text-secondary);
        }

        /* Content Container */
        .content {
            padding: 30px;
        }

        /* Stats Cards */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 15px;
            padding: 25px;
            box-shadow: var(--shadow-sm);
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, var(--primary), var(--secondary));
        }

        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: var(--shadow-lg);
        }

        .stat-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 15px;
        }

        .stat-icon {
            width: 50px;
            height: 50px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: white;
        }

        .stat-icon.blue { background: linear-gradient(135deg, #667eea, #764ba2); }
        .stat-icon.green { background: linear-gradient(135deg, #10b981, #059669); }
        .stat-icon.orange { background: linear-gradient(135deg, #f59e0b, #d97706); }
        .stat-icon.red { background: linear-gradient(135deg, #ef4444, #dc2626); }
        .stat-icon.purple { background: linear-gradient(135deg, #8b5cf6, #7c3aed); }

        .stat-value {
            font-size: 32px;
            font-weight: bold;
            color: var(--text-primary);
            margin-bottom: 5px;
        }

        .stat-label {
            font-size: 14px;
            color: var(--text-secondary);
        }

        /* Tables */
        .table-card {
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 15px;
            padding: 25px;
            box-shadow: var(--shadow-sm);
            margin-bottom: 30px;
        }

        .table-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        .table-header h3 {
            font-size: 18px;
            color: var(--text-primary);
        }

        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 500;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
            box-shadow: var(--shadow-md);
        }

        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
        }

        .table-wrapper {
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
            border-radius: 8px;
        }

        @media (max-width: 768px) {
            .table-wrapper {
                margin: 0 -18px;
                padding: 0 18px;
            }
        }

        table {
            width: 100%;
            border-collapse: collapse;
        }

        thead {
            background: var(--bg-secondary);
        }

        th, td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid var(--border-color);
        }

        th {
            font-weight: 600;
            color: var(--text-secondary);
            font-size: 13px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        td {
            color: var(--text-primary);
        }

        tbody tr {
            transition: background 0.3s ease;
        }

        tbody tr:hover {
            background: var(--bg-secondary);
        }

        .badge {
            display: inline-block;
            padding: 5px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 500;
        }

        .badge-success { background: rgba(16, 185, 129, 0.1); color: var(--success); }
        .badge-warning { background: rgba(245, 158, 11, 0.1); color: var(--warning); }
        .badge-danger { background: rgba(239, 68, 68, 0.1); color: var(--danger); }
        .badge-info { background: rgba(59, 130, 246, 0.1); color: var(--info); }

        /* Toast Notifications */
        .toast-container {
            position: fixed;
            top: 80px;
            right: 20px;
            z-index: 9999;
            max-width: 400px;
        }

        .toast {
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 10px;
            box-shadow: var(--shadow-lg);
            display: flex;
            align-items: flex-start;
            gap: 15px;
            animation: slideIn 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        @keyframes slideIn {
            from {
                transform: translateX(400px);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }

        .toast::before {
            content: '';
            position: absolute;
            left: 0;
            top: 0;
            bottom: 0;
            width: 4px;
        }

        .toast.emergency::before { background: var(--danger); }
        .toast.success::before { background: var(--success); }

        .toast-icon {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 20px;
            flex-shrink: 0;
        }

        .toast.emergency .toast-icon {
            background: rgba(239, 68, 68, 0.1);
            color: var(--danger);
        }

        .toast.success .toast-icon {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
        }

        .toast-content h4 {
            font-size: 14px;
            margin-bottom: 5px;
            color: var(--text-primary);
        }

        .toast-content p {
            font-size: 13px;
            color: var(--text-secondary);
            line-height: 1.5;
        }

        .toast-close {
            position: absolute;
            top: 10px;
            right: 10px;
            background: none;
            border: none;
            color: var(--text-secondary);
            cursor: pointer;
            font-size: 18px;
        }

        /* Responsive */
        /* Tablets y pantallas medianas */
        @media (max-width: 1024px) {
            .sidebar {
                transform: translateX(-100%);
            }

            .main-content {
                margin-left: 0;
            }

            .stats-grid {
                grid-template-columns: repeat(2, 1fr);
                gap: 15px;
            }

            .content {
                padding: 20px;
            }
        }

        /* Tablets pequeñas */
        @media (max-width: 768px) {
            .header {
                flex-direction: column;
                gap: 15px;
                padding: 15px 20px;
            }

            .search-box input {
                width: 100%;
                font-size: 14px;
            }

            .header-right {
                flex-wrap: wrap;
                width: 100%;
                justify-content: center;
                gap: 10px;
            }

            .theme-switcher, .lang-switcher {
                order: -1;
                width: 100%;
                justify-content: center;
            }

            .lang-btn, .theme-btn {
                font-size: 12px;
                padding: 6px 12px;
            }

            .stats-grid {
                grid-template-columns: 1fr;
                gap: 12px;
            }

            .stat-card {
                padding: 18px;
            }

            .stat-value {
                font-size: 28px;
            }

            .stat-label {
                font-size: 13px;
            }

            .content {
                padding: 15px;
            }

            .card {
                padding: 18px;
                border-radius: 12px;
            }

            .card-header h3 {
                font-size: 18px;
            }

            table {
                font-size: 13px;
            }

            .btn {
                padding: 8px 12px;
                font-size: 12px;
            }
        }

        /* Móviles */
        @media (max-width: 480px) {
            body {
                font-size: 14px;
            }

            .header {
                padding: 12px 15px;
            }

            .header-left h1 {
                font-size: 20px;
            }

            .breadcrumb {
                font-size: 12px;
            }

            .user-info {
                padding: 8px 12px;
            }

            .user-info span {
                font-size: 13px;
            }

            .content {
                padding: 12px;
            }

            .stats-grid {
                gap: 10px;
            }

            .stat-card {
                padding: 15px;
            }

            .stat-icon {
                width: 45px;
                height: 45px;
                font-size: 20px;
            }

            .stat-value {
                font-size: 24px;
            }

            .stat-label {
                font-size: 12px;
            }

            .card {
                padding: 15px;
                margin-bottom: 12px;
            }

            .card-header {
                margin-bottom: 12px;
                padding-bottom: 10px;
            }

            .card-header h3 {
                font-size: 16px;
            }

            table {
                font-size: 12px;
            }

            table th,
            table td {
                padding: 8px 6px;
            }

            .btn {
                padding: 6px 10px;
                font-size: 11px;
            }

            .badge {
                padding: 4px 8px;
                font-size: 10px;
            }

            /* Hacer tabla scrollable horizontalmente */
            .table-container {
                overflow-x: auto;
                -webkit-overflow-scrolling: touch;
            }

            table {
                min-width: 600px;
            }
        }

        /* Móviles muy pequeños */
        @media (max-width: 360px) {
            .header-left h1 {
                font-size: 18px;
            }

            .stat-card {
                padding: 12px;
            }

            .stat-value {
                font-size: 22px;
            }

            .card {
                padding: 12px;
            }
        }

        /* Landscape mode en móviles */
        @media (max-width: 768px) and (orientation: landscape) {
            .stats-grid {
                grid-template-columns: repeat(2, 1fr);
            }

            .header {
                padding: 10px 15px;
            }

            .content {
                padding: 15px;
            }
        }
    </style>
</head>
<body data-theme="light">
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <h2>
                <i class="fas fa-shield-alt"></i>
                Azuero Safety Services S.A
            </h2>
            <p style="font-size: 11px; opacity: 0.85; margin-top: 5px;">Somos tu seguridad</p>
        </div>

        <div class="sidebar-menu">
            <a href="dashboard.php" class="menu-item active">
                <i class="fas fa-home"></i>
                <span><?php echo $t['dashboard']; ?></span>
            </a>

            <div class="menu-section">Alertas</div>
            
            <a href="alertas.php" class="menu-item">
                <i class="fas fa-bell"></i>
                <span><?php echo $t['all_alerts']; ?></span>
            </a>

            <a href="alertas.php?assigned=me" class="menu-item">
                <i class="fas fa-tasks"></i>
                <span><?php echo $t['my_alerts']; ?></span>
            </a>

            <?php if ($operator['role'] === 'admin'): ?>
            <div class="menu-section">Administración</div>
            
            <a href="admin-panel.php" class="menu-item">
                <i class="fas fa-user-shield"></i>
                <span>Panel Admin</span>
            </a>
            <?php endif; ?>

            <div class="menu-section">Sistema</div>

            <a href="logout.php" class="menu-item">
                <i class="fas fa-sign-out-alt"></i>
                <span><?php echo $t['logout']; ?></span>
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Header -->
        <div class="header">
            <div class="header-left">
                <h1><?php echo $t['dashboard']; ?></h1>
                <div class="breadcrumb">
                    <i class="fas fa-home"></i> / <?php echo $t['dashboard']; ?>
                </div>
            </div>

            <div class="header-right">
                <div class="search-box">
                    <input type="text" placeholder="<?php echo $t['search']; ?>" id="searchInput">
                    <i class="fas fa-search"></i>
                </div>

                <div class="notification-bell" onclick="checkNewAlerts()">
                    <i class="fas fa-bell"></i>
                    <span class="notification-badge" id="notificationCount"><?php echo $stats['active_alerts']; ?></span>
                </div>

                <div class="theme-switcher">
                    <button class="theme-btn active" data-theme="light" onclick="changeTheme('light')">
                        ☀️
                    </button>
                    <button class="theme-btn" data-theme="dark" onclick="changeTheme('dark')">
                        🌙
                    </button>
                    <button class="theme-btn" data-theme="blue-dark" onclick="changeTheme('blue-dark')">
                        🌊
                    </button>
                </div>

                <div class="lang-switcher">
                    <button class="lang-btn <?php echo $lang === 'es' ? 'active' : ''; ?>" onclick="changeLanguage('es')">
                        🇪🇸 ES
                    </button>
                    <button class="lang-btn <?php echo $lang === 'en' ? 'active' : ''; ?>" onclick="changeLanguage('en')">
                        🇺🇸 EN
                    </button>
                </div>

                <div class="user-profile">
                    <div class="user-avatar">
                        <?php echo strtoupper(substr($operator['full_name'], 0, 1)); ?>
                    </div>
                    <div class="user-info">
                        <h4><?php echo htmlspecialchars($operator['full_name']); ?></h4>
                        <p><?php echo ucfirst($operator['role']); ?></p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Toast Container -->
        <div class="toast-container" id="toastContainer"></div>

        <!-- Content -->
        <div class="content">
            <!-- Stats Cards -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-header">
                        <div>
                            <div class="stat-value" id="activeAlertsCount"><?php echo $stats['active_alerts']; ?></div>
                            <div class="stat-label"><?php echo $t['active_alerts']; ?></div>
                        </div>
                        <div class="stat-icon orange">
                            <i class="fas fa-exclamation-circle"></i>
                        </div>
                    </div>
                </div>

                <div class="stat-card">
                    <div class="stat-header">
                        <div>
                            <div class="stat-value" id="pendingCount"><?php echo $stats['pending']; ?></div>
                            <div class="stat-label"><?php echo $t['pending']; ?></div>
                        </div>
                        <div class="stat-icon red">
                            <i class="fas fa-clock"></i>
                        </div>
                    </div>
                </div>

                <div class="stat-card">
                    <div class="stat-header">
                        <div>
                            <div class="stat-value" id="inProgressCount"><?php echo $stats['in_progress']; ?></div>
                            <div class="stat-label"><?php echo $t['in_progress']; ?></div>
                        </div>
                        <div class="stat-icon blue">
                            <i class="fas fa-spinner"></i>
                        </div>
                    </div>
                </div>

                <div class="stat-card">
                    <div class="stat-header">
                        <div>
                            <div class="stat-value" id="resolvedTodayCount"><?php echo $stats['resolved_today']; ?></div>
                            <div class="stat-label"><?php echo $t['resolved_today']; ?></div>
                        </div>
                        <div class="stat-icon green">
                            <i class="fas fa-check-circle"></i>
                        </div>
                    </div>
                </div>

                <div class="stat-card">
                    <div class="stat-header">
                        <div>
                            <div class="stat-value" id="myAssignmentsCount"><?php echo $stats['my_assignments']; ?></div>
                            <div class="stat-label"><?php echo $t['my_assignments']; ?></div>
                        </div>
                        <div class="stat-icon purple">
                            <i class="fas fa-user-check"></i>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Recent Alerts Table -->
            <div class="table-card">
                <div class="table-header">
                    <h3><?php echo $t['recent_alerts']; ?></h3>
                    <a href="alertas.php" class="btn btn-primary">
                        <i class="fas fa-list"></i> <?php echo $t['view_all']; ?>
                    </a>
                </div>
                <div class="table-wrapper">
                    <table id="alertsTable">
                        <thead>
                            <tr>
                                <th><?php echo $t['id']; ?></th>
                                <th><?php echo $t['citizen']; ?></th>
                                <th><?php echo $t['type']; ?></th>
                                <th>Mensaje</th>
                                <th><?php echo $t['priority']; ?></th>
                                <th><?php echo $t['status']; ?></th>
                                <th><?php echo $t['date']; ?></th>
                                <th><?php echo $t['actions']; ?></th>
                            </tr>
                        </thead>
                        <tbody id="alertsTableBody">
                            <?php if (empty($recent_alerts)): ?>
                            <tr>
                                <td colspan="8" style="text-align: center; padding: 40px;">
                                    <i class="fas fa-inbox" style="font-size: 48px; color: var(--text-secondary); opacity: 0.5;"></i>
                                    <p style="margin-top: 15px; color: var(--text-secondary);"><?php echo $t['no_alerts']; ?></p>
                                </td>
                            </tr>
                            <?php else: ?>
                                <?php foreach($recent_alerts as $alert): ?>
                                <tr>
                                    <td>#<?php echo $alert['id']; ?></td>
                                    <td>
                                        <div style="display: flex; align-items: center; gap: 10px;">
                                            <div class="user-avatar" style="width: 32px; height: 32px; font-size: 12px;">
                                                <?php echo strtoupper(substr($alert['user_name'] ?? 'U', 0, 1)); ?>
                                            </div>
                                            <div>
                                                <div style="font-weight: 500;"><?php echo htmlspecialchars($alert['user_name'] ?? $t['unknown']); ?></div>
                                                <div style="font-size: 12px; color: var(--text-secondary);"><?php echo htmlspecialchars($alert['user_email'] ?? ''); ?></div>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <?php
                                        $type_badges = [
                                            'medical' => ['icon' => 'fa-heartbeat', 'color' => 'danger', 'text' => $t['medical']],
                                            'fire' => ['icon' => 'fa-fire', 'color' => 'warning', 'text' => $t['fire']],
                                            'security' => ['icon' => 'fa-shield-alt', 'color' => 'info', 'text' => $t['security']]
                                        ];
                                        $badge = $type_badges[$alert['alert_type']] ?? ['icon' => 'fa-bell', 'color' => 'info', 'text' => $alert['alert_type']];
                                        ?>
                                        <span class="badge badge-<?php echo $badge['color']; ?>">
                                            <i class="fas <?php echo $badge['icon']; ?>"></i> <?php echo $badge['text']; ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if (!empty($alert['description'])): ?>
                                            <div style="max-width: 200px; overflow: hidden; text-overflow: ellipsis; white-space: nowrap; font-size: 13px; color: var(--text-secondary);" title="<?php echo htmlspecialchars($alert['description']); ?>">
                                                <i class="fas fa-comment-dots" style="color: var(--danger);"></i>
                                                <?php echo htmlspecialchars(substr($alert['description'], 0, 50)) . (strlen($alert['description']) > 50 ? '...' : ''); ?>
                                            </div>
                                        <?php else: ?>
                                            <span style="font-size: 12px; color: var(--text-secondary); opacity: 0.5;">—</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php
                                        $priority_colors = ['high' => 'danger', 'medium' => 'warning', 'low' => 'info'];
                                        $priority_text = ['high' => $t['high'], 'medium' => $t['medium'], 'low' => $t['low']];
                                        $color = $priority_colors[$alert['priority']] ?? 'info';
                                        $text = $priority_text[$alert['priority']] ?? strtoupper($alert['priority']);
                                        ?>
                                        <span class="badge badge-<?php echo $color; ?>"><?php echo $text; ?></span>
                                    </td>
                                    <td>
                                        <?php
                                        $status_colors = ['pending' => 'warning', 'in_progress' => 'info', 'resolved' => 'success'];
                                        $status_text = ['pending' => $t['pending_status'], 'in_progress' => $t['in_progress_status'], 'resolved' => $t['resolved']];
                                        $color = $status_colors[$alert['status']] ?? 'info';
                                        $text = $status_text[$alert['status']] ?? strtoupper($alert['status']);
                                        ?>
                                        <span class="badge badge-<?php echo $color; ?>"><?php echo $text; ?></span>
                                    </td>
                                    <td><?php echo date('d/m/Y H:i', strtotime($alert['created_at'])); ?></td>
                                    <td>
                                        <div style="display: flex; gap: 5px;">
                                            <a href="ver-alerta.php?id=<?php echo $alert['id']; ?>" class="btn btn-primary btn-sm" title="Ver Alerta">
                                                <i class="fas fa-eye"></i>
                                            </a>
                                            <a href="ver-historial-medico.php?user_id=<?php echo $alert['user_id']; ?>" class="btn btn-primary btn-sm" title="Historial Médico" style="background: #ef4444;">
                                                <i class="fas fa-file-medical"></i>
                                            </a>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Alarm Sound -->
    <audio id="alarmSound" preload="auto">
        <source src="data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmwhBTGJ0fPTgjMGHm7A7+OZSA0PVKrl7K1aFgpBmeHyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgpBmeDyvmwhBTGI0fPTgjMGHm/B7+OYRw0PVKvl7K1aFgo=" type="audio/wav">
    </audio>

    <script>
        // Theme Management
        function changeTheme(theme) {
            document.body.setAttribute('data-theme', theme);
            localStorage.setItem('theme', theme);

            document.querySelectorAll('.theme-btn').forEach(btn => {
                btn.classList.remove('active');
                if(btn.getAttribute('data-theme') === theme) {
                    btn.classList.add('active');
                }
            });
        }

        // Load saved theme
        const savedTheme = localStorage.getItem('theme') || 'light';
        changeTheme(savedTheme);

        // Language Management
        function changeLanguage(lang) {
            // Reload page with new language parameter
            const url = new URL(window.location.href);
            url.searchParams.set('lang', lang);
            window.location.href = url.toString();
        }
        window.changeLanguage = changeLanguage;

        // Notification state and utilities
        var lastCheckTs = parseInt(localStorage.getItem('lastCheckTs') || '0', 10) || Math.floor(Date.now()/1000);
        var notifiedAlerts = [];
        try { notifiedAlerts = JSON.parse(localStorage.getItem('notifiedAlerts') || '[]'); } catch(e) { notifiedAlerts = []; }
        function saveNotified() {
            try { localStorage.setItem('notifiedAlerts', JSON.stringify(notifiedAlerts.slice(-100))); } catch(e) {}
        }
        var lastAlarmMs = 0;
        var alarmInterval = null;
        var alarmActive = false;

        // Crear sonido de sirena con Web Audio API (más fuerte y realista)
        var audioContext = null;
        function playSirenSound() {
            try {
                if (!audioContext) {
                    audioContext = new (window.AudioContext || window.webkitAudioContext)();
                }

                var oscillator = audioContext.createOscillator();
                var gainNode = audioContext.createGain();

                oscillator.connect(gainNode);
                gainNode.connect(audioContext.destination);

                // Sirena de emergencia: alterna entre dos frecuencias
                oscillator.frequency.setValueAtTime(800, audioContext.currentTime);
                oscillator.frequency.setValueAtTime(400, audioContext.currentTime + 0.25);
                oscillator.frequency.setValueAtTime(800, audioContext.currentTime + 0.5);
                oscillator.frequency.setValueAtTime(400, audioContext.currentTime + 0.75);

                // Volumen alto
                gainNode.gain.setValueAtTime(0.7, audioContext.currentTime);
                gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 1);

                oscillator.start(audioContext.currentTime);
                oscillator.stop(audioContext.currentTime + 1);

                console.log('🚨 Siren sound played');
            } catch(err) {
                console.log('Web Audio API error:', err);
            }
        }
        
        function playAlarmThrottled() {
            var now = Date.now();
            if (now - lastAlarmMs > 8000) {
                lastAlarmMs = now;

                // Reproducir sonido de sirena (Web Audio API - más fuerte)
                playSirenSound();

                // Fallback: reproducir audio HTML5
                var a = document.getElementById('alarmSound');
                if (a) {
                    try {
                        a.currentTime = 0;
                        a.volume = 1.0; // Volumen máximo
                        a.play().catch(function(err) {
                            console.log('Audio play failed:', err);
                        });
                    } catch(err) {
                        console.log('Audio error:', err);
                    }
                }

                // Vibrar el celular (3 pulsos fuertes: 300ms vibrar, 100ms pausa, 300ms vibrar, 100ms pausa, 300ms vibrar)
                if ('vibrate' in navigator) {
                    try {
                        navigator.vibrate([300, 100, 300, 100, 300]);
                        console.log('📳 Vibration triggered');
                    } catch(err) {
                        console.log('Vibration error:', err);
                    }
                }
            }
        }
        
        // Nueva función para alarma persistente
        function startPersistentAlarm() {
            if (alarmActive) return; // Ya está sonando
            
            alarmActive = true;
            var a = document.getElementById('alarmSound');
            
            // Reproducir inmediatamente
            if (a) {
                try { 
                    a.currentTime = 0; 
                    a.play(); 
                } catch(err) {
                    console.warn('No se pudo reproducir alarma:', err);
                }
            }
            
            // Repetir cada 3 segundos hasta que se detenga
            alarmInterval = setInterval(function() {
                if (a && alarmActive) {
                    try { 
                        a.currentTime = 0; 
                        a.play(); 
                    } catch(err) {}
                }
            }, 3000);
        }
        
        function stopAlarm() {
            alarmActive = false;
            if (alarmInterval) {
                clearInterval(alarmInterval);
                alarmInterval = null;
            }
            var a = document.getElementById('alarmSound');
            if (a) {
                try { a.pause(); a.currentTime = 0; } catch(err) {}
            }
        }

        // Toast Notifications
        function showToast(type, title, message) {
            const toast = document.createElement('div');
            toast.className = 'toast ' + type;
            
            const icons = {
                'emergency': 'fa-exclamation-circle',
                'success': 'fa-check-circle'
            };
            
            // Botón para detener alarma en notificaciones de emergencia
            var stopAlarmBtn = '';
            if (type === 'emergency') {
                stopAlarmBtn = `
                    <button class="btn btn-sm btn-danger" onclick="stopAlarm(); this.disabled=true; this.innerHTML='<i class=\\'fas fa-check\\'></i> Silenciada';" 
                            style="margin: 10px 0 5px 0; width: 100%; font-size: 12px;">
                        <i class="fas fa-volume-mute"></i> Detener Alarma
                    </button>`;
            }
            
            toast.innerHTML = `
                <div class="toast-icon">
                    <i class="fas ${icons[type]}"></i>
                </div>
                <div class="toast-content">
                    <h4>${title}</h4>
                    <p>${message}</p>
                    ${stopAlarmBtn}
                </div>
                <button class="toast-close" onclick="this.parentElement.remove()">
                    <i class="fas fa-times"></i>
                </button>
            `;
            
            document.getElementById('toastContainer').appendChild(toast);
            
            if(type === 'emergency') {
                startPersistentAlarm(); // Usar nueva función de alarma persistente
            }
            
            setTimeout(() => {
                toast.style.animation = 'slideIn 0.3s ease reverse';
                setTimeout(() => toast.remove(), 300);
            }, 5000);
        }

        // Check for new alerts
        function checkNewAlerts() {
            var url = 'check-new-alerts.php?last_check=' + encodeURIComponent(lastCheckTs || 0);
            
            console.log('[Dashboard] 🔄 Checking alerts... last_check=' + lastCheckTs);
            
            fetch(url)
                .then(response => response.json())
                .then(data => {
                    console.log('[Dashboard] 📦 Response:', data);
                    
                    if(!data || data.error) {
                        console.error('[Dashboard] ❌ Error:', data ? data.error : 'No data');
                        return;
                    }
                    
                    var newAlerts = (data.alerts || []).filter(function(a){
                        return a && a.timestamp && a.timestamp > (lastCheckTs || 0) && notifiedAlerts.indexOf(a.id) === -1;
                    });
                    
                    console.log('[Dashboard] 🔍 New alerts filtered:', newAlerts.length);
                    
                    if(newAlerts.length > 0) {
                        console.log('[Dashboard] 🚨 NEW ALERT DETECTED!', newAlerts[0]);
                        
                        if (data.stats) {
                            var totalActive = data.stats.pending + data.stats.in_progress;
                            var el;
                            if (el = document.getElementById('notificationCount')) el.textContent = totalActive;
                            if (el = document.getElementById('activeAlertsCount')) el.textContent = totalActive;
                            if (el = document.getElementById('pendingCount')) el.textContent = data.stats.pending;
                            if (el = document.getElementById('inProgressCount')) el.textContent = data.stats.in_progress;
                        }
                        var alert = newAlerts[0];
                        function esc(s){return String(s||'').replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;').replace(/"/g,'&quot;').replace(/'/g,'&#39;');}
                        var desc = alert.description ? esc(alert.description).slice(0,140) : '';
                        var extra = desc ? ' — "' + desc + (alert.description.length>140 ? '…' : '') + '"' : '';
                        
                        // Agregar información de ubicación GPS si existe
                        var gpsInfo = '';
                        if (alert.latitude && alert.longitude) {
                            gpsInfo = '<br><small style="font-size:12px;opacity:0.9;"><i class="fas fa-map-marker-alt"></i> GPS: ' + 
                                     parseFloat(alert.latitude).toFixed(6) + ', ' + 
                                     parseFloat(alert.longitude).toFixed(6) + '</small>';
                        }
                        
                        var linkHtml = ' <a href="ver-alerta.php?id=' + alert.id + '" class="btn btn-primary btn-sm" style="margin-left:8px;text-decoration:none;">Ver alerta</a>';
                        showToast('emergency', '¡Nueva Alerta!', 'Se ha recibido una alerta ' + esc(alert.alert_type || 'emergencia') + extra + '.' + gpsInfo + linkHtml);
                        newAlerts.forEach(function(a){ notifiedAlerts.push(a.id); });
                        saveNotified();
                    }
                    if (typeof data.timestamp === 'number') {
                        lastCheckTs = data.timestamp;
                    } else {
                        lastCheckTs = Math.floor(Date.now()/1000);
                    }
                    try { localStorage.setItem('lastCheckTs', String(lastCheckTs)); } catch(e) {}
                })
                .catch(error => {
                    console.error('[Dashboard] ❌ Fetch error:', error);
                });
        }

        // Poll every 3 seconds - Iniciar automáticamente
        setInterval(checkNewAlerts, 3000);
        checkNewAlerts(); // Primera llamada inmediata

        // Search functionality
        document.getElementById('searchInput')?.addEventListener('input', function(e) {
            const searchTerm = e.target.value.toLowerCase();
            const rows = document.querySelectorAll('#alertsTableBody tr');
            
            rows.forEach(row => {
                const text = row.textContent.toLowerCase();
                row.style.display = text.includes(searchTerm) ? '' : 'none';
            });
        });
        
        // ============================================
        // PWA: Registro del Service Worker
        // ============================================
        if ('serviceWorker' in navigator) {
            navigator.serviceWorker.register('service-worker.js')
                .then((registration) => {
                    console.log('[PWA] Service Worker registrado:', registration.scope);
                })
                .catch((error) => {
                    console.error('[PWA] Error al registrar Service Worker:', error);
                });
        }
    </script>
</body>
</html>
