<?php
require_once 'config.php';

// Multi-language support - Default English for US customers
$lang = $_SESSION['lang'] ?? 'en';

// Guardar idioma en sesión
if (isset($_GET['lang'])) {
    $_SESSION['lang'] = $_GET['lang'];
    header('Location: index.php');
    exit;
}

$translations = [
    'es' => [
        'title' => 'Azuero Safety',
        'subtitle' => 'Sistema de Respuesta a Emergencias',
        'email' => 'Email',
        'username' => 'Usuario',
        'password' => 'Contraseña',
        'remember' => 'Recordarme',
        'login' => 'Iniciar Sesión',
        'secure' => 'Conexión segura y cifrada',
        'access_note' => 'Ingresa con las credenciales proporcionadas por el administrador',
        'error_credentials' => 'Usuario o contraseña incorrectos',
        'error_empty' => 'Por favor ingresa todos los campos requeridos',
        'enter_email' => 'Ingresa tu email',
        'enter_username' => 'Ingresa tu usuario',
        'enter_password' => 'Ingresa tu contraseña',
        'copyright' => 'Todos los derechos reservados',
        'emergency_system' => 'Sistema de Respuesta a Emergencias'
    ],
    'en' => [
        'title' => 'Azuero Safety',
        'subtitle' => 'Emergency Response System',
        'email' => 'Email',
        'username' => 'Username',
        'password' => 'Password',
        'remember' => 'Remember me',
        'login' => 'Sign In',
        'secure' => 'Secure and encrypted connection',
        'access_note' => 'Enter with credentials provided by administrator',
        'error_credentials' => 'Incorrect credentials',
        'error_empty' => 'Please enter all required fields',
        'enter_email' => 'Enter your email',
        'enter_username' => 'Enter your username',
        'enter_password' => 'Enter your password',
        'copyright' => 'All rights reserved',
        'emergency_system' => 'Emergency Response System'
    ]
];

$t = $translations[$lang];

$error = '';

// Check for remember me cookie
if (!isset($_SESSION['user_id']) && isset($_COOKIE['remember_token'])) {
    try {
        $token = $_COOKIE['remember_token'];
        // Try operators first
        $stmt = $conn->prepare("SELECT id, username, full_name, email, role, 'operator' as user_type FROM operators WHERE remember_token = ? AND status = 'active'");
        $stmt->execute([$token]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            // Try citizens
            $stmt = $conn->prepare("SELECT id, username, full_name, email, 'citizen' as user_type FROM users WHERE remember_token = ? AND status = 'active'");
            $stmt->execute([$token]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
        }
        
        if ($user) {
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['user_type'] = $user['user_type'];
            $_SESSION['full_name'] = $user['full_name'];
            $_SESSION['lang'] = $lang;
            
            if ($user['user_type'] == 'operator') {
                $_SESSION['operator_id'] = $user['id'];
                $_SESSION['role'] = $user['role'];
                header('Location: ' . ($user['role'] === 'admin' ? 'admin-panel.php' : 'dashboard.php'));
                exit();
            } else {
                header('Location: panel-ciudadano.php');
                exit();
            }
        }
    } catch(PDOException $e) {
        // Ignore cookie errors
    }
}

$error = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $email = trim($_POST['email'] ?? '');
    $username = trim($_POST['username'] ?? '');
    $password = trim($_POST['password'] ?? '');
    $remember = isset($_POST['remember']);
    
    if (empty($email) || empty($username) || empty($password)) {
        $error = $t['error_empty'];
    } else {
        try {
            // Primero buscar en operadores (validar email Y username)
            $stmt = $conn->prepare("
                SELECT 
                    id as operator_id,
                    username,
                    password,
                    full_name as name,
                    email,
                    role,
                    status,
                    'operator' as user_type 
                FROM operators 
                WHERE email = ? AND username = ? AND status = 'active'
            ");
            $stmt->execute([$email, $username]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Si no es operador, buscar en usuarios (ciudadanos)
            if (!$user) {
                $stmt = $conn->prepare("
                    SELECT 
                        id,
                        username,
                        full_name,
                        email,
                        phone,
                        password,
                        status,
                        blood_type,
                        emergency_contact,
                        'citizen' as user_type 
                    FROM users 
                    WHERE email = ? AND username = ? AND status = 'active'
                ");
                $stmt->execute([$email, $username]);
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
            }
            
            if ($user && password_verify($password, $user['password'])) {
                $_SESSION['username'] = $username;
                $_SESSION['user_type'] = $user['user_type'];
                $_SESSION['lang'] = $lang;
                
                // Implementar Remember Me
                if ($remember) {
                    $token = bin2hex(random_bytes(32));
                    $expiry = time() + (30 * 24 * 60 * 60); // 30 días
                    
                    // Guardar token en BD
                    if ($user['user_type'] == 'operator') {
                        try {
                            $updateStmt = $conn->prepare("UPDATE operators SET remember_token = ? WHERE id = ?");
                            $updateStmt->execute([$token, $user['operator_id']]);
                        } catch (PDOException $e) {
                            // Ignorar si la columna no existe
                        }
                    } else {
                        try {
                            $updateStmt = $conn->prepare("UPDATE users SET remember_token = ? WHERE id = ?");
                            $updateStmt->execute([$token, $user['id']]);
                        } catch (PDOException $e) {
                            // Ignorar si la columna no existe
                        }
                    }
                    
                    // Establecer cookie
                    setcookie('remember_token', $token, $expiry, '/', '', false, true);
                }
                
                if ($user['user_type'] == 'operator') {
                    $_SESSION['operator_id'] = $user['operator_id'];
                    $_SESSION['user_id'] = $user['operator_id'];
                    $_SESSION['full_name'] = $user['name'];
                    $_SESSION['role'] = $user['role'];
                    $_SESSION['email'] = $user['email'];
                    
                    // Actualizar último login (solo si la columna existe)
                    try {
                        $updateStmt = $conn->prepare("UPDATE operators SET last_login = NOW() WHERE id = ?");
                        $updateStmt->execute([$user['operator_id']]);
                    } catch (PDOException $e) {
                        // Ignorar error si la columna no existe
                    }
                    
                    // Redirigir según el rol
                    if ($user['role'] === 'admin') {
                        header('Location: admin-panel.php');
                        exit();
                    } else {
                        header('Location: dashboard.php');
                        exit();
                    }
                } else {
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['full_name'] = $user['full_name'];
                    
                    // Ciudadano - verificar si completó su perfil
                    if (empty($user['blood_type']) || empty($user['emergency_contact'])) {
                        header('Location: completar-perfil.php');
                        exit();
                    } else {
                        header('Location: panel-ciudadano.php');
                        exit();
                    }
                }
            } else {
                $error = $t['error_credentials'];
            }
        } catch(PDOException $e) {
            $error = 'Error: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <meta name="theme-color" content="#667eea">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="black-translucent">
    <meta name="apple-mobile-web-app-title" content="Azuero Safety">
    <title><?php echo $t['title']; ?> - Login</title>

    <!-- PWA Manifest -->
    <link rel="manifest" href="manifest.json">

    <!-- Icons -->
    <link rel="icon" type="image/png" sizes="192x192" href="icon-192x192.png">
    <link rel="apple-touch-icon" href="icon-192x192.png">

    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        @keyframes gradientShift {
            0% { background-position: 0% 50%; }
            50% { background-position: 100% 50%; }
            100% { background-position: 0% 50%; }
        }
        
        @keyframes fadeInUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        @keyframes pulse {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.05); }
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: linear-gradient(-45deg, #667eea, #764ba2, #f093fb, #4facfe);
            background-size: 400% 400%;
            animation: gradientShift 15s ease infinite;
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
            position: relative;
            overflow: hidden;
        }
        
        body::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: 
                radial-gradient(circle at 20% 50%, rgba(255, 255, 255, 0.1) 0%, transparent 50%),
                radial-gradient(circle at 80% 80%, rgba(255, 255, 255, 0.1) 0%, transparent 50%);
            pointer-events: none;
        }
        
        .login-container {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border-radius: 24px;
            box-shadow: 
                0 20px 60px rgba(0, 0, 0, 0.3),
                0 0 0 1px rgba(255, 255, 255, 0.1) inset;
            padding: 50px 45px;
            width: 100%;
            max-width: 460px;
            position: relative;
            animation: fadeInUp 0.6s ease-out;
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
        
        .lang-switcher {
            position: absolute;
            top: 20px;
            right: 20px;
            display: flex;
            gap: 8px;
            z-index: 10;
        }
        
        .lang-btn {
            padding: 8px 14px;
            border: 2px solid rgba(102, 126, 234, 0.2);
            background: rgba(255, 255, 255, 0.9);
            backdrop-filter: blur(10px);
            border-radius: 10px;
            cursor: pointer;
            font-size: 12px;
            font-weight: 600;
            transition: all 0.3s ease;
            text-decoration: none;
            color: #667eea;
            display: flex;
            align-items: center;
            gap: 4px;
        }
        
        .lang-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(102, 126, 234, 0.3);
            background: rgba(255, 255, 255, 1);
        }
        
        .lang-btn.active {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-color: transparent;
            box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
        }
        
        .logo { 
            text-align: center; 
            margin-bottom: 35px;
            padding-top: 10px;
        }
        
        .logo-icon { 
            font-size: 70px; 
            margin-bottom: 15px;
            animation: pulse 3s ease-in-out infinite;
            filter: drop-shadow(0 4px 10px rgba(0, 0, 0, 0.1));
        }
        
        h1 { 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            text-align: center; 
            margin-bottom: 8px; 
            font-size: 32px;
            font-weight: 700;
            letter-spacing: -0.5px;
        }
        
        .subtitle { 
            color: #6b7280; 
            text-align: center; 
            margin-bottom: 40px; 
            font-size: 15px;
            font-weight: 500;
        }
        
        .form-group { 
            margin-bottom: 22px;
            position: relative;
        }
        
        label {
            display: block;
            color: #374151;
            font-weight: 600;
            margin-bottom: 10px;
            font-size: 14px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        label .icon {
            font-size: 16px;
            opacity: 0.7;
        }
        
        .input-wrapper {
            position: relative;
        }
        
        input[type="text"], 
        input[type="password"],
        input[type="email"] {
            width: 100%;
            padding: 14px 18px;
            padding-left: 45px;
            border: 2px solid #e5e7eb;
            border-radius: 12px;
            font-size: 15px;
            transition: all 0.3s ease;
            background: white;
            color: #1f2937;
            font-weight: 500;
        }
        
        input[type="text"]:focus, 
        input[type="password"]:focus,
        input[type="email"]:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 4px rgba(102, 126, 234, 0.1);
            transform: translateY(-1px);
        }
        
        .input-icon {
            position: absolute;
            left: 16px;
            top: 50%;
            transform: translateY(-50%);
            font-size: 18px;
            color: #9ca3af;
            pointer-events: none;
            transition: color 0.3s;
        }
        
        input:focus + .input-icon {
            color: #667eea;
        }
        
        .remember-group {
            display: flex;
            align-items: center;
            gap: 10px;
            margin-bottom: 25px;
        }
        
        .remember-group input[type="checkbox"] {
            width: 18px;
            height: 18px;
            border: 2px solid #d1d5db;
            border-radius: 5px;
            cursor: pointer;
            accent-color: #667eea;
        }
        
        .remember-group label {
            margin: 0;
            font-weight: 500;
            font-size: 14px;
            color: #6b7280;
            cursor: pointer;
        }
        
        .btn-login {
            width: 100%;
            padding: 16px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4);
            position: relative;
            overflow: hidden;
        }
        
        .btn-login::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.3), transparent);
            transition: left 0.5s;
        }
        
        .btn-login:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.5);
        }
        
        .btn-login:hover::before {
            left: 100%;
        }
        
        .btn-login:active {
            transform: translateY(0);
        }
        
        .error-message {
            background: linear-gradient(135deg, #fee2e2 0%, #fecaca 100%);
            color: #991b1b;
            padding: 14px 18px;
            border-radius: 12px;
            margin-bottom: 25px;
            font-size: 14px;
            text-align: center;
            border: 1px solid #fca5a5;
            font-weight: 500;
            animation: fadeInUp 0.3s ease-out;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }
        
        .error-message::before {
            content: '⚠️';
            font-size: 18px;
        }
        
        .footer { 
            text-align: center; 
            margin-top: 30px; 
            color: #6b7280; 
            font-size: 12px;
            line-height: 1.8;
        }
        
        .footer p {
            margin: 6px 0;
        }
        
        .secure-badge {
            display: inline-flex;
            align-items: center;
            gap: 6px;
            background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
            color: #065f46;
            padding: 8px 14px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            margin-top: 15px;
            border: 1px solid #6ee7b7;
        }
        
        .divider {
            display: flex;
            align-items: center;
            text-align: center;
            margin: 25px 0;
            color: #9ca3af;
            font-size: 13px;
        }
        
        .divider::before,
        .divider::after {
            content: '';
            flex: 1;
            border-bottom: 1px solid #e5e7eb;
        }
        
        .divider span {
            padding: 0 15px;
            font-weight: 500;
        }
        
        /* Responsive - Tablets */
        @media (max-width: 768px) {
            body {
                padding: 15px;
            }

            .login-container {
                padding: 40px 35px;
                max-width: 500px;
            }

            .lang-switcher {
                top: 15px;
                right: 15px;
            }

            h1 {
                font-size: 28px;
            }

            .subtitle {
                font-size: 15px;
            }

            .logo-icon {
                font-size: 65px;
            }
        }

        /* Responsive - Móviles */
        @media (max-width: 480px) {
            body {
                padding: 10px;
                align-items: flex-start;
                padding-top: 20px;
            }

            .login-container {
                padding: 35px 25px;
                max-width: 100%;
                border-radius: 20px;
            }

            .lang-switcher {
                top: 12px;
                right: 12px;
                gap: 6px;
            }

            .lang-btn {
                padding: 6px 10px;
                font-size: 11px;
                border-radius: 8px;
            }

            .logo {
                margin-bottom: 25px;
            }

            h1 {
                font-size: 24px;
                margin-bottom: 8px;
            }

            .subtitle {
                font-size: 14px;
            }

            .logo-icon {
                font-size: 55px;
                margin-bottom: 12px;
            }

            .form-group {
                margin-bottom: 18px;
            }

            label {
                font-size: 13px;
                margin-bottom: 6px;
            }

            input[type="text"],
            input[type="password"],
            input[type="email"] {
                font-size: 16px; /* Previene zoom en iOS */
                padding: 14px 16px 14px 45px;
                height: 50px;
            }

            .input-icon {
                font-size: 18px;
            }

            .btn-login {
                padding: 16px;
                font-size: 16px;
                height: 54px;
            }

            .remember-group {
                margin: 18px 0;
            }

            .remember-group label {
                font-size: 14px;
            }

            .error-message,
            .success-message {
                padding: 14px;
                font-size: 14px;
                margin-bottom: 18px;
            }

            .divider {
                margin: 20px 0;
                font-size: 12px;
            }

            .footer {
                font-size: 11px;
                margin-top: 20px;
            }
        }

        /* Responsive - Móviles pequeños */
        @media (max-width: 360px) {
            .login-container {
                padding: 30px 20px;
            }

            h1 {
                font-size: 22px;
            }

            .logo-icon {
                font-size: 50px;
            }

            input[type="text"],
            input[type="password"],
            input[type="email"] {
                padding: 12px 14px 12px 40px;
                height: 48px;
            }

            .btn-login {
                padding: 14px;
                font-size: 15px;
                height: 50px;
            }
        }

        /* Landscape mode en móviles */
        @media (max-width: 768px) and (orientation: landscape) {
            body {
                padding-top: 10px;
                padding-bottom: 10px;
            }

            .login-container {
                padding: 25px 30px;
            }

            .logo {
                margin-bottom: 15px;
            }

            .logo-icon {
                font-size: 45px;
                margin-bottom: 8px;
            }

            h1 {
                font-size: 22px;
                margin-bottom: 5px;
            }

            .subtitle {
                font-size: 13px;
            }

            .form-group {
                margin-bottom: 12px;
            }

            input[type="text"],
            input[type="password"],
            input[type="email"] {
                height: 44px;
                padding: 10px 14px 10px 40px;
            }

            .btn-login {
                height: 46px;
                padding: 12px;
            }

            .remember-group {
                margin: 12px 0;
            }

            .divider {
                margin: 15px 0;
            }
        }
    </style>
</head>
<body>
    <div class="login-container">
        <div class="lang-switcher">
            <a href="?lang=es" class="lang-btn <?php echo $lang == 'es' ? 'active' : ''; ?>">🇪🇸 ES</a>
            <a href="?lang=en" class="lang-btn <?php echo $lang == 'en' ? 'active' : ''; ?>">🇺🇸 EN</a>
        </div>
        
        <div class="logo">
            <div class="logo-icon">🚨</div>
            <h1><?php echo $t['title']; ?></h1>
            <p class="subtitle"><?php echo $t['subtitle']; ?></p>
        </div>
        
        <?php if ($error): ?>
        <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <form method="POST" action="">
            <div class="form-group">
                <label for="email">
                    <span class="icon">📧</span> <?php echo $t['email']; ?>
                </label>
                <div class="input-wrapper">
                    <input type="email" id="email" name="email" placeholder="<?php echo $t['enter_email']; ?>" required autofocus>
                    <div class="input-icon">📧</div>
                </div>
            </div>
            
            <div class="form-group">
                <label for="username">
                    <span class="icon">👤</span> <?php echo $t['username']; ?>
                </label>
                <div class="input-wrapper">
                    <input type="text" id="username" name="username" placeholder="<?php echo $t['enter_username']; ?>" required>
                    <div class="input-icon">👤</div>
                </div>
            </div>
            
            <div class="form-group">
                <label for="password">
                    <span class="icon">🔒</span> <?php echo $t['password']; ?>
                </label>
                <div class="input-wrapper">
                    <input type="password" id="password" name="password" placeholder="<?php echo $t['enter_password']; ?>" required>
                    <div class="input-icon">🔒</div>
                </div>
            </div>
            
            <div class="remember-group">
                <input type="checkbox" id="remember" name="remember">
                <label for="remember"><?php echo $t['remember']; ?></label>
            </div>
            
            <button type="submit" class="btn-login">
                <i class="fas fa-sign-in-alt"></i> <?php echo $t['login']; ?>
            </button>
            
            <div class="divider">
                <span><?php echo $t['secure']; ?></span>
            </div>
        </form>
        
        <div class="footer">
            <div class="secure-badge">
                🔒 256-bit SSL Encryption
            </div>
            <p style="margin-top: 20px;">ℹ️ <?php echo $t['access_note']; ?></p>
            <p style="margin-top: 15px; color: #999; font-weight: 600;">&copy; <?php echo date('Y'); ?> <?php echo $t['title']; ?></p>
            <p style="margin-top: 5px; font-size: 11px; color: #999;"><?php echo $t['emergency_system']; ?> · <?php echo $t['copyright']; ?></p>
        </div>
    </div>

    <script>
        // PWA - Service Worker Registration
        if ('serviceWorker' in navigator) {
            window.addEventListener('load', () => {
                navigator.serviceWorker.register('service-worker.js')
                    .then(registration => {
                        console.log('Service Worker registrado:', registration);
                    })
                    .catch(error => {
                        console.log('Error al registrar Service Worker:', error);
                    });
            });
        }

        // PWA - Install Prompt
        let deferredPrompt;

        window.addEventListener('beforeinstallprompt', (e) => {
            e.preventDefault();
            deferredPrompt = e;

            // Mostrar banner de instalación personalizado si lo deseas
            console.log('App puede ser instalada');
        });

        window.addEventListener('appinstalled', () => {
            console.log('App instalada exitosamente');
            deferredPrompt = null;
        });

        // Mejorar experiencia táctil en móviles
        document.addEventListener('DOMContentLoaded', function() {
            // Prevenir zoom en doble tap en iOS
            let lastTouchEnd = 0;
            document.addEventListener('touchend', function(event) {
                const now = (new Date()).getTime();
                if (now - lastTouchEnd <= 300) {
                    event.preventDefault();
                }
                lastTouchEnd = now;
            }, false);

            // Auto-focus en email solo en desktop
            if (window.innerWidth > 768) {
                const emailInput = document.getElementById('email');
                if (emailInput) {
                    emailInput.focus();
                }
            }
        });
    </script>
</body>
</html>
