<?php
error_reporting(E_ALL);
ini_set('display_errors', 0); // No mostrar errores HTML

require_once 'config.php';

// Headers CORS y JSON
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Manejar preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Verificar que sea operador admin
if (!isset($_SESSION['operator_id'])) {
    http_response_code(403);
    echo json_encode(['success' => false, 'error' => 'No autorizado - No hay sesión activa']);
    exit();
}

// Verificar que sea admin para ciertas acciones
try {
    $stmt = $conn->prepare("SELECT role FROM operators WHERE id = ?");
    $stmt->execute([$_SESSION['operator_id']]);
    $currentOperator = $stmt->fetch(PDO::FETCH_ASSOC);
    $isAdmin = $currentOperator && $currentOperator['role'] === 'admin';
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Error al verificar permisos']);
    exit();
}

$action = $_POST['action'] ?? $_GET['action'] ?? '';

// Si no hay acción, devolver error
if (empty($action)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'No se especificó acción']);
    exit();
}

try {
    switch($action) {
        case 'list':
            listOperators();
            break;
        case 'create':
            if (!$isAdmin) throw new Exception('Solo administradores pueden crear operadores');
            createOperator();
            break;
        case 'update':
            if (!$isAdmin) throw new Exception('Solo administradores pueden editar operadores');
            updateOperator();
            break;
        case 'toggle_status':
            if (!$isAdmin) throw new Exception('Solo administradores pueden cambiar el estado');
            toggleOperatorStatus();
            break;
        case 'reset_password':
            resetOperatorPassword();
            break;
        case 'set_password':
            if (!$isAdmin) throw new Exception('Solo administradores pueden establecer contraseñas');
            setOperatorPassword();
            break;
        case 'change_password':
            changeOperatorPassword();
            break;
        case 'update_role':
            if (!$isAdmin) throw new Exception('Solo administradores pueden cambiar roles');
            updateOperatorRole();
            break;
        case 'delete':
            if (!$isAdmin) throw new Exception('Solo administradores pueden eliminar operadores');
            deleteOperator();
            break;
        default:
            throw new Exception('Acción no válida');
    }
} catch(Exception $e) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
}

function updateOperator() {
    global $conn;
    
    $operatorId = $_POST['operator_id'] ?? 0;
    $name = trim($_POST['name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $username = trim($_POST['username'] ?? '');
    $role = $_POST['role'] ?? 'operator';
    
    if (!$operatorId) {
        throw new Exception('ID de operador inválido');
    }
    
    if (empty($name) || empty($email) || empty($username)) {
        throw new Exception('Nombre, email y username son obligatorios');
    }
    
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        throw new Exception('Email inválido');
    }
    
    if (!in_array($role, ['operator', 'admin'])) {
        throw new Exception('Rol inválido');
    }
    
    // Verificar que email sea único (excepto el mismo operador)
    $stmt = $conn->prepare("SELECT id FROM operators WHERE email = ? AND id != ?");
    $stmt->execute([$email, $operatorId]);
    if ($stmt->fetch()) {
        throw new Exception('El email ya está en uso por otro operador');
    }
    
    // Verificar que username sea único (excepto el mismo operador)
    $stmt = $conn->prepare("SELECT id FROM operators WHERE username = ? AND id != ?");
    $stmt->execute([$username, $operatorId]);
    if ($stmt->fetch()) {
        throw new Exception('El username ya está en uso por otro operador');
    }
    
    // No permitir cambiar su propio rol
    if ($operatorId == $_SESSION['operator_id']) {
        $stmt = $conn->prepare("SELECT role FROM operators WHERE id = ?");
        $stmt->execute([$operatorId]);
        $currentRole = $stmt->fetch(PDO::FETCH_ASSOC)['role'];
        $role = $currentRole; // Mantener el rol actual
    }
    
    // Actualizar operador
    $stmt = $conn->prepare("
        UPDATE operators 
        SET full_name = ?, email = ?, username = ?, role = ?, updated_at = NOW()
        WHERE id = ?
    ");
    $stmt->execute([$name, $email, $username, $role, $operatorId]);
    
    error_log("Operator {$_SESSION['operator_id']} actualizó operador {$operatorId} ({$email})");
    
    echo json_encode([
        'success' => true,
        'message' => 'Operador actualizado correctamente'
    ]);
}

function changeOperatorPassword() {
    global $conn, $isAdmin;
    
    $operatorId = $_POST['operator_id'] ?? 0;
    $newPassword = trim($_POST['new_password'] ?? '');
    
    if (!$operatorId) {
        throw new Exception('ID de operador inválido');
    }
    
    // Los operadores solo pueden cambiar su propia contraseña, los admins pueden cambiar cualquiera
    if (!$isAdmin && $operatorId != $_SESSION['operator_id']) {
        throw new Exception('No tienes permiso para cambiar esta contraseña');
    }
    
    if (empty($newPassword)) {
        throw new Exception('Contraseña no puede estar vacía');
    }
    
    if (strlen($newPassword) < 6) {
        throw new Exception('La contraseña debe tener al menos 6 caracteres');
    }
    
    $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
    
    $stmt = $conn->prepare("UPDATE operators SET password = ?, updated_at = NOW() WHERE id = ?");
    $stmt->execute([$hashedPassword, $operatorId]);
    
    // Obtener info del operador
    $stmt = $conn->prepare("SELECT email, username, full_name FROM operators WHERE id = ?");
    $stmt->execute([$operatorId]);
    $operator = $stmt->fetch(PDO::FETCH_ASSOC);
    
    error_log("Operator {$_SESSION['operator_id']} cambió contraseña del operador {$operatorId} ({$operator['email']})");
    
    echo json_encode([
        'success' => true,
        'message' => 'Contraseña actualizada correctamente',
        'operator_email' => $operator['email'],
        'username' => $operator['username']
    ]);
}

function listOperators() {
    global $conn;
    
    try {
        $search = $_GET['search'] ?? '';
        
        // Query base
        $sql = "
        SELECT 
            o.id as operator_id,
            o.username,
            o.full_name as name,
            o.email,
            o.role,
            o.status,
            o.last_login,
            o.created_at,
            COUNT(DISTINCT a.id) as alerts_handled
        FROM operators o
        LEFT JOIN alerts a ON o.id = a.assigned_operator_id
        WHERE 1=1
    ";
    
    $params = [];
    
    // Filtro de búsqueda
    if (!empty($search)) {
        $sql .= " AND (o.full_name LIKE ? OR o.email LIKE ? OR o.username LIKE ?)";
        $searchParam = "%{$search}%";
        $params[] = $searchParam;
        $params[] = $searchParam;
        $params[] = $searchParam;
    }
    
    $sql .= " GROUP BY o.id ORDER BY o.created_at DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
    $operators = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Estadísticas
    $statsQuery = "
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active,
            SUM(CASE WHEN role = 'admin' THEN 1 ELSE 0 END) as admins
        FROM operators
    ";
    $stmt = $conn->query($statsQuery);
    $stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Total de alertas atendidas
    $stmt = $conn->query("SELECT COUNT(DISTINCT id) as total FROM alerts WHERE assigned_operator_id IS NOT NULL");
    $stats['total_alerts_handled'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
        echo json_encode([
            'success' => true,
            'operators' => $operators,
            'stats' => $stats
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => 'Error al listar operadores: ' . $e->getMessage()
        ]);
    }
}

function createOperator() {
    global $conn;
    
    $name = trim($_POST['name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $role = $_POST['role'] ?? 'operator';
    
    if (empty($name) || empty($email) || empty($password)) {
        throw new Exception('Todos los campos son obligatorios');
    }
    
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        throw new Exception('Email inválido');
    }
    
    if (strlen($password) < 6) {
        throw new Exception('La contraseña debe tener al menos 6 caracteres');
    }
    
    if (!in_array($role, ['operator', 'admin'])) {
        throw new Exception('Rol inválido');
    }
    
    // Verificar si el email ya existe
    $stmt = $conn->prepare("SELECT id FROM operators WHERE email = ?");
    $stmt->execute([$email]);
    if ($stmt->fetch()) {
        throw new Exception('El email ya está registrado');
    }
    
    // Generar username único basado en el email
    $username = strtolower(explode('@', $email)[0]);
    $baseUsername = $username;
    $counter = 1;
    
    // Verificar que el username sea único
    $stmt = $conn->prepare("SELECT id FROM operators WHERE username = ?");
    $stmt->execute([$username]);
    while ($stmt->fetch()) {
        $username = $baseUsername . $counter;
        $counter++;
        $stmt->execute([$username]);
    }
    
    // Crear operador
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
    $stmt = $conn->prepare("
        INSERT INTO operators (username, full_name, email, password, role, status, created_at)
        VALUES (?, ?, ?, ?, ?, 'active', NOW())
    ");
    $stmt->execute([$username, $name, $email, $hashedPassword, $role]);
    
    $newId = $conn->lastInsertId();
    
    // Log
    error_log("Operator {$_SESSION['operator_id']} created new operator {$newId} ({$email})");
    
    echo json_encode([
        'success' => true,
        'message' => 'Operador creado correctamente',
        'operator_id' => $newId,
        'username' => $username
    ]);
}

function toggleOperatorStatus() {
    global $conn;
    
    $operatorId = $_POST['operator_id'] ?? 0;
    $newStatus = $_POST['status'] ?? '';
    
    if (!$operatorId || !in_array($newStatus, ['active', 'inactive'])) {
        throw new Exception('Parámetros inválidos');
    }
    
    // No permitir desactivarse a sí mismo
    if ($operatorId == $_SESSION['operator_id']) {
        throw new Exception('No puedes cambiar tu propio estado');
    }
    
    $stmt = $conn->prepare("UPDATE operators SET status = ? WHERE id = ?");
    $stmt->execute([$newStatus, $operatorId]);
    
    error_log("Operator {$_SESSION['operator_id']} changed status of operator {$operatorId} to {$newStatus}");
    
    echo json_encode([
        'success' => true,
        'message' => 'Estado actualizado correctamente',
        'new_status' => $newStatus
    ]);
}

function resetOperatorPassword() {
    global $conn, $isAdmin;
    
    $operatorId = $_POST['operator_id'] ?? 0;
    $newPassword = trim($_POST['new_password'] ?? '');
    
    if (!$operatorId) {
        throw new Exception('ID de operador inválido');
    }
    
    // Los operadores solo pueden resetear su propia contraseña, los admins pueden resetear cualquiera
    if (!$isAdmin && $operatorId != $_SESSION['operator_id']) {
        throw new Exception('No tienes permiso para resetear esta contraseña');
    }
    
    // Si se proporciona contraseña específica, usarla. Si no, generar automática
    if (!empty($newPassword)) {
        if (strlen($newPassword) < 4) {
            throw new Exception('La contraseña debe tener al menos 4 caracteres');
        }
        $tempPassword = $newPassword;
        $isCustomPassword = true;
    } else {
        $tempPassword = 'Oper' . rand(1000, 9999) . '!';
        $isCustomPassword = false;
    }
    
    $hashedPassword = password_hash($tempPassword, PASSWORD_DEFAULT);
    
    $stmt = $conn->prepare("UPDATE operators SET password = ? WHERE id = ?");
    $stmt->execute([$hashedPassword, $operatorId]);
    
    // Obtener info del operador
    $stmt = $conn->prepare("SELECT email, full_name as name FROM operators WHERE id = ?");
    $stmt->execute([$operatorId]);
    $operator = $stmt->fetch(PDO::FETCH_ASSOC);
    
    $logType = $isCustomPassword ? 'cambió' : 'reseteo';
    error_log("Operator {$_SESSION['operator_id']} {$logType} password for operator {$operatorId} ({$operator['email']})");
    
    echo json_encode([
        'success' => true,
        'message' => $isCustomPassword ? 'Contraseña cambiada correctamente' : 'Contraseña restablecida correctamente',
        'temp_password' => $tempPassword,
        'operator_email' => $operator['email'],
        'is_custom' => $isCustomPassword
    ]);
}

function updateOperatorRole() {
    global $conn;
    
    $operatorId = $_POST['operator_id'] ?? 0;
    $newRole = $_POST['role'] ?? '';
    
    if (!$operatorId || !in_array($newRole, ['operator', 'admin'])) {
        throw new Exception('Parámetros inválidos');
    }
    
    // No permitir cambiar su propio rol
    if ($operatorId == $_SESSION['operator_id']) {
        throw new Exception('No puedes cambiar tu propio rol');
    }
    
    $stmt = $conn->prepare("UPDATE operators SET role = ? WHERE id = ?");
    $stmt->execute([$newRole, $operatorId]);
    
    error_log("Operator {$_SESSION['operator_id']} changed role of operator {$operatorId} to {$newRole}");
    
    echo json_encode([
        'success' => true,
        'message' => 'Rol actualizado correctamente',
        'new_role' => $newRole
    ]);
}

function deleteOperator() {
    global $conn;
    
    $operatorId = $_POST['operator_id'] ?? 0;
    
    if (!$operatorId) {
        throw new Exception('ID de operador inválido');
    }
    
    // No permitir eliminarse a sí mismo
    if ($operatorId == $_SESSION['operator_id']) {
        throw new Exception('No puedes eliminar tu propia cuenta');
    }
    
    // Verificar si tiene alertas asignadas activas
    $stmt = $conn->prepare("SELECT COUNT(*) as active FROM alerts WHERE operator_id = ? AND status IN ('pending', 'in_progress')");
    $stmt->execute([$operatorId]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($result['active'] > 0) {
        throw new Exception('No se puede eliminar: el operador tiene alertas activas asignadas');
    }
    
    // Eliminar operador
    $stmt = $conn->prepare("DELETE FROM operators WHERE id = ?");
    $stmt->execute([$operatorId]);
    
    error_log("Operator {$_SESSION['operator_id']} deleted operator {$operatorId}");

    echo json_encode([
        'success' => true,
        'message' => 'Operador eliminado correctamente'
    ]);
}

function setOperatorPassword() {
    global $conn;

    $operatorId = $_POST['operator_id'] ?? 0;
    $newPassword = trim($_POST['password'] ?? '');

    if (!$operatorId) {
        throw new Exception('ID de operador inválido');
    }

    if (empty($newPassword)) {
        throw new Exception('La contraseña no puede estar vacía');
    }

    if (strlen($newPassword) < 4) {
        throw new Exception('La contraseña debe tener al menos 4 caracteres');
    }

    // Verificar que el operador existe
    $stmt = $conn->prepare("SELECT id, username, full_name FROM operators WHERE id = ?");
    $stmt->execute([$operatorId]);
    $operator = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$operator) {
        throw new Exception('Operador no encontrado');
    }

    // Actualizar contraseña
    $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
    $stmt = $conn->prepare("UPDATE operators SET password = ? WHERE id = ?");
    $stmt->execute([$hashedPassword, $operatorId]);

    error_log("Admin {$_SESSION['operator_id']} set password for operator {$operatorId} ({$operator['username']})");

    echo json_encode([
        'success' => true,
        'message' => 'Contraseña establecida correctamente',
        'operator_name' => $operator['full_name'],
        'username' => $operator['username']
    ]);
}
?>
