<?php
require_once 'config.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit;
}

$message = '';
$generated_password = '';

// Solo admins pueden gestionar usuarios
if ($_SESSION['role'] != 'admin') {
    header('Location: dashboard.php');
    exit;
}

// Procesar acciones
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $action = $_POST['action'] ?? '';
    
    try {
        if ($action == 'create') {
            $user_type = $_POST['user_type'];
            $username = $_POST['username'];
            
            // Generar contraseña aleatoria si no se proporciona
            $password = !empty($_POST['password']) ? $_POST['password'] : bin2hex(random_bytes(4));
            $generated_password = $password; // Guardar para mostrar al admin
            $password_hash = password_hash($password, PASSWORD_DEFAULT);
            
            $full_name = $_POST['full_name'];
            $email = $_POST['email'];
            $phone = $_POST['phone'];
            
            if ($user_type == 'operator') {
                // Crear operador
                $role = $_POST['role'];
                $stmt = $conn->prepare("INSERT INTO operators (username, password, full_name, email, phone, role) VALUES (?, ?, ?, ?, ?, ?)");
                $stmt->execute([$username, $password_hash, $full_name, $email, $phone, $role]);
                $message = "✅ Operador creado correctamente";
            } else {
                // Crear ciudadano - sin service_code
                $cedula = $_POST['cedula'];
                
                $stmt = $conn->prepare("INSERT INTO users (username, password, full_name, cedula, email, phone) VALUES (?, ?, ?, ?, ?, ?)");
                $stmt->execute([$username, $password_hash, $full_name, $cedula, $email, $phone]);
                $message = "✅ Ciudadano creado correctamente";
            }
        } elseif ($action == 'toggle_status_operator') {
            $id = $_POST['id'];
            $new_status = $_POST['new_status'];
            $stmt = $conn->prepare("UPDATE operators SET status = ? WHERE id = ?");
            $stmt->execute([$new_status, $id]);
            $message = "✅ Estado actualizado";
        } elseif ($action == 'toggle_status_citizen') {
            $id = $_POST['id'];
            $new_status = $_POST['new_status'];
            $stmt = $conn->prepare("UPDATE users SET status = ? WHERE id = ?");
            $stmt->execute([$new_status, $id]);
            $message = "✅ Estado actualizado";
        } elseif ($action == 'delete_operator') {
            $id = $_POST['id'];
            if ($id != $_SESSION['user_id']) {
                $stmt = $conn->prepare("DELETE FROM operators WHERE id = ?");
                $stmt->execute([$id]);
                $message = "✅ Operador eliminado";
            } else {
                $message = "❌ No puedes eliminar tu propia cuenta";
            }
        } elseif ($action == 'delete_citizen') {
            $id = $_POST['id'];
            $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
            $stmt->execute([$id]);
            $message = "✅ Ciudadano eliminado";
        }
    } catch(PDOException $e) {
        $message = "❌ Error: " . $e->getMessage();
    }
}

// Obtener operadores y ciudadanos
try {
    $stmt = $conn->query("SELECT * FROM operators ORDER BY created_at DESC");
    $operators = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $stmt = $conn->query("SELECT * FROM users ORDER BY created_at DESC");
    $citizens = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch(PDOException $e) {
    $operators = [];
    $citizens = [];
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestión de Usuarios - Azuero Safety</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; background: #f5f5f5; }
        
        .header {
            background: #1e3c72;
            color: white;
            padding: 15px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        
        .header h1 { font-size: 24px; }
        .nav { display: flex; gap: 20px; align-items: center; }
        .nav a { color: white; text-decoration: none; padding: 8px 15px; border-radius: 5px; transition: background 0.3s; }
        .nav a:hover { background: rgba(255,255,255,0.1); }
        .nav a.active { background: rgba(255,255,255,0.2); }
        .logout-btn { background: #c62828; padding: 8px 20px; border-radius: 5px; }
        
        .container { max-width: 1400px; margin: 30px auto; padding: 0 20px; }
        
        .message {
            background: #e8f5e9;
            color: #2e7d32;
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            border-left: 4px solid #2e7d32;
        }
        
        .password-alert {
            background: #fff3cd;
            color: #856404;
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            border-left: 4px solid #ffc107;
        }
        
        .password-alert strong {
            font-size: 16px;
            display: block;
            margin-bottom: 5px;
        }
        
        .card {
            background: white;
            border-radius: 10px;
            padding: 25px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 20px;
        }
        
        .card h2 {
            color: #1e3c72;
            margin-bottom: 20px;
            font-size: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
            border-bottom: 2px solid #eee;
        }
        
        .tab {
            padding: 12px 24px;
            cursor: pointer;
            border: none;
            background: none;
            font-size: 14px;
            font-weight: 600;
            color: #666;
            border-bottom: 3px solid transparent;
            transition: all 0.3s;
        }
        
        .tab:hover {
            color: #1e3c72;
        }
        
        .tab.active {
            color: #1e3c72;
            border-bottom-color: #1e3c72;
        }
        
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
        }
        
        .form-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 15px;
        }
        
        .form-group {
            display: flex;
            flex-direction: column;
        }
        
        label {
            font-weight: 600;
            color: #333;
            margin-bottom: 5px;
            font-size: 14px;
        }
        
        input, select {
            padding: 10px;
            border: 2px solid #ddd;
            border-radius: 5px;
            font-size: 14px;
            transition: border-color 0.3s;
        }
        
        input:focus, select:focus {
            outline: none;
            border-color: #1e3c72;
        }
        
        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
            transition: all 0.3s;
        }
        .btn-primary { background: #1e3c72; color: white; }
        .btn-primary:hover { background: #2a5298; }
        .btn-success { background: #2e7d32; color: white; padding: 6px 12px; font-size: 12px; }
        .btn-danger { background: #c62828; color: white; padding: 6px 12px; font-size: 12px; }
        .btn-warning { background: #f57c00; color: white; padding: 6px 12px; font-size: 12px; }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        th {
            background: #f5f5f5;
            padding: 12px;
            text-align: left;
            font-weight: 600;
            color: #333;
            border-bottom: 2px solid #ddd;
            font-size: 14px;
        }
        
        td {
            padding: 12px;
            border-bottom: 1px solid #eee;
            font-size: 14px;
        }
        
        tr:hover {
            background: #f9f9f9;
        }
        
        .badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
        }
        .badge.active { background: #e8f5e9; color: #2e7d32; }
        .badge.inactive { background: #ffebee; color: #c62828; }
        .badge.admin { background: #e3f2fd; color: #1976d2; }
        .badge.supervisor { background: #fff3e0; color: #f57c00; }
        .badge.operator { background: #f5f5f5; color: #666; }
        
        .actions {
            display: flex;
            gap: 5px;
        }
        
        .info-box {
            background: #e3f2fd;
            border-left: 4px solid #1976d2;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            font-size: 14px;
            color: #555;
        }
        
        #operator-fields, #citizen-fields {
            display: none;
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>🚨 Azuero Safety</h1>
        <div class="nav">
            <a href="dashboard.php">📊 Dashboard</a>
            <a href="alertas.php">🚨 Alertas</a>
            <a href="operadores.php" class="active">👥 Usuarios</a>
            <span style="opacity: 0.7;">👤 <?php echo htmlspecialchars($_SESSION['full_name']); ?></span>
            <a href="logout.php" class="logout-btn">Salir</a>
        </div>
    </div>
    
    <div class="container">
        <?php if ($message): ?>
        <div class="message"><?php echo $message; ?></div>
        <?php endif; ?>
        
        <?php if ($generated_password): ?>
        <div class="password-alert">
            <strong>⚠️ IMPORTANTE: Contraseña generada</strong>
            <p>Contraseña temporal: <strong style="font-size: 18px; font-family: monospace;"><?php echo htmlspecialchars($generated_password); ?></strong></p>
            <p style="margin-top: 5px; font-size: 13px;">Proporciona esta contraseña al usuario. Deberá cambiarla en su primer acceso.</p>
        </div>
        <?php endif; ?>
        
        <div class="card">
            <h2>➕ Crear Nuevo Usuario</h2>
            
            <div class="info-box">
                ℹ️ <strong>Nuevo sistema unificado:</strong> Ahora todos los usuarios (operadores y ciudadanos) inician sesión con usuario y contraseña desde la misma página. Los ciudadanos completarán su perfil médico en el primer acceso.
            </div>
            
            <form method="POST" id="createForm">
                <input type="hidden" name="action" value="create">
                
                <div class="form-group" style="margin-bottom: 20px;">
                    <label>Tipo de Usuario *</label>
                    <select name="user_type" id="userType" required onchange="toggleUserFields()">
                        <option value="">-- Selecciona tipo --</option>
                        <option value="operator">👮 Operador del Sistema</option>
                        <option value="citizen">👤 Ciudadano</option>
                    </select>
                </div>
                
                <div class="form-grid">
                    <div class="form-group">
                        <label>Usuario * <small>(login)</small></label>
                        <input type="text" name="username" required>
                    </div>
                    <div class="form-group">
                        <label>Contraseña <small>(dejar vacío para auto-generar)</small></label>
                        <input type="text" name="password" placeholder="Se generará automáticamente">
                    </div>
                    <div class="form-group">
                        <label>Nombre Completo *</label>
                        <input type="text" name="full_name" required>
                    </div>
                    <div class="form-group">
                        <label>Email *</label>
                        <input type="email" name="email" required>
                    </div>
                    <div class="form-group">
                        <label>Teléfono</label>
                        <input type="text" name="phone">
                    </div>
                </div>
                
                <!-- Campos específicos de operador -->
                <div id="operator-fields">
                    <div class="form-grid" style="margin-top: 15px;">
                        <div class="form-group">
                            <label>Rol en el Sistema *</label>
                            <select name="role">
                                <option value="operator">Operador</option>
                                <option value="supervisor">Supervisor</option>
                                <option value="admin">Administrador</option>
                            </select>
                        </div>
                    </div>
                </div>
                
                <!-- Campos específicos de ciudadano -->
                <div id="citizen-fields">
                    <div class="form-grid" style="margin-top: 15px;">
                        <div class="form-group">
                            <label>Cédula *</label>
                            <input type="text" name="cedula" placeholder="8-888-8888">
                        </div>
                    </div>
                    <div class="info-box" style="margin-top: 15px;">
                        ℹ️ El ciudadano deberá completar su información médica (tipo de sangre, alergias, condiciones, contacto de emergencia) en su primer inicio de sesión.
                    </div>
                </div>
                
                <button type="submit" class="btn btn-primary" style="margin-top: 15px;">✅ Crear Usuario</button>
            </form>
        </div>
        
        <div class="tabs">
            <button class="tab active" onclick="switchTab('operators')">👮 Operadores (<?php echo count($operators); ?>)</button>
            <button class="tab" onclick="switchTab('citizens')">👤 Ciudadanos (<?php echo count($citizens); ?>)</button>
        </div>
        
        <!-- Tabla de Operadores -->
        <div id="operators-tab" class="tab-content active">
            <div class="card">
                <h2>👮 Lista de Operadores</h2>
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Usuario</th>
                            <th>Nombre</th>
                            <th>Email</th>
                            <th>Teléfono</th>
                            <th>Rol</th>
                            <th>Estado</th>
                            <th>Fecha Registro</th>
                            <th>Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($operators as $op): ?>
                        <tr>
                            <td><strong>#<?php echo $op['id']; ?></strong></td>
                            <td><?php echo htmlspecialchars($op['username']); ?></td>
                            <td><?php echo htmlspecialchars($op['full_name']); ?></td>
                            <td><?php echo htmlspecialchars($op['email']); ?></td>
                            <td><?php echo htmlspecialchars($op['phone'] ?? '-'); ?></td>
                            <td><span class="badge <?php echo $op['role']; ?>"><?php echo strtoupper($op['role']); ?></span></td>
                            <td><span class="badge <?php echo $op['status']; ?>"><?php echo strtoupper($op['status']); ?></span></td>
                            <td><?php echo date('d/m/Y', strtotime($op['created_at'])); ?></td>
                            <td>
                                <div class="actions">
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="action" value="toggle_status_operator">
                                        <input type="hidden" name="id" value="<?php echo $op['id']; ?>">
                                        <input type="hidden" name="new_status" value="<?php echo $op['status'] == 'active' ? 'inactive' : 'active'; ?>">
                                        <button type="submit" class="btn <?php echo $op['status'] == 'active' ? 'btn-warning' : 'btn-success'; ?>">
                                            <?php echo $op['status'] == 'active' ? '⏸️' : '▶️'; ?>
                                        </button>
                                    </form>
                                    
                                    <?php if ($op['id'] != $_SESSION['user_id']): ?>
                                    <form method="POST" style="display: inline;" onsubmit="return confirm('¿Seguro que deseas eliminar este operador?');">
                                        <input type="hidden" name="action" value="delete_operator">
                                        <input type="hidden" name="id" value="<?php echo $op['id']; ?>">
                                        <button type="submit" class="btn btn-danger">🗑️</button>
                                    </form>
                                    <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <!-- Tabla de Ciudadanos -->
        <div id="citizens-tab" class="tab-content">
            <div class="card">
                <h2>👤 Lista de Ciudadanos</h2>
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Usuario</th>
                            <th>Nombre</th>
                            <th>Cédula</th>
                            <th>Email</th>
                            <th>Teléfono</th>
                            <th>Perfil</th>
                            <th>Estado</th>
                            <th>Fecha Registro</th>
                            <th>Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($citizens as $citizen): ?>
                        <tr>
                            <td><strong>#<?php echo $citizen['id']; ?></strong></td>
                            <td><?php echo htmlspecialchars($citizen['username'] ?? '-'); ?></td>
                            <td><?php echo htmlspecialchars($citizen['full_name']); ?></td>
                            <td><?php echo htmlspecialchars($citizen['cedula']); ?></td>
                            <td><?php echo htmlspecialchars($citizen['email']); ?></td>
                            <td><?php echo htmlspecialchars($citizen['phone'] ?? '-'); ?></td>
                            <td>
                                <?php 
                                $profile_complete = !empty($citizen['blood_type']) && !empty($citizen['emergency_contact']);
                                echo $profile_complete ? '✅ Completo' : '⏳ Pendiente';
                                ?>
                            </td>
                            <td><span class="badge <?php echo $citizen['status']; ?>"><?php echo strtoupper($citizen['status']); ?></span></td>
                            <td><?php echo date('d/m/Y', strtotime($citizen['created_at'])); ?></td>
                            <td>
                                <div class="actions">
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="action" value="toggle_status_citizen">
                                        <input type="hidden" name="id" value="<?php echo $citizen['id']; ?>">
                                        <input type="hidden" name="new_status" value="<?php echo $citizen['status'] == 'active' ? 'inactive' : 'active'; ?>">
                                        <button type="submit" class="btn <?php echo $citizen['status'] == 'active' ? 'btn-warning' : 'btn-success'; ?>">
                                            <?php echo $citizen['status'] == 'active' ? '⏸️' : '▶️'; ?>
                                        </button>
                                    </form>
                                    
                                    <form method="POST" style="display: inline;" onsubmit="return confirm('¿Seguro que deseas eliminar este ciudadano?');">
                                        <input type="hidden" name="action" value="delete_citizen">
                                        <input type="hidden" name="id" value="<?php echo $citizen['id']; ?>">
                                        <button type="submit" class="btn btn-danger">🗑️</button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    
    <script>
        function switchTab(tab) {
            // Cambiar tabs activas
            document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
            document.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));
            
            event.target.classList.add('active');
            document.getElementById(tab + '-tab').classList.add('active');
        }
        
        function toggleUserFields() {
            const userType = document.getElementById('userType').value;
            const operatorFields = document.getElementById('operator-fields');
            const citizenFields = document.getElementById('citizen-fields');
            
            if (userType === 'operator') {
                operatorFields.style.display = 'block';
                citizenFields.style.display = 'none';
                // Hacer campos de operador requeridos
                document.querySelector('[name="role"]').required = true;
                document.querySelector('[name="cedula"]').required = false;
            } else if (userType === 'citizen') {
                operatorFields.style.display = 'none';
                citizenFields.style.display = 'block';
                // Hacer campos de ciudadano requeridos
                document.querySelector('[name="role"]').required = false;
                document.querySelector('[name="cedula"]').required = true;
            } else {
                operatorFields.style.display = 'none';
                citizenFields.style.display = 'none';
            }
        }
    </script>
</body>
</html>
