<?php
require_once 'config.php';

// Headers para evitar caché
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Cache-Control: post-check=0, pre-check=0', false);
header('Pragma: no-cache');

// Verificar que sea un ciudadano logueado
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] != 'citizen') {
    header('Location: index.php');
    exit;
}

// Verificar que tenga el perfil completo
$stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
$stmt->execute([$_SESSION['user_id']]);
$user = $stmt->fetch();

if (empty($user['blood_type']) || empty($user['emergency_contact'])) {
    header('Location: completar-perfil.php');
    exit;
}

// Obtener idioma del usuario de la base de datos
$lang = $user['language'] ?? 'es';

// Si viene por parámetro GET, actualizar en BD y sesión
if (isset($_GET['lang']) && in_array($_GET['lang'], ['es', 'en'])) {
    $lang = $_GET['lang'];
    $_SESSION['lang'] = $lang;
    
    // Guardar en la base de datos para persistencia
    try {
        $stmt = $conn->prepare("UPDATE users SET language = ? WHERE id = ?");
        $stmt->execute([$lang, $_SESSION['user_id']]);
    } catch(PDOException $e) {
        // Si falla, continuamos con el idioma en sesión
    }
}

$translations = [
    'es' => [
        'title' => 'Panel de Emergencias',
        'welcome' => 'Bienvenido',
        'profile' => 'Mi Perfil',
        'logout' => 'Cerrar Sesión',
        'medical_btn' => 'EMERGENCIA MÉDICA',
        'medical_desc' => 'Situación de vida o muerte - Alerta inmediata',
        'general_btn' => 'EMERGENCIA GENERAL',
        'general_desc' => 'Incendio, accidente, rescate - Describe la situación',
        'police_btn' => 'ASISTENCIA POLICIAL',
        'police_desc' => 'Seguridad, crimen, ayuda policial',
        'my_alerts' => 'Mis Alertas Activas',
        'alert_type' => 'Tipo',
        'status' => 'Estado',
        'created' => 'Creada',
        'assigned' => 'Operador Asignado',
        'no_alerts' => 'No tienes alertas activas',
        'pending' => 'Pendiente',
        'in_progress' => 'En Progreso',
        'resolved' => 'Resuelta',
        'medical' => 'Médica',
        'fire' => 'Incendio',
        'accident' => 'Accidente',
        'police' => 'Policial',
        'other' => 'Otra',
        'cancel_alert' => 'Cancelar',
        'view_details' => 'Ver detalles',
        'gps_enable' => 'Activar GPS',
        'gps_wait' => 'Obteniendo ubicación...',
        'gps_success' => 'Ubicación obtenida',
        'gps_error' => 'No se pudo obtener ubicación',
        'sending' => 'Enviando alerta...',
        'code' => 'Código de Servicio'
    ],
    'en' => [
        'title' => 'Emergency Panel',
        'welcome' => 'Welcome',
        'profile' => 'My Profile',
        'logout' => 'Logout',
        'medical_btn' => 'MEDICAL EMERGENCY',
        'medical_desc' => 'Life or death situation - Immediate alert',
        'general_btn' => 'GENERAL EMERGENCY',
        'general_desc' => 'Fire, accident, rescue - Describe the situation',
        'police_btn' => 'POLICE ASSISTANCE',
        'police_desc' => 'Security, crime, police help',
        'my_alerts' => 'My Active Alerts',
        'alert_type' => 'Type',
        'status' => 'Status',
        'created' => 'Created',
        'assigned' => 'Assigned Operator',
        'no_alerts' => 'No active alerts',
        'pending' => 'Pending',
        'in_progress' => 'In Progress',
        'resolved' => 'Resolved',
        'medical' => 'Medical',
        'fire' => 'Fire',
        'accident' => 'Accident',
        'police' => 'Police',
        'other' => 'Other',
        'cancel_alert' => 'Cancel',
        'view_details' => 'View details',
        'gps_enable' => 'Enable GPS',
        'gps_wait' => 'Getting location...',
        'gps_success' => 'Location obtained',
        'gps_error' => 'Could not get location',
        'sending' => 'Sending alert...',
        'code' => 'Service Code'
    ]
];

$t = $translations[$lang];

// Ya tenemos $user del inicio, no necesitamos volver a buscarlo
// $stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
// $stmt->execute([$_SESSION['user_id']]);
// $user = $stmt->fetch(PDO::FETCH_ASSOC);

$stmt = $conn->prepare("
    SELECT a.*, o.username as operator_name 
    FROM alerts a 
    LEFT JOIN operators o ON a.assigned_operator_id = o.id 
    WHERE a.user_id = ? AND a.status IN ('pending', 'in_progress') 
    ORDER BY a.created_at DESC
");
$stmt->execute([$_SESSION['user_id']]);
$active_alerts = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title><?php echo $t['title']; ?> - Azuero Safety Services S.A</title>

    <!-- PWA Meta Tags -->
    <meta name="theme-color" content="#dc3545">
    <meta name="mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="black-translucent">
    <meta name="apple-mobile-web-app-title" content="AzueroSafe">
    <meta name="description" content="Sistema de Emergencias - Azuero Safety Services S.A - Somos tu seguridad">

    <!-- PWA Manifest -->
    <link rel="manifest" href="manifest.json">

    <!-- Icons -->
    <link rel="icon" type="image/png" sizes="192x192" href="icon-192x192.png">
    <link rel="apple-touch-icon" href="icon-192x192.png">
    
    <!-- PWA Manifest -->
    <link rel="manifest" href="manifest.json">
    
    <!-- Icons -->
    <link rel="icon" type="image/png" sizes="192x192" href="icons/icon-192x192.png">
    <link rel="apple-touch-icon" sizes="192x192" href="icons/icon-192x192.png">
    <link rel="icon" type="image/png" sizes="512x512" href="icons/icon-512x512.png">
    
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        :root {
            --primary: #667eea;
            --primary-dark: #5568d3;
            --secondary: #764ba2;
            --success: #10b981;
            --warning: #f59e0b;
            --danger: #ef4444;
            --info: #3b82f6;
            --medical: #dc2626;
            --fire: #ea580c;
            --police: #2563eb;
            --bg-primary: #ffffff;
            --bg-secondary: #f8fafc;
            --bg-card: #ffffff;
            --text-primary: #1e293b;
            --text-secondary: #64748b;
            --border-color: #e2e8f0;
            --shadow-sm: 0 1px 3px rgba(0,0,0,0.1);
            --shadow-md: 0 4px 6px rgba(0,0,0,0.1);
            --shadow-lg: 0 10px 25px rgba(0,0,0,0.15);
            --shadow-xl: 0 20px 40px rgba(0,0,0,0.2);
        }

        [data-theme="dark"] {
            --primary: #818cf8;
            --primary-dark: #6366f1;
            --bg-primary: #0f172a;
            --bg-secondary: #1e293b;
            --bg-card: #1e293b;
            --text-primary: #f1f5f9;
            --text-secondary: #94a3b8;
            --border-color: #334155;
        }

        [data-theme="blue-dark"] {
            --primary: #60a5fa;
            --primary-dark: #3b82f6;
            --secondary: #06b6d4;
            --bg-primary: #0c1e35;
            --bg-secondary: #1a2942;
            --bg-card: #1a2942;
            --text-primary: #e0f2fe;
            --text-secondary: #7dd3fc;
            --border-color: #1e3a5f;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: var(--bg-secondary);
            min-height: 100vh;
            color: var(--text-primary);
            transition: all 0.3s ease;
        }
        
        /* ========================================
           HEADER MODERNO
           ======================================== */
        .header {
            background: var(--bg-card);
            border-bottom: 1px solid var(--border-color);
            padding: 20px 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: var(--shadow-md);
            position: sticky;
            top: 0;
            z-index: 999;
        }

        .header-left {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 12px;
            font-size: 24px;
            font-weight: 700;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .logo i {
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            font-size: 28px;
        }
        
        .user-info { 
            display: flex; 
            align-items: center; 
            gap: 15px; 
        }
        
        .user-avatar { 
            width: 50px; 
            height: 50px; 
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white; 
            border-radius: 50%; 
            display: flex; 
            align-items: center; 
            justify-content: center; 
            font-size: 20px; 
            font-weight: bold;
            box-shadow: var(--shadow-md);
        }
        
        .user-details h2 { 
            font-size: 16px; 
            color: var(--text-primary); 
            margin-bottom: 4px;
            font-weight: 600;
        }
        
        .user-details p { 
            font-size: 13px; 
            color: var(--text-secondary);
            display: flex;
            align-items: center;
            gap: 6px;
        }
        
        .header-actions { 
            display: flex; 
            gap: 12px;
            align-items: center;
        }

        .theme-switcher {
            display: flex;
            gap: 8px;
            background: var(--bg-secondary);
            padding: 6px;
            border-radius: 25px;
            border: 1px solid var(--border-color);
        }

        .theme-btn {
            padding: 8px 16px;
            border: none;
            background: transparent;
            color: var(--text-secondary);
            cursor: pointer;
            border-radius: 20px;
            font-size: 14px;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .theme-btn:hover {
            background: var(--bg-card);
        }

        .theme-btn.active {
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            box-shadow: var(--shadow-sm);
        }
        
        .btn { 
            padding: 10px 20px; 
            border-radius: 10px; 
            text-decoration: none; 
            font-weight: 600; 
            transition: all 0.3s ease;
            cursor: pointer; 
            border: none; 
            font-size: 14px;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        
        .btn-profile { 
            background: var(--bg-secondary);
            color: var(--text-primary);
            border: 1px solid var(--border-color);
        }
        
        .btn-profile:hover { 
            background: var(--bg-card);
            transform: translateY(-2px);
            box-shadow: var(--shadow-md);
        }
        
        .btn-logout { 
            background: linear-gradient(135deg, var(--danger), #dc2626);
            color: white;
        }
        
        .btn-logout:hover { 
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }

        /* ========================================
           CONTENEDOR PRINCIPAL
           ======================================== */
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 30px 20px;
        }

        .page-title {
            font-size: 28px;
            font-weight: 700;
            color: var(--text-primary);
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .page-subtitle {
            font-size: 15px;
            color: var(--text-secondary);
            margin-bottom: 30px;
        }
        
        
        /* ========================================
           BOTONES DE EMERGENCIA MEJORADOS
           ======================================== */
        .emergency-buttons {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(320px, 1fr));
            gap: 25px;
            margin-bottom: 40px;
        }
        
        .emergency-card {
            background: var(--bg-card);
            border-radius: 20px;
            padding: 35px;
            box-shadow: var(--shadow-md);
            text-align: center;
            cursor: pointer;
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
            overflow: hidden;
            border: 2px solid transparent;
        }

        .emergency-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 6px;
            background: var(--card-gradient);
            transition: height 0.3s ease;
        }

        .emergency-card:hover::before {
            height: 100%;
            opacity: 0.08;
        }
        
        .emergency-card:hover { 
            transform: translateY(-8px) scale(1.02); 
            box-shadow: var(--shadow-xl);
            border-color: var(--card-color);
        }

        .emergency-card:active {
            transform: translateY(-4px) scale(1.01);
        }
        
        .medical-card { 
            --card-color: var(--medical);
            --card-gradient: linear-gradient(135deg, #dc2626, #991b1b);
        }
        
        .general-card { 
            --card-color: var(--fire);
            --card-gradient: linear-gradient(135deg, #ea580c, #c2410c);
        }
        
        .police-card { 
            --card-color: var(--police);
            --card-gradient: linear-gradient(135deg, #2563eb, #1d4ed8);
        }
        
        .card-icon-wrapper {
            width: 90px;
            height: 90px;
            margin: 0 auto 20px;
            background: var(--card-gradient);
            border-radius: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: var(--shadow-lg);
            transition: all 0.3s ease;
        }

        .emergency-card:hover .card-icon-wrapper {
            transform: scale(1.1) rotate(5deg);
        }
        
        .card-icon { 
            font-size: 48px;
            filter: drop-shadow(0 2px 4px rgba(0,0,0,0.2));
        }
        
        .card-title { 
            font-size: 20px; 
            font-weight: 700; 
            margin-bottom: 12px; 
            color: var(--card-color);
            letter-spacing: 0.5px;
        }
        
        .card-desc { 
            color: var(--text-secondary); 
            font-size: 14px; 
            line-height: 1.6;
        }

        .card-badge {
            display: inline-block;
            margin-top: 15px;
            padding: 6px 14px;
            background: var(--bg-secondary);
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            color: var(--card-color);
            border: 1px solid var(--card-color);
        }
        
        
        /* ========================================
           SECCIÓN DE ALERTAS ACTIVAS MEJORADA
           ======================================== */
        .alerts-section {
            background: var(--bg-card);
            border-radius: 20px;
            padding: 30px;
            box-shadow: var(--shadow-md);
            border: 1px solid var(--border-color);
        }
        
        .section-title { 
            font-size: 22px; 
            font-weight: 700; 
            margin-bottom: 25px; 
            color: var(--text-primary);
            display: flex;
            align-items: center;
            gap: 12px;
            padding-bottom: 15px;
            border-bottom: 2px solid var(--border-color);
        }

        .section-title i {
            color: var(--primary);
        }
        
        .alert-item {
            background: var(--bg-secondary);
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 15px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            border-left: 5px solid;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .alert-item::before {
            content: '';
            position: absolute;
            left: 0;
            top: 0;
            bottom: 0;
            width: 5px;
            background: var(--alert-color);
            transition: width 0.3s ease;
        }

        .alert-item:hover {
            transform: translateX(5px);
            box-shadow: var(--shadow-md);
        }

        .alert-item:hover::before {
            width: 8px;
        }
        
        .alert-item.medical { 
            border-color: var(--medical);
            --alert-color: var(--medical);
        }
        
        .alert-item.fire { 
            border-color: var(--fire);
            --alert-color: var(--fire);
        }
        
        .alert-item.police { 
            border-color: var(--police);
            --alert-color: var(--police);
        }
        
        .alert-item.other { 
            border-color: #9e9e9e;
            --alert-color: #9e9e9e;
        }
        
        .alert-info { 
            flex: 1; 
        }
        
        .alert-type { 
            font-weight: 700; 
            color: var(--text-primary); 
            margin-bottom: 8px;
            font-size: 16px;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .alert-type-icon {
            width: 32px;
            height: 32px;
            border-radius: 8px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            background: var(--alert-color);
            color: white;
            font-size: 16px;
        }
        
        .alert-status { 
            font-size: 13px; 
            color: var(--text-secondary);
            display: flex;
            flex-direction: column;
            gap: 4px;
        }

        .alert-status-row {
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .alert-status i {
            color: var(--primary);
        }
        
        .status-badge {
            padding: 8px 16px;
            border-radius: 25px;
            font-size: 13px;
            font-weight: 700;
            margin-left: 15px;
            white-space: nowrap;
            display: inline-flex;
            align-items: center;
            gap: 6px;
            box-shadow: var(--shadow-sm);
        }

        .status-badge i {
            font-size: 10px;
        }
        
        .status-pending { 
            background: linear-gradient(135deg, #fef3c7, #fde68a);
            color: #92400e;
        }
        
        .status-progress { 
            background: linear-gradient(135deg, #dbeafe, #bfdbfe);
            color: #1e3a8a;
        }

        .status-resolved {
            background: linear-gradient(135deg, #d1fae5, #a7f3d0);
            color: #065f46;
        }
        
        .no-alerts { 
            text-align: center; 
            padding: 60px 20px; 
            color: var(--text-secondary); 
            font-size: 16px;
        }

        .no-alerts i {
            font-size: 64px;
            margin-bottom: 20px;
            opacity: 0.3;
        }

        .operator-response {
            margin-top: 12px;
            padding: 12px 15px;
            background: var(--bg-card);
            border-left: 3px solid var(--info);
            border-radius: 8px;
            font-size: 13px;
            color: var(--text-primary);
            font-style: italic;
            box-shadow: var(--shadow-sm);
        }

        .operator-response::before {
            content: '💬 ';
            margin-right: 6px;
        }
        
        
        /* ========================================
           MODALES MODERNOS
           ======================================== */
        .modal { 
            display: none; 
            position: fixed; 
            z-index: 2000; 
            left: 0; 
            top: 0; 
            width: 100%; 
            height: 100%; 
            background: rgba(0,0,0,0.75);
            backdrop-filter: blur(8px);
            animation: fadeIn 0.3s ease;
        }
        
        .modal.active { 
            display: flex; 
            align-items: center; 
            justify-content: center; 
        }
        
        .modal-content { 
            background: var(--bg-card);
            border-radius: 20px;
            padding: 35px;
            max-width: 550px;
            width: 90%;
            max-height: 85vh;
            overflow-y: auto;
            box-shadow: var(--shadow-xl);
            animation: slideUp 0.4s cubic-bezier(0.4, 0, 0.2, 1);
            border: 1px solid var(--border-color);
        }

        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }

        @keyframes slideUp {
            from { 
                opacity: 0;
                transform: translateY(30px) scale(0.95);
            }
            to { 
                opacity: 1;
                transform: translateY(0) scale(1);
            }
        }
        
        .modal-header { 
            display: flex; 
            justify-content: space-between; 
            align-items: center; 
            margin-bottom: 25px;
            padding-bottom: 20px;
            border-bottom: 2px solid var(--border-color);
        }
        
        .modal-title { 
            font-size: 24px; 
            font-weight: 700; 
            color: var(--text-primary);
            display: flex;
            align-items: center;
            gap: 12px;
        }
        
        .close-modal { 
            background: var(--bg-secondary);
            border: none; 
            font-size: 24px; 
            cursor: pointer; 
            color: var(--text-secondary);
            width: 36px;
            height: 36px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
        }

        .close-modal:hover {
            background: var(--danger);
            color: white;
            transform: rotate(90deg);
        }
        
        .form-group { 
            margin-bottom: 20px; 
        }
        
        .form-group label { 
            display: block; 
            font-weight: 600; 
            margin-bottom: 10px; 
            color: var(--text-primary);
            font-size: 14px;
        }
        
        .form-group input, 
        .form-group select, 
        .form-group textarea { 
            width: 100%; 
            padding: 14px 16px; 
            border: 2px solid var(--border-color);
            border-radius: 10px; 
            background: var(--bg-secondary);
            color: var(--text-primary);
            font-family: inherit; 
            font-size: 14px;
            transition: all 0.3s ease;
        }

        .form-group input:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: var(--primary);
            background: var(--bg-card);
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .form-group textarea { 
            min-height: 120px; 
            resize: vertical;
        }
        
        .btn-submit { 
            width: 100%; 
            padding: 16px; 
            background: linear-gradient(135deg, var(--danger), #dc2626);
            color: white; 
            border: none; 
            border-radius: 12px; 
            font-size: 16px; 
            font-weight: 700; 
            cursor: pointer;
            transition: all 0.3s ease;
            box-shadow: var(--shadow-md);
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }
        
        .btn-submit:hover { 
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }

        .btn-submit:active {
            transform: translateY(0);
        }
        
        .btn-cancel { 
            width: 100%; 
            padding: 16px; 
            background: var(--bg-secondary);
            color: var(--text-primary);
            border: 2px solid var(--border-color);
            border-radius: 12px; 
            font-size: 16px; 
            font-weight: 600; 
            cursor: pointer; 
            margin-top: 12px;
            transition: all 0.3s ease;
        }
        
        .btn-cancel:hover { 
            background: var(--bg-card);
            border-color: var(--text-secondary);
        }
        
        .gps-status { 
            padding: 14px 18px; 
            border-radius: 12px; 
            margin-bottom: 20px; 
            text-align: center; 
            font-weight: 600;
            font-size: 14px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }
        
        .gps-success { 
            background: rgba(16, 185, 129, 0.15);
            color: var(--success);
            border: 2px solid var(--success);
        }
        
        .gps-error { 
            background: rgba(239, 68, 68, 0.15);
            color: var(--danger);
            border: 2px solid var(--danger);
        }
        
        .gps-loading { 
            background: rgba(59, 130, 246, 0.15);
            color: var(--info);
            border: 2px solid var(--info);
        }
        
        .yes-no-buttons { 
            display: flex; 
            gap: 15px; 
        }
        
        .yes-no-buttons button { 
            flex: 1; 
            padding: 24px; 
            font-size: 18px; 
            font-weight: 700; 
            border: none; 
            border-radius: 15px; 
            cursor: pointer; 
            transition: all 0.3s ease;
            box-shadow: var(--shadow-md);
        }
        
        .btn-yes { 
            background: linear-gradient(135deg, var(--danger), #dc2626);
            color: white; 
        }
        
        .btn-yes:hover { 
            transform: translateY(-4px);
            box-shadow: var(--shadow-lg);
        }
        
        .btn-no { 
            background: linear-gradient(135deg, var(--info), #2563eb);
            color: white; 
        }
        
        .btn-no:hover { 
            transform: translateY(-4px);
            box-shadow: var(--shadow-lg);
        }
        
        .loading { 
            display: none; 
            text-align: center; 
            padding: 30px; 
        }
        
        .loading.active { 
            display: block; 
        }
        
        .spinner { 
            border: 4px solid var(--border-color);
            border-top: 4px solid var(--primary);
            border-radius: 50%; 
            width: 48px; 
            height: 48px; 
            animation: spin 1s linear infinite; 
            margin: 0 auto 20px; 
        }
        
        @keyframes spin { 
            0% { transform: rotate(0deg); } 
            100% { transform: rotate(360deg); } 
        }
        
        /* ========================================
           RESPONSIVE DESIGN PARA MÓVILES
           ======================================== */
        @media (max-width: 768px) {
            body {
                padding: 10px;
            }
            
            .header {
                padding: 15px;
                flex-direction: column;
                gap: 15px;
            }
            
            .logo {
                font-size: 16px;
            }
            
            .logo i {
                font-size: 24px;
            }
            
            .logo div span:first-child {
                font-size: 16px;
            }
            
            .logo div span:last-child {
                font-size: 10px;
            }
            
            .theme-switcher {
                flex-wrap: wrap;
                justify-content: center;
            }
            
            .theme-btn {
                font-size: 11px;
                padding: 8px 12px;
            }
            
            .user-info {
                width: 100%;
                justify-content: center;
                text-align: center;
                flex-direction: column;
            }
            
            .user-details h2 {
                font-size: 16px;
            }
            
            .user-details p {
                font-size: 11px;
            }
            
            .header-actions {
                width: 100%;
                justify-content: center;
                flex-wrap: wrap;
            }
            
            .btn {
                padding: 12px 16px;
                font-size: 13px;
            }
            
            /* Botones de emergencia en columna en móvil */
            .emergency-buttons {
                grid-template-columns: 1fr;
                gap: 20px;
                padding: 0 5px;
            }

            .emergency-card {
                padding: 30px 20px;
                min-height: 140px;
                cursor: pointer;
                -webkit-tap-highlight-color: transparent;
                touch-action: manipulation;
            }

            .emergency-card:active {
                transform: scale(0.98);
            }

            .card-icon {
                font-size: 55px;
                margin-bottom: 12px;
            }

            .card-title {
                font-size: 19px;
                margin-bottom: 8px;
            }

            .card-desc {
                font-size: 14px;
                line-height: 1.4;
            }
            
            /* Sección de alertas */
            .alerts-section {
                padding: 20px 15px;
            }
            
            .section-title {
                font-size: 18px;
            }
            
            .alert-item {
                flex-direction: column;
                align-items: flex-start;
                gap: 10px;
            }
            
            .status-badge {
                margin-left: 0;
            }
            
            /* Modales */
            .modal-content {
                width: 95%;
                padding: 20px;
                max-height: 90vh;
            }
            
            .modal-title {
                font-size: 20px;
            }
            
            .form-group label {
                font-size: 13px;
            }
            
            .form-group input,
            .form-group select,
            .form-group textarea {
                padding: 10px;
                font-size: 14px;
            }
            
            .yes-no-buttons button {
                padding: 15px;
                font-size: 16px;
            }
            
            .btn-submit,
            .btn-cancel {
                padding: 12px;
                font-size: 15px;
            }
        }
        
        /* Mejoras para pantallas muy pequeñas */
        @media (max-width: 480px) {
            body {
                padding: 5px;
            }

            .container {
                padding: 15px 10px;
            }

            .logo {
                font-size: 14px !important;
            }

            .logo i {
                font-size: 20px !important;
            }

            .logo div span:first-child {
                font-size: 14px !important;
            }

            .logo div span:last-child {
                font-size: 9px !important;
            }

            .theme-switcher {
                display: none; /* Ocultar en pantallas muy pequeñas */
            }

            .lang-switcher {
                padding: 6px 10px;
                font-size: 11px;
            }

            .user-avatar {
                width: 40px;
                height: 40px;
                font-size: 16px;
            }

            .emergency-buttons {
                gap: 15px;
            }

            .emergency-card {
                padding: 25px 15px;
                min-height: 130px;
            }

            .card-icon {
                font-size: 48px;
            }

            .card-title {
                font-size: 17px;
            }

            .card-desc {
                font-size: 13px;
            }

            .btn {
                padding: 10px 14px;
                font-size: 12px;
            }

            .modal-content {
                width: 98%;
                padding: 18px;
            }

            .modal-title {
                font-size: 18px;
            }
        }

        /* Landscape mode en móviles */
        @media (max-width: 768px) and (orientation: landscape) {
            .header {
                padding: 10px 15px;
            }

            .emergency-buttons {
                grid-template-columns: repeat(3, 1fr);
                gap: 10px;
            }

            .emergency-card {
                padding: 20px 15px;
                min-height: 110px;
            }

            .card-icon {
                font-size: 40px;
                margin-bottom: 8px;
            }

            .card-title {
                font-size: 16px;
                margin-bottom: 5px;
            }

            .card-desc {
                font-size: 12px;
            }

            .modal-content {
                max-height: 95vh;
            }
        }
            
            .btn {
                padding: 10px 14px;
                font-size: 12px;
            }
            
            .header {
                padding: 12px 15px;
            }
        }
        
        /* ========================================
           ESTILOS PARA RESPUESTAS DEL OPERADOR
           ======================================== */
        .operator-response {
            margin-top: 10px;
            padding: 10px;
            background: #e3f2fd;
            border-left: 3px solid #2196F3;
            border-radius: 5px;
            font-size: 13px;
            color: #1565c0;
            font-style: italic;
        }
        
        .status-resolved {
            background: #e8f5e9;
            color: #2e7d32;
        }
        
        /* ========================================
           ANIMACIONES PARA NOTIFICACIONES
           ======================================== */
        @keyframes slideIn {
            from {
                transform: translateX(400px);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }
        
        @keyframes slideOut {
            from {
                transform: translateX(0);
                opacity: 1;
            }
            to {
                transform: translateX(400px);
                opacity: 0;
            }
        }
        
        @keyframes pulse {
            0%, 100% {
                transform: scale(1);
            }
            50% {
                transform: scale(1.05);
            }
        }
        
        .notification-badge {
            animation: pulse 2s infinite;
        }
    </style>
</head>
<body data-theme="light">
    <!-- HEADER MODERNO -->
    <div class="header">
        <div class="header-left">
            <div class="logo">
                <i class="fas fa-shield-heart"></i>
                <div style="display: flex; flex-direction: column;">
                    <span style="font-size: 18px; font-weight: 700;">Azuero Safety Services S.A</span>
                    <span style="font-size: 10px; opacity: 0.8; margin-top: -2px;">Somos tu seguridad</span>
                </div>
            </div>
        </div>
        
        <div class="header-actions">
            <div class="theme-switcher">
                <button class="theme-btn active" data-theme="light" onclick="changeTheme('light')">
                    <i class="fas fa-sun"></i> Light
                </button>
                <button class="theme-btn" data-theme="dark" onclick="changeTheme('dark')">
                    <i class="fas fa-moon"></i> Dark
                </button>
                <button class="theme-btn" data-theme="blue-dark" onclick="changeTheme('blue-dark')">
                    <i class="fas fa-water"></i> Ocean
                </button>
            </div>
            
            <a href="?lang=<?php echo $lang == 'es' ? 'en' : 'es'; ?>" class="btn btn-profile">
                <i class="fas fa-language"></i> <?php echo $lang == 'es' ? 'EN' : 'ES'; ?>
            </a>
            <a href="ciudadano-perfil.php" class="btn btn-profile">
                <i class="fas fa-user-circle"></i> <?php echo $t['profile']; ?>
            </a>
            <a href="logout.php" class="btn btn-logout">
                <i class="fas fa-sign-out-alt"></i> <?php echo $t['logout']; ?>
            </a>
        </div>
    </div>
    
    <!-- CONTENEDOR PRINCIPAL -->
    <div class="container">
        <!-- Usuario Info Card -->
        <div style="background: var(--bg-card); border-radius: 20px; padding: 25px; margin-bottom: 30px; box-shadow: var(--shadow-md); border: 1px solid var(--border-color);">
            <div style="display: flex; align-items: center; gap: 20px;">
                <div class="user-avatar" style="width: 70px; height: 70px; font-size: 28px;">
                    <?php echo strtoupper(substr($user['full_name'], 0, 1)); ?>
                </div>
                <div style="flex: 1;">
                    <h2 style="font-size: 24px; font-weight: 700; color: var(--text-primary); margin-bottom: 8px;">
                        <?php echo $t['welcome']; ?>, <?php echo htmlspecialchars($user['full_name']); ?>! 👋
                    </h2>
                    <p style="font-size: 14px; color: var(--text-secondary); display: flex; align-items: center; gap: 8px;">
                        <i class="fas fa-envelope"></i> <?php echo htmlspecialchars($user['email']); ?>
                        <?php if (!empty($user['phone'])): ?>
                        <span style="margin-left: 15px;"><i class="fas fa-phone"></i> <?php echo htmlspecialchars($user['phone']); ?></span>
                        <?php endif; ?>
                    </p>
                </div>
            </div>
        </div>

        <!-- Título de Sección -->
        <div class="page-title">
            <i class="fas fa-exclamation-triangle"></i>
            <?php echo $lang == 'es' ? 'Panel de Emergencias' : 'Emergency Panel'; ?>
        </div>
        <div class="page-subtitle">
            <?php echo $lang == 'es' ? 'Presiona el botón correspondiente a tu situación de emergencia' : 'Press the button corresponding to your emergency situation'; ?>
        </div>
    
        <!-- BOTONES DE EMERGENCIA -->
        <div class="emergency-buttons">
            <div id="medicalBtn" class="emergency-card medical-card">
                <div class="card-icon-wrapper">
                    <div class="card-icon">🚑</div>
                </div>
                <div class="card-title"><?php echo $t['medical_btn']; ?></div>
                <div class="card-desc"><?php echo $t['medical_desc']; ?></div>
                <div class="card-badge">
                    <i class="fas fa-bolt"></i> <?php echo $lang == 'es' ? 'URGENTE' : 'URGENT'; ?>
                </div>
            </div>
        
            <div class="emergency-card general-card" onclick="openGeneralModal()">
                <div class="card-icon-wrapper">
                    <div class="card-icon">�</div>
                </div>
                <div class="card-title"><?php echo $t['general_btn']; ?></div>
                <div class="card-desc"><?php echo $t['general_desc']; ?></div>
                <div class="card-badge">
                    <i class="fas fa-fire"></i> <?php echo $lang == 'es' ? 'EMERGENCIA' : 'EMERGENCY'; ?>
                </div>
            </div>
        
            <div class="emergency-card police-card" onclick="openPoliceModal()">
                <div class="card-icon-wrapper">
                    <div class="card-icon">👮</div>
                </div>
                <div class="card-title"><?php echo $t['police_btn']; ?></div>
                <div class="card-desc"><?php echo $t['police_desc']; ?></div>
                <div class="card-badge">
                    <i class="fas fa-shield-alt"></i> <?php echo $lang == 'es' ? 'SEGURIDAD' : 'SECURITY'; ?>
                </div>
            </div>
        </div>
    
        <!-- SECCIÓN DE ALERTAS ACTIVAS -->
        <div class="alerts-section">
            <div class="section-title">
                <i class="fas fa-bell"></i>
                <?php echo $t['my_alerts']; ?>
            </div>
            <?php if (count($active_alerts) > 0): ?>
                <?php foreach ($active_alerts as $alert): ?>
                <div class="alert-item <?php echo $alert['alert_type']; ?>">
                    <div class="alert-info">
                        <div class="alert-type">
                            <span class="alert-type-icon">
                                <?php 
                                $icons = ['medical' => '🚑', 'fire' => '🔥', 'police' => '👮', 'other' => '⚠️'];
                                echo $icons[$alert['alert_type']] ?? '⚠️'; 
                                ?>
                            </span>
                            <?php echo strtoupper($alert['alert_type']); ?> - <?php echo strtoupper($alert['priority']); ?>
                        </div>
                        <div class="alert-status">
                            <div class="alert-status-row">
                                <i class="fas fa-clock"></i>
                                <span><?php echo $t['created']; ?>: <?php echo date('d/m/Y H:i', strtotime($alert['created_at'])); ?></span>
                            </div>
                            <?php if ($alert['operator_name']): ?>
                            <div class="alert-status-row">
                                <i class="fas fa-user-shield"></i>
                                <span><?php echo $t['assigned']; ?>: <?php echo htmlspecialchars($alert['operator_name']); ?></span>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    <span class="status-badge status-<?php echo $alert['status'] == 'pending' ? 'pending' : 'progress'; ?>">
                        <i class="fas fa-circle"></i>
                        <?php echo $alert['status'] == 'pending' ? $t['pending'] : $t['in_progress']; ?>
                    </span>
                </div>
                <?php endforeach; ?>
            <?php else: ?>
                <div class="no-alerts">
                    <i class="fas fa-check-circle"></i>
                    <p><?php echo $t['no_alerts']; ?></p>
                </div>
            <?php endif; ?>
        </div>
    </div>
    
    <!-- Modal Emergencia General -->
    <div id="generalModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <div class="modal-title">🚨 <?php echo $t['general_btn']; ?></div>
                <button class="close-modal" onclick="closeModal('generalModal')">&times;</button>
            </div>
            <div id="generalGpsStatus"></div>
            <form id="generalForm" onsubmit="submitGeneralEmergency(event)">
                <div class="form-group">
                    <label><?php echo $lang == 'es' ? 'Tipo de Emergencia' : 'Emergency Type'; ?></label>
                    <select name="emergency_type" required>
                        <option value="fire"><?php echo $lang == 'es' ? '🔥 Incendio' : '🔥 Fire'; ?></option>
                        <option value="accident"><?php echo $lang == 'es' ? '🚗 Accidente' : '🚗 Accident'; ?></option>
                        <option value="rescue"><?php echo $lang == 'es' ? '🆘 Rescate' : '🆘 Rescue'; ?></option>
                        <option value="other"><?php echo $lang == 'es' ? '⚠️ Otra' : '⚠️ Other'; ?></option>
                    </select>
                </div>
                <div class="form-group">
                    <label><?php echo $lang == 'es' ? 'Describe la situación' : 'Describe the situation'; ?></label>
                    <textarea name="description" required></textarea>
                </div>
                <input type="hidden" name="latitude" id="generalLat">
                <input type="hidden" name="longitude" id="generalLng">
                <button type="submit" class="btn-submit"><?php echo $lang == 'es' ? 'Enviar Alerta' : 'Send Alert'; ?></button>
                <button type="button" class="btn-cancel" onclick="closeModal('generalModal')"><?php echo $lang == 'es' ? 'Cancelar' : 'Cancel'; ?></button>
            </form>
            <div id="generalLoading" class="loading">
                <div class="spinner"></div>
                <p><?php echo $t['sending']; ?></p>
            </div>
        </div>
    </div>
    
    <!-- Modal Asistencia Policial -->
    <div id="policeModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <div class="modal-title">👮 <?php echo $t['police_btn']; ?></div>
                <button class="close-modal" onclick="closeModal('policeModal')">&times;</button>
            </div>
            <div id="policeGpsStatus"></div>
            <div id="policeQuestion" style="display: block;">
                <h3 style="text-align: center; margin-bottom: 20px; color: #333;">
                    <?php echo $lang == 'es' ? '¿Es una emergencia inmediata?' : 'Is this an immediate emergency?'; ?>
                </h3>
                <div class="yes-no-buttons">
                    <button class="btn-yes" onclick="policeImmediate()">
                        <?php echo $lang == 'es' ? 'SÍ - URGENTE' : 'YES - URGENT'; ?>
                    </button>
                    <button class="btn-no" onclick="policeDescribe()">
                        <?php echo $lang == 'es' ? 'NO - Puedo describir' : 'NO - I can describe'; ?>
                    </button>
                </div>
            </div>
            <form id="policeForm" onsubmit="submitPoliceEmergency(event)" style="display: none;">
                <div class="form-group">
                    <label><?php echo $lang == 'es' ? 'Describe la situación' : 'Describe the situation'; ?></label>
                    <textarea name="description" required></textarea>
                </div>
                <input type="hidden" name="latitude" id="policeLat">
                <input type="hidden" name="longitude" id="policeLng">
                <input type="hidden" name="is_immediate" id="policeImmediate">
                <button type="submit" class="btn-submit"><?php echo $lang == 'es' ? 'Enviar Alerta' : 'Send Alert'; ?></button>
                <button type="button" class="btn-cancel" onclick="closeModal('policeModal')"><?php echo $lang == 'es' ? 'Cancelar' : 'Cancel'; ?></button>
            </form>
            <div id="policeLoading" class="loading">
                <div class="spinner"></div>
                <p><?php echo $t['sending']; ?></p>
            </div>
        </div>
    </div>

    <script>
    // Version: <?php echo time(); ?> - Anti doble envío implementado
    // Bandera para prevenir envíos duplicados
    let isSubmitting = false;
    let lastSubmitTime = 0;
    
    function getGPS(callback) {
        if ("geolocation" in navigator) {
            // Opciones mejoradas para obtener GPS más preciso
            const options = {
                enableHighAccuracy: true, // Usar GPS de alta precisión
                timeout: 10000, // Esperar hasta 10 segundos
                maximumAge: 0 // No usar caché
            };
            
            navigator.geolocation.getCurrentPosition(
                function(position) {
                    console.log('📍 GPS obtenido:', position.coords.latitude, position.coords.longitude);
                    callback({
                        success: true,
                        latitude: position.coords.latitude,
                        longitude: position.coords.longitude,
                        accuracy: position.coords.accuracy
                    });
                },
                function(error) {
                    console.error('❌ Error GPS:', error.message);
                    callback({
                        success: false,
                        error: error.message
                    });
                },
                options
            );
        } else {
            callback({
                success: false,
                error: "<?php echo $lang == 'es' ? 'GPS no disponible en este navegador' : 'GPS not available in this browser'; ?>"
            });
        }
    }
    
    function triggerMedicalEmergency(event) {
        // Prevenir propagación del evento
        if (event) {
            event.preventDefault();
            event.stopPropagation();
            event.stopImmediatePropagation();
        }
        
        // Verificar que no haya envío reciente (últimos 3 segundos)
        const now = Date.now();
        if (now - lastSubmitTime < 3000) {
            console.log('❌ Envío demasiado reciente, esperando... (hace ' + (now - lastSubmitTime) + 'ms)');
            return false;
        }
        
        // Prevenir múltiples clics
        if (isSubmitting) {
            console.log('❌ Ya hay un envío en progreso, ignorando clic');
            return false;
        }
        
        // Bloquear INMEDIATAMENTE antes del confirm
        isSubmitting = true;
        lastSubmitTime = now;
        
        // Confirmar con el usuario
        if (!confirm("<?php echo $lang == 'es' ? '¿Confirmar EMERGENCIA MÉDICA inmediata?' : 'Confirm IMMEDIATE MEDICAL EMERGENCY?'; ?>")) {
            console.log('Usuario canceló la confirmación');
            // Desbloquear si cancela
            isSubmitting = false;
            lastSubmitTime = 0;
            return false;
        }
        
        console.log('✅ Enviando alerta médica...', new Date(now).toISOString());
        
        // Deshabilitar botón visualmente
        const btn = document.getElementById('medicalBtn');
        if (btn) {
            btn.style.opacity = '0.5';
            btn.style.pointerEvents = 'none';
            btn.style.cursor = 'not-allowed';
        }
        
        getGPS(function(gps) {
            const formData = new FormData();
            formData.append('type', 'medical');
            formData.append('latitude', gps.success ? gps.latitude : '');
            formData.append('longitude', gps.success ? gps.longitude : '');
            formData.append('timestamp', now); // Agregar timestamp único
            
            console.log('📤 Enviando datos:', {
                type: 'medical',
                latitude: gps.success ? gps.latitude : '',
                longitude: gps.success ? gps.longitude : '',
                timestamp: now
            });
            
            fetch('procesar-alerta.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                console.log('✅ Respuesta del servidor:', data);
                if (data.success) {
                    alert("<?php echo $lang == 'es' ? '✅ Alerta médica enviada. Ayuda en camino.' : '✅ Medical alert sent. Help is on the way.'; ?>");
                    // Actualizar la página después de 1 segundo
                    setTimeout(() => {
                        window.location.href = 'panel-ciudadano.php';
                    }, 1000);
                } else {
                    alert("<?php echo $lang == 'es' ? '❌ Error al enviar alerta: ' + (data.error || 'Error desconocido') : '❌ Error sending alert: ' + (data.error || 'Unknown error'); ?>");
                    isSubmitting = false;
                    if (btn) {
                        btn.style.opacity = '1';
                        btn.style.pointerEvents = 'auto';
                        btn.style.cursor = 'pointer';
                    }
                }
            })
            .catch(error => {
                console.error('❌ Error en el envío:', error);
                alert("<?php echo $lang == 'es' ? '❌ Error al enviar alerta' : '❌ Error sending alert'; ?>");
                isSubmitting = false;
                if (btn) {
                    btn.style.opacity = '1';
                    btn.style.pointerEvents = 'auto';
                    btn.style.cursor = 'pointer';
                }
            });
        });
        
        return false;
    }
    
    function openGeneralModal() {
        document.getElementById('generalModal').classList.add('active');
        document.getElementById('generalGpsStatus').innerHTML = '<div class="gps-status gps-loading"><i class="fas fa-spinner fa-spin"></i> <?php echo $t['gps_wait']; ?></div>';
        
        getGPS(function(gps) {
            if (gps.success) {
                document.getElementById('generalLat').value = gps.latitude;
                document.getElementById('generalLng').value = gps.longitude;
                const accuracyText = gps.accuracy ? ' (±' + Math.round(gps.accuracy) + 'm)' : '';
                document.getElementById('generalGpsStatus').innerHTML = '<div class="gps-status gps-success"><i class="fas fa-check-circle"></i> <?php echo $t['gps_success']; ?>' + accuracyText + '</div>';
            } else {
                document.getElementById('generalGpsStatus').innerHTML = '<div class="gps-status gps-error"><i class="fas fa-exclamation-circle"></i> <?php echo $t['gps_error']; ?>: ' + gps.error + '</div>';
            }
        });
    }
    
    function submitGeneralEmergency(e) {
        e.preventDefault();
        if (isSubmitting) return; // Prevenir doble envío
        
        isSubmitting = true;
        document.getElementById('generalForm').style.display = 'none';
        document.getElementById('generalLoading').classList.add('active');
        
        const formData = new FormData(e.target);
        formData.append('type', 'general');
        
        fetch('procesar-alerta.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert("<?php echo $lang == 'es' ? '✅ Alerta enviada correctamente' : '✅ Alert sent successfully'; ?>");
                location.reload();
            } else {
                alert("<?php echo $lang == 'es' ? '❌ Error al enviar alerta' : '❌ Error sending alert'; ?>");
                document.getElementById('generalForm').style.display = 'block';
                document.getElementById('generalLoading').classList.remove('active');
                isSubmitting = false;
            }
        })
        .catch(error => {
            alert("<?php echo $lang == 'es' ? '❌ Error al enviar alerta' : '❌ Error sending alert'; ?>");
            document.getElementById('generalForm').style.display = 'block';
            document.getElementById('generalLoading').classList.remove('active');
            isSubmitting = false;
        });
    }
    
    function openPoliceModal() {
        document.getElementById('policeModal').classList.add('active');
        document.getElementById('policeQuestion').style.display = 'block';
        document.getElementById('policeForm').style.display = 'none';
        document.getElementById('policeGpsStatus').innerHTML = '';
    }
    
    function policeImmediate() {
        if (isSubmitting) return; // Prevenir doble envío
        
        isSubmitting = true;
        document.getElementById('policeQuestion').style.display = 'none';
        document.getElementById('policeLoading').classList.add('active');
        
        getGPS(function(gps) {
            const formData = new FormData();
            formData.append('type', 'police');
            formData.append('is_immediate', '1');
            formData.append('latitude', gps.success ? gps.latitude : '');
            formData.append('longitude', gps.success ? gps.longitude : '');
            
            fetch('procesar-alerta.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert("<?php echo $lang == 'es' ? '✅ Alerta policial inmediata enviada' : '✅ Immediate police alert sent'; ?>");
                    location.reload();
                } else {
                    alert("<?php echo $lang == 'es' ? '❌ Error al enviar alerta' : '❌ Error sending alert'; ?>");
                    closeModal('policeModal');
                    isSubmitting = false;
                }
            })
            .catch(error => {
                alert("<?php echo $lang == 'es' ? '❌ Error al enviar alerta' : '❌ Error sending alert'; ?>");
                closeModal('policeModal');
                isSubmitting = false;
            });
        });
    }
    
    function policeDescribe() {
        document.getElementById('policeQuestion').style.display = 'none';
        document.getElementById('policeForm').style.display = 'block';
        document.getElementById('policeImmediate').value = '0';
        document.getElementById('policeGpsStatus').innerHTML = '<div class="gps-status gps-loading"><i class="fas fa-spinner fa-spin"></i> <?php echo $t['gps_wait']; ?></div>';
        
        getGPS(function(gps) {
            if (gps.success) {
                document.getElementById('policeLat').value = gps.latitude;
                document.getElementById('policeLng').value = gps.longitude;
                const accuracyText = gps.accuracy ? ' (±' + Math.round(gps.accuracy) + 'm)' : '';
                document.getElementById('policeGpsStatus').innerHTML = '<div class="gps-status gps-success"><i class="fas fa-check-circle"></i> <?php echo $t['gps_success']; ?>' + accuracyText + '</div>';
            } else {
                document.getElementById('policeGpsStatus').innerHTML = '<div class="gps-status gps-error"><i class="fas fa-exclamation-circle"></i> <?php echo $t['gps_error']; ?>: ' + gps.error + '</div>';
            }
        });
    }
    
    function submitPoliceEmergency(e) {
        e.preventDefault();
        if (isSubmitting) return; // Prevenir doble envío
        
        isSubmitting = true;
        document.getElementById('policeForm').style.display = 'none';
        document.getElementById('policeLoading').classList.add('active');
        
        const formData = new FormData(e.target);
        formData.append('type', 'police');
        
        fetch('procesar-alerta.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert("<?php echo $lang == 'es' ? '✅ Alerta policial enviada' : '✅ Police alert sent'; ?>");
                location.reload();
            } else {
                alert("<?php echo $lang == 'es' ? '❌ Error al enviar alerta' : '❌ Error sending alert'; ?>");
                document.getElementById('policeForm').style.display = 'block';
                document.getElementById('policeLoading').classList.remove('active');
                isSubmitting = false;
            }
        })
        .catch(error => {
            alert("<?php echo $lang == 'es' ? '❌ Error al enviar alerta' : '❌ Error sending alert'; ?>");
            document.getElementById('policeForm').style.display = 'block';
            document.getElementById('policeLoading').classList.remove('active');
            isSubmitting = false;
        });
    }
    
    function closeModal(modalId) {
        document.getElementById(modalId).classList.remove('active');
        isSubmitting = false; // Resetear bandera al cerrar modal
    }
    
    window.onclick = function(event) {
        if (event.target.classList.contains('modal')) {
            event.target.classList.remove('active');
        }
    }
    
    // ============ AGREGAR EVENT LISTENERS A LOS BOTONES ============
    document.addEventListener('DOMContentLoaded', function() {
        console.log('🔄 DOM cargado, inicializando botones...');
        
        // Botón médico - usar addEventListener con { once: true } para ejecutar solo una vez por carga
        const medicalBtn = document.getElementById('medicalBtn');
        if (medicalBtn) {
            // Remover cualquier listener previo (por si acaso)
            medicalBtn.replaceWith(medicalBtn.cloneNode(true));
            
            // Agregar el listener al botón nuevo
            const newMedicalBtn = document.getElementById('medicalBtn');
            newMedicalBtn.addEventListener('click', triggerMedicalEmergency);
            console.log('✅ Event listener agregado al botón médico');
        } else {
            console.error('❌ No se encontró el botón médico');
        }

        // Cargar tema guardado
        const savedTheme = localStorage.getItem('citizen_theme') || 'light';
        changeTheme(savedTheme);
    });

    // ============ SISTEMA DE TEMAS ============
    function changeTheme(theme) {
        document.body.setAttribute('data-theme', theme);
        localStorage.setItem('citizen_theme', theme);
        
        // Actualizar botones activos
        document.querySelectorAll('.theme-btn').forEach(function(btn) {
            btn.classList.remove('active');
            if(btn.getAttribute('data-theme') === theme) {
                btn.classList.add('active');
            }
        });
    }
    
    // ============ SISTEMA DE NOTIFICACIONES EN TIEMPO REAL ============
    let lastCheckCitizen = Math.floor(Date.now() / 1000);
    
    function checkMyAlerts() {
        fetch('check-my-alerts.php?last_check=' + lastCheckCitizen)
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    lastCheckCitizen = data.timestamp;
                    
                    // Siempre actualizar la lista de alertas activas
                    if (data.active_alerts) {
                        updateMyAlertsList(data.active_alerts);
                    }
                    
                    // Si hay actualizaciones, notificar al ciudadano
                    if (data.has_updates && data.updated_alerts && data.updated_alerts.length > 0) {
                        data.updated_alerts.forEach(alert => {
                            showCitizenNotification(alert);
                        });
                    }
                }
            })
            .catch(error => console.error('Error checking my alerts:', error));
    }
    
    function showCitizenNotification(alert) {
        const statusText = {
            'pending': '⏳ Pendiente',
            'in_progress': '🔄 En Progreso',
            'resolved': '✅ Resuelta'
        };
        
        const statusColors = {
            'pending': '#f59e0b',
            'in_progress': '#3b82f6',
            'resolved': '#10b981'
        };
        
        // Notificación visual moderna
        const notification = document.createElement('div');
        notification.style.cssText = '\
            position: fixed; \
            top: 80px; \
            right: 20px; \
            background: var(--bg-card); \
            color: var(--text-primary); \
            padding: 20px 25px; \
            border-radius: 15px; \
            font-weight: 600; \
            font-size: 15px;\
            box-shadow: var(--shadow-xl);\
            z-index: 10000; \
            animation: slideIn 0.4s cubic-bezier(0.4, 0, 0.2, 1);\
            max-width: 400px;\
            border-left: 5px solid ' + statusColors[alert.status] + ';\
            border: 2px solid var(--border-color);\
            border-left: 5px solid ' + statusColors[alert.status] + ';';
        
        let message = '<div style="display: flex; align-items: center; gap: 12px; margin-bottom: 10px;">';
        message += '<div style="width: 40px; height: 40px; background: ' + statusColors[alert.status] + '; border-radius: 10px; display: flex; align-items: center; justify-content: center; font-size: 20px;">⚡</div>';
        message += '<div style="flex: 1;"><strong style="display: block; font-size: 16px;">' + statusText[alert.status] + '</strong>';
        
        if (alert.status === 'in_progress') {
            message += '<span style="font-size: 13px; color: var(--text-secondary);">Un operador está atendiendo tu alerta</span>';
        } else if (alert.status === 'resolved') {
            message += '<span style="font-size: 13px; color: var(--text-secondary);">Tu alerta ha sido resuelta</span>';
        }
        message += '</div></div>';
        
        if (alert.operator_response) {
            message += '<div style="margin-top: 12px; padding: 12px; background: var(--bg-secondary); border-radius: 10px; font-size: 13px; font-weight: 500; color: var(--text-secondary);">';
            message += '<i class="fas fa-comment-alt" style="margin-right: 6px; color: ' + statusColors[alert.status] + ';"></i> ';
            message += alert.operator_response;
            message += '</div>';
        }
        
        notification.innerHTML = message;
        document.body.appendChild(notification);
        
        // Sonido de notificación
        try {
            const audio = new Audio('data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmwhBDGH0O/bljkIF2Ey7+mgUxUMUKXh8bllHAU2jdXvxnMoCCV7zO7bky4GEliw6eqnWxAJPZfe8L9qJQQ0iM/u2Z');
            audio.volume = 0.4;
            audio.play().catch(function() {});
        } catch(e) {}
        
        // Auto-ocultar después de 8 segundos
        setTimeout(function() {
            notification.style.animation = 'slideOut 0.3s';
            setTimeout(function() { notification.remove(); }, 300);
        }, 8000);
        
        // Vibración en móviles
        if ('vibrate' in navigator) {
            navigator.vibrate([200, 100, 200]);
        }

        // Notificación del navegador si está permitido
        if ('Notification' in window && Notification.permission === 'granted') {
            new Notification('Azuero Safety Services S.A', {
                body: alert.status === 'resolved' ? 'Tu alerta ha sido resuelta' : 'Actualización de tu alerta',
                icon: '/favicon.ico',
                badge: '/favicon.ico'
            });
        }
    }
    
    function updateMyAlertsList(alerts) {
        const container = document.querySelector('.alerts-section');
        if (!container) {
            console.error('Contenedor .alerts-section no encontrado');
            return;
        }
        
        console.log('Actualizando lista de alertas del ciudadano:', alerts.length);
        
        // Recrear todo el contenido
        container.innerHTML = '<div class="section-title">🔔 <?php echo $t['my_alerts']; ?></div>';
        
        if (alerts && alerts.length > 0) {
            alerts.forEach(alert => {
                const alertDiv = document.createElement('div');
                alertDiv.className = `alert-item ${alert.alert_type || 'other'}`;
                
                const statusEmoji = {
                    'pending': '⏳',
                    'in_progress': '🔄',
                    'resolved': '✅'
                };
                
                const statusText = {
                    'pending': '<?php echo $t['pending']; ?>',
                    'in_progress': '<?php echo $t['in_progress']; ?>',
                    'resolved': 'Resuelta'
                };
                
                const statusClass = {
                    'pending': 'status-pending',
                    'in_progress': 'status-progress',
                    'resolved': 'status-resolved'
                };
                
                let alertHTML = `
                    <div class="alert-info">
                        <div class="alert-type">${(alert.alert_type || 'other').toUpperCase()} - ${(alert.priority || 'medium').toUpperCase()}</div>
                        <div class="alert-status">
                            <?php echo $t['created']; ?>: ${formatCitizenDate(alert.created_at)}
                `;
                
                if (alert.operator_name) {
                    alertHTML += ` | <?php echo $t['assigned']; ?>: ${alert.operator_name}`;
                }
                
                alertHTML += `</div>`;
                
                if (alert.operator_response) {
                    alertHTML += `<div class="operator-response">📝 ${alert.operator_response}</div>`;
                }
                
                alertHTML += `
                    </div>
                    <span class="status-badge ${statusClass[alert.status] || 'status-pending'}">
                        ${statusEmoji[alert.status] || '⏳'} ${statusText[alert.status] || alert.status}
                    </span>
                `;
                
                alertDiv.innerHTML = alertHTML;
                container.appendChild(alertDiv);
            });
        } else {
            const noAlerts = document.createElement('div');
            noAlerts.className = 'no-alerts';
            noAlerts.textContent = '<?php echo $t['no_alerts']; ?>';
            container.appendChild(noAlerts);
        }
    }
    
    // Formatear fecha para ciudadano
    function formatCitizenDate(dateString) {
        if (!dateString) return 'N/A';
        const date = new Date(dateString);
        if (isNaN(date.getTime())) return 'Fecha inválida';
        
        const day = String(date.getDate()).padStart(2, '0');
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const year = date.getFullYear();
        const hours = String(date.getHours()).padStart(2, '0');
        const minutes = String(date.getMinutes()).padStart(2, '0');
        return `${day}/${month}/${year} ${hours}:${minutes}`;
    }
    
    // Iniciar verificación cada 5 segundos
    setInterval(checkMyAlerts, 5000);
    checkMyAlerts();
    
    // Pedir permiso para notificaciones
    if ('Notification' in window && Notification.permission === 'default') {
        Notification.requestPermission();
    }
    
    // ============================================
    // PWA: Registro del Service Worker
    // ============================================
    if ('serviceWorker' in navigator) {
        window.addEventListener('load', () => {
            navigator.serviceWorker.register('service-worker.js')
                .then((registration) => {
                    console.log('[PWA] Service Worker registrado:', registration.scope);
                    
                    // Verificar actualizaciones cada 60 segundos
                    setInterval(() => {
                        registration.update();
                    }, 60000);
                })
                .catch((error) => {
                    console.error('[PWA] Error al registrar Service Worker:', error);
                });
        });
    }
    
    // ============================================
    // PWA: Botón de instalación
    // ============================================
    let deferredPrompt;
    
    window.addEventListener('beforeinstallprompt', (e) => {
        console.log('[PWA] Evento beforeinstallprompt detectado');
        e.preventDefault();
        deferredPrompt = e;
        
        // Mostrar banner de instalación personalizado (opcional)
        showInstallBanner();
    });
    
    function showInstallBanner() {
        // Crear banner de instalación si no existe
        if (document.getElementById('install-banner')) return;
        
        const banner = document.createElement('div');
        banner.id = 'install-banner';
        banner.style.cssText = `
            position: fixed;
            bottom: 20px;
            left: 50%;
            transform: translateX(-50%);
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            padding: 15px 25px;
            border-radius: 50px;
            box-shadow: var(--shadow-xl);
            display: flex;
            align-items: center;
            gap: 15px;
            z-index: 10000;
            animation: slideUp 0.3s ease;
        `;
        
        banner.innerHTML = `
            <i class="fas fa-download" style="font-size: 20px;"></i>
            <span style="font-weight: 500;">Instalar App</span>
            <button id="install-btn" style="
                background: rgba(255,255,255,0.2);
                border: 2px solid white;
                color: white;
                padding: 8px 20px;
                border-radius: 25px;
                cursor: pointer;
                font-weight: 600;
                transition: all 0.3s;
            ">Instalar</button>
            <button id="dismiss-btn" style="
                background: transparent;
                border: none;
                color: white;
                cursor: pointer;
                font-size: 20px;
                padding: 5px;
            "><i class="fas fa-times"></i></button>
        `;
        
        document.body.appendChild(banner);
        
        // Botón instalar
        document.getElementById('install-btn').addEventListener('click', async () => {
            if (!deferredPrompt) return;
            
            deferredPrompt.prompt();
            const { outcome } = await deferredPrompt.userChoice;
            console.log('[PWA] Resultado de instalación:', outcome);
            
            deferredPrompt = null;
            banner.remove();
        });
        
        // Botón cerrar
        document.getElementById('dismiss-btn').addEventListener('click', () => {
            banner.remove();
        });
    }
    
    window.addEventListener('appinstalled', () => {
        console.log('[PWA] App instalada correctamente');
        deferredPrompt = null;
        
        const banner = document.getElementById('install-banner');
        if (banner) banner.remove();
    });
    </script>
</body>
</html>
