<?php
require_once 'config.php';

// Cambiar idioma si se solicita
if (isset($_GET['lang']) && in_array($_GET['lang'], ['es', 'en'])) {
    $_SESSION['lang'] = $_GET['lang'];
    header('Location: registro-ciudadano.php');
    exit;
}

$lang = $_SESSION['lang'] ?? 'es';

$translations = [
    'es' => [
        'title' => 'Registro de Ciudadano',
        'subtitle' => 'Únete al sistema de emergencias Azuero Safety',
        'step1' => 'Paso 1: Información Personal',
        'step2' => 'Paso 2: Información Médica',
        'step3' => 'Paso 3: Contacto de Emergencia',
        'step4' => 'Paso 4: Código de Servicio',
        'full_name' => 'Nombre Completo',
        'email' => 'Correo Electrónico',
        'phone' => 'Teléfono',
        'cedula' => 'Cédula',
        'blood_type' => 'Tipo de Sangre',
        'allergies' => 'Alergias',
        'medical_conditions' => 'Condiciones Médicas',
        'emergency_contact' => 'Nombre del Contacto',
        'emergency_phone' => 'Teléfono de Emergencia',
        'service_code' => 'Código de Servicio',
        'code_help' => 'Ingresa el código que te proporcionó el administrador (Ej: SC-2025-0001)',
        'next' => 'Siguiente',
        'previous' => 'Anterior',
        'register' => 'Completar Registro',
        'back_login' => 'Volver al Login',
        'success' => '✅ ¡Registro completado exitosamente!',
        'your_code' => 'Tu código de servicio es',
        'save_code' => 'Guarda este código, lo necesitarás para acceder',
        'error_code' => '❌ Código de servicio inválido o ya usado',
        'error_email' => '❌ El correo electrónico ya está registrado',
        'error_cedula' => '❌ La cédula ya está registrada'
    ],
    'en' => [
        'title' => 'Citizen Registration',
        'subtitle' => 'Join Azuero Safety emergency system',
        'step1' => 'Step 1: Personal Information',
        'step2' => 'Step 2: Medical Information',
        'step3' => 'Step 3: Emergency Contact',
        'step4' => 'Step 4: Service Code',
        'full_name' => 'Full Name',
        'email' => 'Email Address',
        'phone' => 'Phone Number',
        'cedula' => 'ID Number',
        'blood_type' => 'Blood Type',
        'allergies' => 'Allergies',
        'medical_conditions' => 'Medical Conditions',
        'emergency_contact' => 'Contact Name',
        'emergency_phone' => 'Emergency Phone',
        'service_code' => 'Service Code',
        'code_help' => 'Enter the code provided by administrator (Ex: SC-2025-0001)',
        'next' => 'Next',
        'previous' => 'Previous',
        'register' => 'Complete Registration',
        'back_login' => 'Back to Login',
        'success' => '✅ Registration completed successfully!',
        'your_code' => 'Your service code is',
        'save_code' => 'Save this code, you will need it to access',
        'error_code' => '❌ Invalid or already used service code',
        'error_email' => '❌ Email address already registered',
        'error_cedula' => '❌ ID number already registered'
    ]
];

$t = $translations[$lang];
$message = '';
$registered_code = '';

// Solo procesar cuando se hace el registro final
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['complete_registration'])) {
    $service_code = strtoupper(trim($_POST['service_code'] ?? ''));
    
    try {
        // Verificar código de servicio
        $stmt = $conn->prepare("SELECT * FROM service_codes WHERE code = ? AND status = 'available' AND expires_at > NOW()");
        $stmt->execute([$service_code]);
        $codeData = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$codeData) {
            $message = $t['error_code'];
            $step = 4;
        } else {
            // Verificar email único
            $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
            $stmt->execute([$_POST['email']]);
            if ($stmt->fetch()) {
                $message = $t['error_email'];
                $step = 4;
            } else {
                // Verificar cédula única
                if (!empty($_POST['cedula'])) {
                    $stmt = $conn->prepare("SELECT id FROM users WHERE cedula = ?");
                    $stmt->execute([$_POST['cedula']]);
                    if ($stmt->fetch()) {
                        $message = $t['error_cedula'];
                        $step = 4;
                    }
                }
                
                if (empty($message)) {
                    // Insertar usuario
                    $stmt = $conn->prepare("
                        INSERT INTO users (
                            service_code, full_name, email, phone, cedula, 
                            blood_type, allergies, medical_conditions, current_medications,
                            emergency_contact_name, emergency_contact_phone, emergency_contact_relation,
                            status, created_at
                        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'active', NOW())
                    ");
                    $stmt->execute([
                        $service_code,
                        $_POST['full_name'],
                        $_POST['email'],
                        $_POST['phone'],
                        $_POST['cedula'] ?? '',
                        $_POST['blood_type'] ?? '',
                        $_POST['allergies'] ?? '',
                        $_POST['medical_conditions'] ?? '',
                        $_POST['current_medications'] ?? '',
                        $_POST['emergency_contact_name'] ?? '',
                        $_POST['emergency_contact_phone'] ?? '',
                        $_POST['emergency_contact_relation'] ?? ''
                    ]);
                    
                    $user_id = $conn->lastInsertId();
                    
                    // Actualizar código de servicio
                    $stmt = $conn->prepare("UPDATE service_codes SET status = 'assigned', assigned_to = ? WHERE code = ?");
                    $stmt->execute([$user_id, $service_code]);
                    
                    $registered_code = $service_code;
                    $success = true;
                    
                    $registered_code = $service_code;
                }
            }
        }
    } catch(PDOException $e) {
        $message = "Error: " . $e->getMessage();
    }
}

$success = !empty($registered_code);
?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $t['title']; ?> - Azuero Safety</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #2e7d32 0%, #388e3c 100%);
            min-height: 100vh;
            padding: 40px 20px;
        }
        .container { max-width: 700px; margin: 0 auto; background: white; border-radius: 15px; box-shadow: 0 10px 40px rgba(0,0,0,0.3); overflow: hidden; }
        .header { background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%); color: white; padding: 30px; text-align: center; }
        .header h1 { font-size: 28px; margin-bottom: 10px; }
        .progress-bar { display: flex; padding: 20px; background: #f5f5f5; }
        .progress-step { flex: 1; text-align: center; position: relative; padding: 10px; }
        .progress-step::after { content: ''; position: absolute; top: 20px; right: 0; width: 100%; height: 2px; background: #ddd; z-index: 0; }
        .progress-step:last-child::after { display: none; }
        .progress-step.active .step-number { background: #2e7d32; color: white; }
        .progress-step.completed .step-number { background: #1976d2; color: white; }
        .step-number { width: 40px; height: 40px; border-radius: 50%; background: #ddd; color: #666; display: flex; align-items: center; justify-content: center; margin: 0 auto 10px; font-weight: bold; position: relative; z-index: 1; }
        .step-label { font-size: 12px; color: #666; }
        .content { padding: 40px; }
        .step-content { display: none; }
        .step-content.active { display: block; animation: fadeIn 0.3s; }
        @keyframes fadeIn { from { opacity: 0; transform: translateY(10px); } to { opacity: 1; transform: translateY(0); } }
        .form-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin-bottom: 20px; }
        .form-group { display: flex; flex-direction: column; }
        label { font-weight: 600; color: #333; margin-bottom: 8px; font-size: 14px; }
        input, select, textarea { padding: 12px; border: 2px solid #ddd; border-radius: 8px; font-size: 14px; transition: border-color 0.3s; font-family: inherit; }
        input:focus, select:focus, textarea:focus { outline: none; border-color: #2e7d32; }
        textarea { resize: vertical; min-height: 80px; }
        .button-group { display: flex; gap: 10px; margin-top: 30px; }
        .btn { flex: 1; padding: 14px 24px; border: none; border-radius: 8px; font-size: 16px; font-weight: 600; cursor: pointer; transition: all 0.3s; }
        .btn-primary { background: #2e7d32; color: white; }
        .btn-primary:hover { background: #388e3c; }
        .btn-secondary { background: #f5f5f5; color: #333; }
        .btn-secondary:hover { background: #e0e0e0; }
        .message { padding: 15px; border-radius: 8px; margin-bottom: 20px; text-align: center; }
        .message.error { background: #ffebee; color: #c62828; border: 2px solid #c62828; }
        .help-text { font-size: 12px; color: #666; margin-top: 5px; }
        .success-card { text-align: center; padding: 40px; }
        .success-card .icon { font-size: 80px; margin-bottom: 20px; }
        .success-card h2 { color: #2e7d32; margin-bottom: 15px; }
        .code-display { background: #e8f5e9; padding: 20px; border-radius: 10px; margin: 20px 0; border: 3px dashed #2e7d32; }
        .code-display .code { font-size: 32px; font-weight: bold; color: #1e3c72; letter-spacing: 2px; margin: 10px 0; }
        .lang-switcher { position: absolute; top: 20px; right: 20px; display: flex; gap: 5px; }
        .lang-btn { padding: 5px 10px; border: 2px solid rgba(255,255,255,0.3); background: rgba(255,255,255,0.1); color: white; border-radius: 5px; font-size: 12px; font-weight: 600; text-decoration: none; transition: all 0.3s; }
        .lang-btn.active { background: rgba(255,255,255,0.3); }
    </style>
</head>
<body>
    <div class="lang-switcher">
        <a href="?lang=es" class="lang-btn <?php echo $lang == 'es' ? 'active' : ''; ?>">🇪🇸 ES</a>
        <a href="?lang=en" class="lang-btn <?php echo $lang == 'en' ? 'active' : ''; ?>">🇺🇸 EN</a>
    </div>
    
    <div class="container">
        <div class="header">
            <h1>👥 <?php echo $t['title']; ?></h1>
            <p><?php echo $t['subtitle']; ?></p>
        </div>
        
        <?php if (!$success): ?>
        <div class="progress-bar">
            <div class="progress-step active">
                <div class="step-number">1</div>
                <div class="step-label">Personal</div>
            </div>
            <div class="progress-step">
                <div class="step-number">2</div>
                <div class="step-label">Médica</div>
            </div>
            <div class="progress-step">
                <div class="step-number">3</div>
                <div class="step-label">Emergencia</div>
            </div>
            <div class="progress-step">
                <div class="step-number">4</div>
                <div class="step-label">Código</div>
            </div>
        </div>
        <?php endif; ?>
        
        <div class="content">
            <?php if ($message): ?>
            <div class="message error"><?php echo $message; ?></div>
            <?php endif; ?>
            
            <?php if ($success): ?>
            <div class="success-card">
                <div class="icon">✅</div>
                <h2><?php echo $t['success']; ?></h2>
                <div class="code-display">
                    <p><?php echo $t['your_code']; ?>:</p>
                    <div class="code"><?php echo $registered_code; ?></div>
                    <p style="color: #666; font-size: 14px;"><?php echo $t['save_code']; ?></p>
                </div>
                <a href="ciudadano-login.php" class="btn btn-primary" style="display: inline-block; text-decoration: none;">→ Acceder a Mi Panel</a>
                <br><br>
                <a href="index.php" style="color: #666; font-size: 14px;">← <?php echo $t['back_login']; ?></a>
            </div>
            <?php else: ?>
            <form id="registrationForm" method="POST">
                <input type="hidden" name="complete_registration" value="1">
                
                <!-- Step 1: Personal Info -->
                <div class="step-content active" data-step="1">
                    <h2 style="margin-bottom: 20px;">📋 <?php echo $t['step1']; ?></h2>
                    <div class="form-grid">
                        <div class="form-group">
                            <label><?php echo $t['full_name']; ?> *</label>
                            <input type="text" name="full_name" id="full_name" required>
                        </div>
                        <div class="form-group">
                            <label><?php echo $t['cedula']; ?></label>
                            <input type="text" name="cedula" id="cedula">
                        </div>
                        <div class="form-group">
                            <label><?php echo $t['email']; ?> *</label>
                            <input type="email" name="email" id="email" required>
                        </div>
                        <div class="form-group">
                            <label><?php echo $t['phone']; ?> *</label>
                            <input type="text" name="phone" id="phone" required>
                        </div>
                    </div>
                    <div class="button-group">
                        <button type="button" class="btn btn-secondary" onclick="window.location='index.php'">← <?php echo $t['back_login']; ?></button>
                        <button type="button" class="btn btn-primary" onclick="nextStep(2)"><?php echo $t['next']; ?> →</button>
                    </div>
                </div>
                
                <!-- Step 2: Medical Info -->
                <div class="step-content" data-step="2">
                    <h2 style="margin-bottom: 20px;">⚕️ <?php echo $t['step2']; ?></h2>
                    <div class="form-group" style="margin-bottom: 20px;">
                        <label><?php echo $t['blood_type']; ?></label>
                        <select name="blood_type" id="blood_type">
                            <option value="">Seleccionar...</option>
                            <option value="O+">O+</option>
                            <option value="O-">O-</option>
                            <option value="A+">A+</option>
                            <option value="A-">A-</option>
                            <option value="B+">B+</option>
                            <option value="B-">B-</option>
                            <option value="AB+">AB+</option>
                            <option value="AB-">AB-</option>
                        </select>
                    </div>
                    <div class="form-group" style="margin-bottom: 20px;">
                        <label><?php echo $t['allergies']; ?></label>
                        <textarea name="allergies" id="allergies"></textarea>
                    </div>
                    <div class="form-group" style="margin-bottom: 20px;">
                        <label><?php echo $t['medical_conditions']; ?></label>
                        <textarea name="medical_conditions" id="medical_conditions"></textarea>
                    </div>
                    <div class="form-group">
                        <label><?php echo $lang == 'es' ? 'Medicamentos Actuales' : 'Current Medications'; ?></label>
                        <textarea name="current_medications" id="current_medications"></textarea>
                    </div>
                    <div class="button-group">
                        <button type="button" class="btn btn-secondary" onclick="prevStep(1)">← <?php echo $t['previous']; ?></button>
                        <button type="button" class="btn btn-primary" onclick="nextStep(3)"><?php echo $t['next']; ?> →</button>
                    </div>
                </div>
                
                <!-- Step 3: Emergency Contact -->
                <div class="step-content" data-step="3">
                    <h2 style="margin-bottom: 20px;">🆘 <?php echo $t['step3']; ?></h2>
                    <p style="color: #666; margin-bottom: 20px; font-size: 14px;">
                        <?php echo $lang == 'es' 
                            ? '📞 Ingresa los datos de una persona de confianza que podamos contactar en caso de emergencia.' 
                            : '📞 Enter the details of a trusted person we can contact in case of emergency.'; ?>
                    </p>
                    <div class="form-grid">
                        <div class="form-group">
                            <label><?php echo $t['emergency_contact']; ?> *</label>
                            <input type="text" name="emergency_contact_name" id="emergency_contact_name" placeholder="<?php echo $lang == 'es' ? 'Nombre completo del contacto' : 'Full name of contact'; ?>" required>
                        </div>
                        <div class="form-group">
                            <label><?php echo $t['emergency_phone']; ?> *</label>
                            <input type="text" name="emergency_contact_phone" id="emergency_contact_phone" placeholder="<?php echo $lang == 'es' ? '6000-0000' : '6000-0000'; ?>" required>
                        </div>
                    </div>
                    <div class="form-group">
                        <label><?php echo $lang == 'es' ? 'Relación con el Contacto' : 'Relationship with Contact'; ?> *</label>
                        <input type="text" name="emergency_contact_relation" id="emergency_contact_relation" placeholder="<?php echo $lang == 'es' ? 'Ej: Madre, Esposo, Hermana' : 'Ex: Mother, Spouse, Sister'; ?>" required>
                    </div>
                    <div class="button-group">
                        <button type="button" class="btn btn-secondary" onclick="prevStep(2)">← <?php echo $t['previous']; ?></button>
                        <button type="button" class="btn btn-primary" onclick="nextStep(4)"><?php echo $t['next']; ?> →</button>
                    </div>
                </div>
                
                <!-- Step 4: Service Code -->
                <div class="step-content" data-step="4">
                    <h2 style="margin-bottom: 20px;">🔑 <?php echo $t['step4']; ?></h2>
                    <div class="form-group">
                        <label><?php echo $t['service_code']; ?> *</label>
                        <input type="text" name="service_code" id="service_code" placeholder="SC-2025-0001" maxlength="12" required style="text-transform: uppercase; font-size: 20px; text-align: center; letter-spacing: 3px;">
                        <p class="help-text">💡 <?php echo $t['code_help']; ?></p>
                    </div>
                    <div class="button-group">
                        <button type="button" class="btn btn-secondary" onclick="prevStep(3)">← <?php echo $t['previous']; ?></button>
                        <button type="submit" class="btn btn-primary">✅ <?php echo $t['register']; ?></button>
                    </div>
                </div>
            </form>
            
            <script>
            let currentStep = 1;
            
            function showStep(step) {
                // Ocultar todos los pasos
                document.querySelectorAll('.step-content').forEach(el => {
                    el.classList.remove('active');
                    el.style.display = 'none';
                });
                
                // Mostrar paso actual
                const stepEl = document.querySelector(`.step-content[data-step="${step}"]`);
                if (stepEl) {
                    stepEl.classList.add('active');
                    stepEl.style.display = 'block';
                }
                
                // Actualizar barra de progreso
                document.querySelectorAll('.progress-step').forEach((el, index) => {
                    el.classList.remove('active', 'completed');
                    if (index + 1 < step) {
                        el.classList.add('completed');
                    } else if (index + 1 === step) {
                        el.classList.add('active');
                    }
                });
                
                currentStep = step;
            }
            
            function validateStep(step) {
                const stepEl = document.querySelector(`.step-content[data-step="${step}"]`);
                const inputs = stepEl.querySelectorAll('input[required], select[required], textarea[required]');
                
                for (let input of inputs) {
                    if (!input.value.trim()) {
                        input.focus();
                        alert('<?php echo $lang == "es" ? "Por favor completa todos los campos requeridos" : "Please complete all required fields"; ?>');
                        return false;
                    }
                }
                return true;
            }
            
            function nextStep(step) {
                if (validateStep(currentStep)) {
                    showStep(step);
                    window.scrollTo(0, 0);
                }
            }
            
            function prevStep(step) {
                showStep(step);
                window.scrollTo(0, 0);
            }
            
            // Mostrar paso 1 al cargar
            document.addEventListener('DOMContentLoaded', function() {
                showStep(1);
            });
            </script>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>
