<?php
require_once 'config.php';

// Verificar que sea operador
if (!isset($_SESSION['operator_id'])) {
    header('Location: index.php');
    exit();
}

// Obtener información del operador
try {
    $stmt = $conn->prepare("SELECT * FROM operators WHERE id = ?");
    $stmt->execute([$_SESSION['operator_id']]);
    $operator = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$operator) {
        session_destroy();
        header('Location: index.php');
        exit();
    }
} catch (PDOException $e) {
    die("Error al obtener información del operador: " . $e->getMessage());
}

// Configuración de idioma
$lang = $operator['language'] ?? 'en';

$translations = [
    'es' => [
        'medical_history' => 'Historial Médico',
        'user_info' => 'Información del Usuario',
        'medical_info' => 'Información Médica',
        'emergency_contact' => 'Contacto de Emergencia',
        'alert_history' => 'Historial de Alertas',
        'back' => 'Volver',
        'full_name' => 'Nombre Completo',
        'email' => 'Email',
        'phone' => 'Teléfono',
        'cedula' => 'Cédula',
        'blood_type' => 'Tipo de Sangre',
        'allergies' => 'Alergias',
        'medical_conditions' => 'Condiciones Médicas',
        'current_medications' => 'Medicamentos Actuales',
        'contact_name' => 'Nombre del Contacto',
        'contact_phone' => 'Teléfono del Contacto',
        'contact_relation' => 'Relación',
        'not_specified' => 'No especificado',
        'none' => 'Ninguna',
        'status' => 'Estado',
        'active' => 'Activo',
        'inactive' => 'Inactivo',
        'member_since' => 'Miembro desde',
        'total_alerts' => 'Total de Alertas',
        'medical_alerts' => 'Alertas Médicas',
        'alert_type' => 'Tipo',
        'date' => 'Fecha',
        'priority' => 'Prioridad',
        'alert_status' => 'Estado',
        'view_alert' => 'Ver Alerta',
        'no_alerts' => 'Sin alertas registradas',
        'service_code' => 'Código de Servicio'
    ],
    'en' => [
        'medical_history' => 'Medical History',
        'user_info' => 'User Information',
        'medical_info' => 'Medical Information',
        'emergency_contact' => 'Emergency Contact',
        'alert_history' => 'Alert History',
        'back' => 'Back',
        'full_name' => 'Full Name',
        'email' => 'Email',
        'phone' => 'Phone',
        'cedula' => 'ID Number',
        'blood_type' => 'Blood Type',
        'allergies' => 'Allergies',
        'medical_conditions' => 'Medical Conditions',
        'current_medications' => 'Current Medications',
        'contact_name' => 'Contact Name',
        'contact_phone' => 'Contact Phone',
        'contact_relation' => 'Relationship',
        'not_specified' => 'Not specified',
        'none' => 'None',
        'status' => 'Status',
        'active' => 'Active',
        'inactive' => 'Inactive',
        'member_since' => 'Member since',
        'total_alerts' => 'Total Alerts',
        'medical_alerts' => 'Medical Alerts',
        'alert_type' => 'Type',
        'date' => 'Date',
        'priority' => 'Priority',
        'alert_status' => 'Status',
        'view_alert' => 'View Alert',
        'no_alerts' => 'No alerts registered',
        'service_code' => 'Service Code'
    ]
];

$t = $translations[$lang];

// Obtener ID del usuario
$user_id = $_GET['user_id'] ?? 0;

if (!$user_id) {
    header('Location: admin-panel.php');
    exit();
}

// Obtener información completa del usuario
try {
    $stmt = $conn->prepare("
        SELECT u.*,
               COUNT(DISTINCT a.id) as total_alerts,
               SUM(CASE WHEN a.alert_type = 'medical' THEN 1 ELSE 0 END) as medical_alert_count,
               SUM(CASE WHEN a.alert_type = 'fire' THEN 1 ELSE 0 END) as fire_alert_count,
               SUM(CASE WHEN a.alert_type = 'security' THEN 1 ELSE 0 END) as security_alert_count,
               MAX(a.created_at) as last_alert_date
        FROM users u
        LEFT JOIN alerts a ON u.id = a.user_id
        WHERE u.id = ?
        GROUP BY u.id
    ");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        header('Location: admin-panel.php');
        exit();
    }
} catch (PDOException $e) {
    die("Error: " . $e->getMessage());
}

// Obtener historial de alertas
try {
    $stmt = $conn->prepare("
        SELECT a.*, o.full_name as operator_name
        FROM alerts a
        LEFT JOIN operators o ON a.assigned_operator_id = o.id
        WHERE a.user_id = ?
        ORDER BY a.created_at DESC
        LIMIT 50
    ");
    $stmt->execute([$user_id]);
    $alerts = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $alerts = [];
}
?>
<!DOCTYPE html>
<html lang="<?php echo $lang; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <meta name="theme-color" content="#667eea">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="black-translucent">
    <meta name="apple-mobile-web-app-title" content="Historial Médico">
    <title><?php echo $t['medical_history']; ?> - <?php echo htmlspecialchars($user['full_name']); ?></title>

    <!-- PWA Manifest -->
    <link rel="manifest" href="manifest.json">

    <!-- Icons -->
    <link rel="icon" type="image/png" sizes="192x192" href="icon-192x192.png">
    <link rel="apple-touch-icon" href="icon-192x192.png">

    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary: #667eea;
            --primary-dark: #5568d3;
            --secondary: #764ba2;
            --success: #10b981;
            --warning: #f59e0b;
            --danger: #ef4444;
            --info: #3b82f6;
            --bg-primary: #ffffff;
            --bg-secondary: #f8fafc;
            --bg-card: #ffffff;
            --text-primary: #1e293b;
            --text-secondary: #64748b;
            --border-color: #e2e8f0;
            --shadow-sm: 0 1px 3px rgba(0,0,0,0.1);
            --shadow-md: 0 4px 6px rgba(0,0,0,0.1);
            --shadow-lg: 0 10px 25px rgba(0,0,0,0.15);
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: var(--bg-secondary);
            color: var(--text-primary);
            padding: 20px;
            -webkit-font-smoothing: antialiased;
            -moz-osx-font-smoothing: grayscale;
        }

        .container {
            max-width: 1400px;
            margin: 0 auto;
        }

        /* PWA Install Banner */
        .install-banner {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            padding: 15px 20px;
            display: none;
            align-items: center;
            justify-content: space-between;
            box-shadow: 0 -4px 20px rgba(0,0,0,0.2);
            z-index: 9999;
            animation: slideUp 0.3s ease;
        }

        @keyframes slideUp {
            from { transform: translateY(100%); }
            to { transform: translateY(0); }
        }

        .install-banner-content {
            flex: 1;
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .install-banner-icon {
            font-size: 32px;
        }

        .install-banner-text h4 {
            margin: 0;
            font-size: 16px;
            font-weight: 600;
        }

        .install-banner-text p {
            margin: 3px 0 0 0;
            font-size: 13px;
            opacity: 0.9;
        }

        .install-banner-buttons {
            display: flex;
            gap: 10px;
        }

        .install-btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            font-size: 14px;
        }

        .install-btn-primary {
            background: white;
            color: var(--primary);
        }

        .install-btn-primary:hover {
            transform: scale(1.05);
        }

        .install-btn-secondary {
            background: rgba(255,255,255,0.2);
            color: white;
        }

        .install-btn-secondary:hover {
            background: rgba(255,255,255,0.3);
        }

        .header {
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            padding: 30px;
            border-radius: 15px;
            margin-bottom: 30px;
            box-shadow: var(--shadow-lg);
        }

        .header h1 {
            font-size: 32px;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .header p {
            opacity: 0.9;
            font-size: 16px;
        }

        .back-btn {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            padding: 10px 20px;
            background: rgba(255, 255, 255, 0.2);
            color: white;
            text-decoration: none;
            border-radius: 8px;
            transition: all 0.3s ease;
            margin-bottom: 15px;
        }

        .back-btn:hover {
            background: rgba(255, 255, 255, 0.3);
            transform: translateX(-5px);
        }

        .grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .card {
            background: var(--bg-card);
            border-radius: 15px;
            padding: 25px;
            box-shadow: var(--shadow-md);
            border: 1px solid var(--border-color);
        }

        .card-header {
            display: flex;
            align-items: center;
            gap: 12px;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid var(--border-color);
        }

        .card-header i {
            font-size: 24px;
            color: var(--primary);
        }

        .card-header h2 {
            font-size: 20px;
            color: var(--text-primary);
        }

        .info-row {
            display: flex;
            justify-content: space-between;
            padding: 12px 0;
            border-bottom: 1px solid var(--border-color);
        }

        .info-row:last-child {
            border-bottom: none;
        }

        .info-label {
            font-weight: 600;
            color: var(--text-secondary);
            font-size: 14px;
        }

        .info-value {
            color: var(--text-primary);
            font-size: 14px;
            text-align: right;
            max-width: 60%;
        }

        .badge {
            display: inline-block;
            padding: 5px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 500;
        }

        .badge-success { background: rgba(16, 185, 129, 0.1); color: var(--success); }
        .badge-danger { background: rgba(239, 68, 68, 0.1); color: var(--danger); }
        .badge-warning { background: rgba(245, 158, 11, 0.1); color: var(--warning); }
        .badge-info { background: rgba(59, 130, 246, 0.1); color: var(--info); }

        .alert-card {
            background: var(--bg-card);
            border-radius: 15px;
            padding: 25px;
            box-shadow: var(--shadow-md);
            border: 1px solid var(--border-color);
        }

        .alert-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px;
            border-bottom: 1px solid var(--border-color);
            transition: background 0.3s ease;
        }

        .alert-item:hover {
            background: var(--bg-secondary);
        }

        .alert-item:last-child {
            border-bottom: none;
        }

        .alert-info {
            flex: 1;
        }

        .alert-type {
            font-weight: 600;
            margin-bottom: 5px;
        }

        .alert-date {
            font-size: 13px;
            color: var(--text-secondary);
        }

        .btn {
            padding: 8px 16px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 13px;
            font-weight: 500;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 6px;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
        }

        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: var(--text-secondary);
        }

        .empty-state i {
            font-size: 64px;
            opacity: 0.3;
            margin-bottom: 20px;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 30px;
        }

        .stat-box {
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            padding: 20px;
            border-radius: 12px;
            text-align: center;
        }

        .stat-value {
            font-size: 32px;
            font-weight: bold;
            margin-bottom: 5px;
        }

        .stat-label {
            font-size: 14px;
            opacity: 0.9;
        }

        .medical-warning {
            background: rgba(239, 68, 68, 0.1);
            border-left: 4px solid var(--danger);
            padding: 15px;
            border-radius: 8px;
            margin-top: 15px;
        }

        .medical-warning i {
            color: var(--danger);
            margin-right: 10px;
        }

        @media (max-width: 768px) {
            body {
                padding: 10px;
            }

            .container {
                padding: 0;
            }

            .header {
                padding: 20px 15px;
                border-radius: 10px;
                margin-bottom: 15px;
            }

            .header h1 {
                font-size: 22px;
                gap: 10px;
            }

            .header h1 i {
                font-size: 20px;
            }

            .header p {
                font-size: 14px;
            }

            .back-btn {
                padding: 8px 15px;
                font-size: 14px;
                margin-bottom: 10px;
            }

            .grid {
                grid-template-columns: 1fr;
                gap: 15px;
                margin-bottom: 15px;
            }

            .stats-grid {
                grid-template-columns: 1fr;
                gap: 10px;
                margin-bottom: 15px;
            }

            .stat-box {
                padding: 15px;
            }

            .stat-value {
                font-size: 28px;
            }

            .stat-label {
                font-size: 13px;
            }

            .card {
                padding: 20px 15px;
                border-radius: 12px;
            }

            .card-header {
                margin-bottom: 15px;
                padding-bottom: 12px;
            }

            .card-header i {
                font-size: 20px;
            }

            .card-header h2 {
                font-size: 18px;
            }

            .info-row {
                flex-direction: column;
                gap: 5px;
                padding: 10px 0;
            }

            .info-label {
                font-size: 13px;
            }

            .info-value {
                text-align: left;
                max-width: 100%;
                font-size: 14px;
            }

            .badge {
                font-size: 11px;
                padding: 4px 10px;
            }

            .alert-card {
                padding: 20px 15px;
                border-radius: 12px;
            }

            .alert-item {
                flex-direction: column;
                align-items: flex-start;
                padding: 12px;
                gap: 10px;
            }

            .alert-info {
                width: 100%;
            }

            .alert-type {
                display: flex;
                flex-wrap: wrap;
                gap: 5px;
                margin-bottom: 8px;
            }

            .alert-date {
                font-size: 12px;
            }

            .btn {
                width: 100%;
                justify-content: center;
                padding: 10px 16px;
                font-size: 14px;
            }

            .medical-warning {
                padding: 12px;
                font-size: 13px;
            }

            .empty-state {
                padding: 40px 15px;
            }

            .empty-state i {
                font-size: 48px;
            }

            /* Ajustes para el banner de instalación en móvil */
            .install-banner {
                flex-direction: column;
                padding: 15px;
                gap: 15px;
            }

            .install-banner-content {
                width: 100%;
            }

            .install-banner-buttons {
                width: 100%;
                flex-direction: column;
            }

            .install-btn {
                width: 100%;
                padding: 12px;
            }
        }

        @media (max-width: 480px) {
            body {
                padding: 5px;
            }

            .header {
                padding: 15px 10px;
            }

            .header h1 {
                font-size: 20px;
            }

            .card {
                padding: 15px 10px;
            }

            .stat-value {
                font-size: 24px;
            }
        }
    </style>
</head>
<body>
    <!-- PWA Install Banner -->
    <div class="install-banner" id="installBanner">
        <div class="install-banner-content">
            <div class="install-banner-icon">
                <i class="fas fa-mobile-alt"></i>
            </div>
            <div class="install-banner-text">
                <h4><?php echo $lang === 'es' ? 'Instalar Aplicación' : 'Install App'; ?></h4>
                <p><?php echo $lang === 'es' ? 'Accede rápidamente desde tu pantalla de inicio' : 'Quick access from your home screen'; ?></p>
            </div>
        </div>
        <div class="install-banner-buttons">
            <button class="install-btn install-btn-primary" id="installBtn">
                <i class="fas fa-download"></i> <?php echo $lang === 'es' ? 'Instalar' : 'Install'; ?>
            </button>
            <button class="install-btn install-btn-secondary" id="dismissBtn">
                <i class="fas fa-times"></i> <?php echo $lang === 'es' ? 'Ahora no' : 'Not now'; ?>
            </button>
        </div>
    </div>

    <div class="container">
        <div class="header">
            <a href="<?php echo $operator['role'] === 'admin' ? 'admin-panel.php' : 'dashboard.php'; ?>" class="back-btn">
                <i class="fas fa-arrow-left"></i> <?php echo $t['back']; ?>
            </a>
            <h1>
                <i class="fas fa-file-medical"></i>
                <?php echo $t['medical_history']; ?>
            </h1>
            <p><?php echo htmlspecialchars($user['full_name']); ?> - ID: #<?php echo $user['id']; ?></p>
        </div>

        <!-- Estadísticas -->
        <div class="stats-grid">
            <div class="stat-box">
                <div class="stat-value"><?php echo $user['total_alerts'] ?? 0; ?></div>
                <div class="stat-label"><?php echo $t['total_alerts']; ?></div>
            </div>
            <div class="stat-box" style="background: linear-gradient(135deg, #ef4444, #dc2626);">
                <div class="stat-value"><?php echo $user['medical_alert_count'] ?? 0; ?></div>
                <div class="stat-label"><?php echo $t['medical_alerts']; ?></div>
            </div>
            <div class="stat-box" style="background: linear-gradient(135deg, #10b981, #059669);">
                <div class="stat-value">
                    <span class="badge badge-<?php echo $user['status'] === 'active' ? 'success' : 'danger'; ?>">
                        <?php echo $user['status'] === 'active' ? $t['active'] : $t['inactive']; ?>
                    </span>
                </div>
                <div class="stat-label"><?php echo $t['status']; ?></div>
            </div>
        </div>

        <!-- Información del Usuario y Médica -->
        <div class="grid">
            <!-- Información Personal -->
            <div class="card">
                <div class="card-header">
                    <i class="fas fa-user"></i>
                    <h2><?php echo $t['user_info']; ?></h2>
                </div>

                <?php if (!empty($user['service_code'])): ?>
                <div class="info-row">
                    <span class="info-label"><?php echo $t['service_code']; ?></span>
                    <span class="info-value"><strong><?php echo htmlspecialchars($user['service_code']); ?></strong></span>
                </div>
                <?php endif; ?>

                <div class="info-row">
                    <span class="info-label"><?php echo $t['full_name']; ?></span>
                    <span class="info-value"><?php echo htmlspecialchars($user['full_name']); ?></span>
                </div>

                <div class="info-row">
                    <span class="info-label"><?php echo $t['email']; ?></span>
                    <span class="info-value"><?php echo htmlspecialchars($user['email']); ?></span>
                </div>

                <div class="info-row">
                    <span class="info-label"><?php echo $t['phone']; ?></span>
                    <span class="info-value"><?php echo htmlspecialchars($user['phone']); ?></span>
                </div>

                <?php if (!empty($user['cedula'])): ?>
                <div class="info-row">
                    <span class="info-label"><?php echo $t['cedula']; ?></span>
                    <span class="info-value"><?php echo htmlspecialchars($user['cedula']); ?></span>
                </div>
                <?php endif; ?>

                <div class="info-row">
                    <span class="info-label"><?php echo $t['member_since']; ?></span>
                    <span class="info-value"><?php echo date('d/m/Y', strtotime($user['created_at'])); ?></span>
                </div>
            </div>

            <!-- Información Médica -->
            <div class="card">
                <div class="card-header">
                    <i class="fas fa-heartbeat"></i>
                    <h2><?php echo $t['medical_info']; ?></h2>
                </div>

                <div class="info-row">
                    <span class="info-label"><?php echo $t['blood_type']; ?></span>
                    <span class="info-value">
                        <?php if (!empty($user['blood_type'])): ?>
                            <strong style="color: var(--danger); font-size: 16px;">
                                <?php echo htmlspecialchars($user['blood_type']); ?>
                            </strong>
                        <?php else: ?>
                            <em style="color: var(--text-secondary);"><?php echo $t['not_specified']; ?></em>
                        <?php endif; ?>
                    </span>
                </div>

                <div class="info-row">
                    <span class="info-label"><?php echo $t['allergies']; ?></span>
                    <span class="info-value">
                        <?php if (!empty($user['allergies'])): ?>
                            <?php echo nl2br(htmlspecialchars($user['allergies'])); ?>
                        <?php else: ?>
                            <em style="color: var(--text-secondary);"><?php echo $t['none']; ?></em>
                        <?php endif; ?>
                    </span>
                </div>

                <div class="info-row">
                    <span class="info-label"><?php echo $t['medical_conditions']; ?></span>
                    <span class="info-value">
                        <?php if (!empty($user['medical_conditions'])): ?>
                            <?php echo nl2br(htmlspecialchars($user['medical_conditions'])); ?>
                        <?php else: ?>
                            <em style="color: var(--text-secondary);"><?php echo $t['none']; ?></em>
                        <?php endif; ?>
                    </span>
                </div>

                <div class="info-row">
                    <span class="info-label"><?php echo $t['current_medications']; ?></span>
                    <span class="info-value">
                        <?php if (!empty($user['current_medications'])): ?>
                            <?php echo nl2br(htmlspecialchars($user['current_medications'])); ?>
                        <?php else: ?>
                            <em style="color: var(--text-secondary);"><?php echo $t['none']; ?></em>
                        <?php endif; ?>
                    </span>
                </div>

                <?php if (!empty($user['allergies']) || !empty($user['medical_conditions'])): ?>
                <div class="medical-warning">
                    <i class="fas fa-exclamation-triangle"></i>
                    <strong>IMPORTANTE:</strong> Este usuario tiene condiciones médicas o alergias registradas. Revisar antes de cualquier intervención.
                </div>
                <?php endif; ?>
            </div>

            <!-- Contacto de Emergencia -->
            <div class="card">
                <div class="card-header">
                    <i class="fas fa-phone-alt"></i>
                    <h2><?php echo $t['emergency_contact']; ?></h2>
                </div>

                <div class="info-row">
                    <span class="info-label"><?php echo $t['contact_name']; ?></span>
                    <span class="info-value">
                        <?php if (!empty($user['emergency_contact_name'])): ?>
                            <?php echo htmlspecialchars($user['emergency_contact_name']); ?>
                        <?php elseif (!empty($user['emergency_contact'])): ?>
                            <?php echo htmlspecialchars($user['emergency_contact']); ?>
                        <?php else: ?>
                            <em style="color: var(--text-secondary);"><?php echo $t['not_specified']; ?></em>
                        <?php endif; ?>
                    </span>
                </div>

                <div class="info-row">
                    <span class="info-label"><?php echo $t['contact_phone']; ?></span>
                    <span class="info-value">
                        <?php if (!empty($user['emergency_contact_phone'])): ?>
                            <a href="tel:<?php echo htmlspecialchars($user['emergency_contact_phone']); ?>"
                               style="color: var(--primary); text-decoration: none; font-weight: 600;">
                                <i class="fas fa-phone"></i> <?php echo htmlspecialchars($user['emergency_contact_phone']); ?>
                            </a>
                        <?php elseif (!empty($user['emergency_phone'])): ?>
                            <a href="tel:<?php echo htmlspecialchars($user['emergency_phone']); ?>"
                               style="color: var(--primary); text-decoration: none; font-weight: 600;">
                                <i class="fas fa-phone"></i> <?php echo htmlspecialchars($user['emergency_phone']); ?>
                            </a>
                        <?php else: ?>
                            <em style="color: var(--text-secondary);"><?php echo $t['not_specified']; ?></em>
                        <?php endif; ?>
                    </span>
                </div>

                <?php if (!empty($user['emergency_contact_relation'])): ?>
                <div class="info-row">
                    <span class="info-label"><?php echo $t['contact_relation']; ?></span>
                    <span class="info-value"><?php echo htmlspecialchars($user['emergency_contact_relation']); ?></span>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Historial de Alertas -->
        <div class="alert-card">
            <div class="card-header">
                <i class="fas fa-history"></i>
                <h2><?php echo $t['alert_history']; ?></h2>
            </div>

            <?php if (empty($alerts)): ?>
                <div class="empty-state">
                    <i class="fas fa-inbox"></i>
                    <p><?php echo $t['no_alerts']; ?></p>
                </div>
            <?php else: ?>
                <?php foreach ($alerts as $alert): ?>
                    <div class="alert-item">
                        <div class="alert-info">
                            <div class="alert-type">
                                <?php
                                $type_icons = [
                                    'medical' => ['icon' => 'fa-heartbeat', 'color' => 'danger', 'text' => 'Médica'],
                                    'fire' => ['icon' => 'fa-fire', 'color' => 'warning', 'text' => 'Incendio'],
                                    'security' => ['icon' => 'fa-shield-alt', 'color' => 'info', 'text' => 'Seguridad']
                                ];
                                $type = $type_icons[$alert['alert_type']] ?? ['icon' => 'fa-bell', 'color' => 'info', 'text' => $alert['alert_type']];
                                ?>
                                <span class="badge badge-<?php echo $type['color']; ?>">
                                    <i class="fas <?php echo $type['icon']; ?>"></i> <?php echo $type['text']; ?>
                                </span>

                                <?php
                                $status_colors = ['pending' => 'warning', 'in_progress' => 'info', 'resolved' => 'success'];
                                $status_text = ['pending' => 'PENDIENTE', 'in_progress' => 'EN PROGRESO', 'resolved' => 'RESUELTA'];
                                ?>
                                <span class="badge badge-<?php echo $status_colors[$alert['status']] ?? 'info'; ?>">
                                    <?php echo $status_text[$alert['status']] ?? strtoupper($alert['status']); ?>
                                </span>

                                <?php
                                $priority_colors = ['high' => 'danger', 'medium' => 'warning', 'low' => 'info'];
                                $priority_text = ['high' => 'ALTA', 'medium' => 'MEDIA', 'low' => 'BAJA'];
                                ?>
                                <span class="badge badge-<?php echo $priority_colors[$alert['priority']] ?? 'info'; ?>">
                                    <?php echo $priority_text[$alert['priority']] ?? strtoupper($alert['priority']); ?>
                                </span>
                            </div>
                            <div class="alert-date">
                                <i class="fas fa-calendar"></i> <?php echo date('d/m/Y H:i', strtotime($alert['created_at'])); ?>
                                <?php if (!empty($alert['operator_name'])): ?>
                                    | <i class="fas fa-user"></i> <?php echo htmlspecialchars($alert['operator_name']); ?>
                                <?php endif; ?>
                            </div>
                            <?php if (!empty($alert['description'])): ?>
                                <div style="margin-top: 8px; font-size: 13px; color: var(--text-secondary);">
                                    <i class="fas fa-comment-dots"></i> <?php echo htmlspecialchars(substr($alert['description'], 0, 100)) . (strlen($alert['description']) > 100 ? '...' : ''); ?>
                                </div>
                            <?php endif; ?>
                        </div>
                        <a href="ver-alerta.php?id=<?php echo $alert['id']; ?>" class="btn btn-primary">
                            <i class="fas fa-eye"></i> <?php echo $t['view_alert']; ?>
                        </a>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>

    <script>
        // Imprimir historial médico
        function printMedicalHistory() {
            window.print();
        }

        // Agregar botón de impresión si es necesario
        document.addEventListener('DOMContentLoaded', function() {
            console.log('Historial médico cargado para usuario #<?php echo $user_id; ?>');
        });
    </script>
</body>
</html>

